"""
Base Language Model Interface

This module defines the base interface for language models.
"""

import json
import os
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Any, Dict, List, Optional

from src.utils.retry_utils import RetryConfig

API_URL = "OPENAI_BASE_URL"
API_KEY = "OPENAI_API_KEY"
CREDENTIALS_PATH = "credentials/cred.json"


class LLMInterface(ABC):
    """
    Interface for language models.

    This abstract class defines the interface that all language model
    implementations must adhere to.
    """

    def __init__(self, retry_config: Optional[RetryConfig] = None):
        """
        Initialise the LLM interface.

        Args:
            retry_config: Configuration for retry behavior
        """
        self.retry_config = retry_config or RetryConfig()

    def load_api_info(self) -> tuple[str, str]:
        """
        Load API URL and API key from environment variables or credentials file.

        Returns:
            tuple: (api_url, api_key)

        Raises:
            ValueError: If API URL or API key could not be loaded
        """

        # Try to load from environment variables first
        api_url = os.environ.get(API_URL)
        api_key = os.environ.get(API_KEY)

        # If environment variables not set, try to load from credentials file
        if not api_url or not api_key:
            import logging

            logger = logging.getLogger(
                f"{self.__class__.__module__}.{self.__class__.__name__}"
            )
            logger.debug(
                "OPENAI_API_KEY & OPENAI_BASE_URL environment variables not set, attempting to load from credentials file"
            )
            try:
                cred_path = Path(CREDENTIALS_PATH)
                if cred_path.exists():
                    with open(cred_path, "r") as f:
                        creds = json.load(f)
                    api_url = api_url or creds.get(API_URL)
                    api_key = api_key or creds.get(API_KEY)
            except FileNotFoundError:
                logger.warning(f"Credentials file {CREDENTIALS_PATH} not found")
            except json.JSONDecodeError:
                logger.warning("Failed to parse credentials file as JSON")
            except Exception as e:
                logger.warning(f"Error loading credentials: {str(e)}")

        if not api_url or not api_key:
            raise ValueError(
                "Missing API credentials: Could not load API URL or API key from environment variables or credentials file"
            )

        return api_url, api_key

    def configure_environment(self, base_url: str, api_key: str) -> None:
        os.environ.update(
            {"OPENAI_API_KEY": f"{api_key}", "OPENAI_BASE_URL": f"{base_url}"}
        )

    @abstractmethod
    def generate(self, messages: List[Dict[str, Any]], **kwargs: Any) -> str:
        """
        Generate text based on the provided messages.

        Args:
            messages: A list of message dictionaries, each with 'role' and 'content' keys
            **kwargs: Additional keyword arguments for the generation process

        Returns:
            The generated text
        """
        pass

    def log_request(self, messages: List[Dict[str, Any]], **kwargs: Any) -> None:
        """
        Log information about a generation request.

        Args:
            messages: A list of message dictionaries, each with 'role' and 'content' keys
            **kwargs: Additional keyword arguments for the generation process
        """
        num_messages = len(messages)
        total_tokens = (
            sum(len(msg.get("content", "")) for msg in messages) // 4
        )  # Rough estimate

        import logging

        logger = logging.getLogger(
            f"{self.__class__.__module__}.{self.__class__.__name__}"
        )
        logger.debug(f"Request: {num_messages} messages, ~{total_tokens} tokens")
        logger.debug(f"Request parameters: {kwargs}")

        for i, msg in enumerate(messages):
            role = msg.get("role", "unknown")
            content = msg.get("content", "")
            content_preview = content[:100] + "..." if len(content) > 100 else content
            logger.debug(f"Message {i}: {role} - {content_preview}")

    def log_response(self, response: str, duration: float) -> None:
        """
        Log information about a generation response.

        Args:
            response: The generated text
            duration: The time taken to generate the response in seconds
        """
        response_tokens = len(response) // 4  # Rough estimate

        import logging

        logger = logging.getLogger(
            f"{self.__class__.__module__}.{self.__class__.__name__}"
        )
        logger.debug(f"Response: ~{response_tokens} tokens in {duration:.2f}s")

        # response_preview = response[:100] + "..." if len(response) > 100 else response
        logger.debug(f"Response: {response}")

    @property
    @abstractmethod
    def model_info(self) -> Dict[str, Any]:
        """
        Get information about the model.

        Returns:
            A dictionary containing information about the model
        """
        pass
