"""
Logging Utilities

This module provides utilities for setting up and using logging throughout the project.
"""

from colorama import Fore, Back, Style, init
import logging
import time
from pathlib import Path
from typing import Optional, Union

from src.utils.logging_config import (
    update_global_logging_config,
    ensure_logging_initialised,
)

init(autoreset=True)


class ProjectLogFilter(logging.Filter):
    """
    Filter that only allows logs from the project's modules.

    This filter is used to prevent logs from third-party libraries
    from being written to the experiment log file.
    """

    def __init__(self, project_prefixes: Optional[list[str]] = None):
        """
        Initialise the filter with a list of project module prefixes.

        Args:
            project_prefixes: List of module name prefixes that should be allowed
                             (default: ["src", "experiment"])
        """
        super().__init__()
        self.project_prefixes = project_prefixes or [
            "src",  # Main source code
            "experiment",  # Experiment-specific code
            "__main__",  # Main module
            "tests",  # Test modules
            "config",  # Configuration modules
            "examples",  # Example scripts
        ]

    def filter(self, record):
        """
        Check if the log record is from a project module.

        Args:
            record: The log record to check

        Returns:
            True if the log record should be logged, False otherwise
        """
        # Always allow logs from the root logger (no name)
        if not record.name:
            return True

        # Always allow logs with no module name (often from nested functions)
        if record.module == "":
            return True

        # Check if the log record is from a project module
        is_project_module = any(
            record.name.startswith(prefix) for prefix in self.project_prefixes
        )

        # If not a project module by name, check if it's a nested function in a project module
        # This helps with nested functions like _load_data in data_handler.py
        if not is_project_module and hasattr(record, "pathname"):
            is_project_path = any(
                prefix in record.pathname
                for prefix in ["/src/", "/tests/", "/examples/"]
            )
            if is_project_path:
                return True

        return is_project_module


def setup_logger(
    experiment_name: str,
    output_dir: Union[str, Path] = "output",
    level: int = None,
    args=None,
) -> logging.Logger:
    """
    Set up a logger specifically for an experiment.

    This creates a logger that logs to both console and a file in the experiment's
    output directory using the centralised logging system to avoid handler duplication.

    Args:
        experiment_name: The name of the experiment
        output_dir: The output directory for the experiment (default: "output")
        level: The logging level (default: determined from args or INFO)
        args: Command-line arguments that may contain 'verbose' flag (default: None)

    Returns:
        A tuple containing:
            - The configured logger
            - The path to the experiment output directory
    """
    # Determine log level based on args or default to INFO
    if level is None:
        if args and hasattr(args, "verbose") and args.verbose:
            level = logging.DEBUG
        elif args and hasattr(args, "quiet") and args.quiet:
            level = logging.WARNING
        else:
            level = logging.INFO

    # Create the log file path
    output_path = Path(output_dir)
    output_path.mkdir(parents=True, exist_ok=True)
    log_file = output_path / "experiment.log"

    # Update global logging configuration for experiment
    # Preserve console_level if it was set by CLI args (quiet/verbose)
    console_level = None
    if args and hasattr(args, "quiet") and args.quiet:
        console_level = logging.WARNING
    elif args and hasattr(args, "verbose") and args.verbose:
        console_level = logging.DEBUG

    update_global_logging_config(
        level=level,
        log_file=log_file,
        experiment_name=experiment_name,
        experiment_output_dir=output_path,
        console_level=console_level,
    )

    # Initialise/re-initialise the centralised logging system with the new configuration
    from src.utils.logging_config import get_global_logging_config

    config = get_global_logging_config()
    config.apply_dict_config()

    # Get the experiment-specific logger using centralised approach
    logger = logging.getLogger(f"experiment.{experiment_name}")
    logger.setLevel(level)

    # Ensure proper propagation - experiment loggers should propagate to root
    # so they use the centralised file handler
    logger.propagate = True

    # Remove any existing handlers from the experiment logger (centralised approach)
    for handler in logger.handlers[:]:
        logger.removeHandler(handler)

    return logger


def log_prompt_usage(logger: logging.Logger, prompt: str, prompt_type: str = "task"):
    """Log the prompt being used with colored output."""
    logger.info(f"{Fore.GREEN}💬 {prompt_type.upper()} PROMPT:{Style.RESET_ALL}")
    logger.info(f"{Fore.WHITE}{Back.BLACK}{prompt}{Style.RESET_ALL}")


def log_evaluation_score(logger: logging.Logger, score: float, dataset_name: str):
    """Log evaluation scores with colored output."""
    dataset_part = f" ({dataset_name})" if dataset_name else ""
    if score >= 0.8:
        color = Fore.GREEN
        emoji = "🎉"
    elif score >= 0.6:
        color = Fore.YELLOW
        emoji = "👍"
    else:
        color = Fore.RED
        emoji = "❌"
    logger.info(
        f"{color}{emoji} EVALUATION SCORE{dataset_part}: {score:.4f}{Style.RESET_ALL}"
    )
