"""
Prompt Utilities

This module provides utility functions for working with prompts.
"""

import logging
from typing import Any, Dict, List

from src.utils.decorator_utils import with_logger


@with_logger
def extract_system_prompt(messages: List[Dict[str, Any]]) -> str:
    """
    Extract the system prompt from a list of messages.

    Args:
        messages: A list of message dictionaries

    Returns:
        The system prompt text

    Raises:
        ValueError: If no system prompt is found
    """
    logger.info(f"Extracting system prompt from {len(messages)} messages")

    for message in messages:
        if message.get("role") == "system":
            content = message.get("content", "")
            if isinstance(content, str):
                logger.info(f"Found system prompt: {content}")
                return content
            elif isinstance(content, list):
                # Handle OpenAI's format with multiple content parts
                logger.info("Found system prompt with multiple content parts")
                text_parts = []
                for part in content:
                    if part.get("type") == "text":
                        text_parts.append(part.get("text", ""))
                result = " ".join(text_parts)
                logger.info(f"Extracted system prompt: {result}")
                return result

    logger.warning("No system prompt found in messages")
    raise ValueError("No system prompt found in messages")


@with_logger
def extract_user_prompts(messages: List[Dict[str, Any]]) -> List[str]:
    """
    Extract user prompts from a list of messages.

    Args:
        messages: A list of message dictionaries

    Returns:
        A list of user prompt texts
    """
    logger.info(f"Extracting user prompts from {len(messages)} messages")
    user_prompts = []

    for message in messages:
        if message.get("role") == "user":
            content = message.get("content", "")
            if isinstance(content, str):
                user_prompts.append(content)
                logger.info(f"Found user prompt: {content}")
            elif isinstance(content, list):
                # Handle OpenAI's format with multiple content parts
                logger.info("Found user prompt with multiple content parts")
                text_parts = []
                for part in content:
                    if part.get("type") == "text":
                        text_parts.append(part.get("text", ""))
                prompt = " ".join(text_parts)
                user_prompts.append(prompt)
                logger.info(f"Extracted user prompt: {prompt}")

    logger.info(f"Extracted {len(user_prompts)} user prompts")
    return user_prompts


@with_logger
def create_system_message(content: str) -> Dict[str, Any]:
    """
    Create a system message.

    Args:
        content: The message content

    Returns:
        A message dictionary
    """
    logger.info(f"Creating system message: {content}")
    return {"role": "system", "content": content}


@with_logger
def create_user_message(content: str) -> Dict[str, Any]:
    """
    Create a user message.

    Args:
        content: The message content

    Returns:
        A message dictionary
    """
    logger.info(f"Creating user message: {content}")
    return {"role": "user", "content": content}


@with_logger
def create_assistant_message(content: str) -> Dict[str, Any]:
    """
    Create an assistant message.

    Args:
        content: The message content

    Returns:
        A message dictionary
    """
    logger.info(f"Creating assistant message: {content}")
    return {"role": "assistant", "content": content}


@with_logger
def create_user_message_with_image(text: str, image_url: str) -> Dict[str, Any]:
    """
    Create a user message with an image.

    Args:
        text: The message text
        image_url: The image URL

    Returns:
        A message dictionary
    """
    logger.info(f"Creating user message with image: {text} and image: {image_url}")
    return {
        "role": "user",
        "content": [
            {"type": "text", "text": text},
            {"type": "image_url", "image_url": {"url": image_url}},
        ],
    }


@with_logger
def log_messages(messages: List[Dict[str, Any]], level: int = logging.info) -> None:
    """
    Log a list of messages at the specified level.

    Args:
        messages: A list of message dictionaries
        level: The logging level to use
    """
    if not logger.isEnabledFor(level):
        return

    logger.log(level, f"Message sequence ({len(messages)} messages):")
    for i, message in enumerate(messages):
        role = message.get("role", "unknown")
        content = message.get("content", "")

        if isinstance(content, str):
            content_preview = content[:100] + "..." if len(content) > 100 else content
            logger.log(level, f"  Message {i}: {role} - {content_preview}")
        elif isinstance(content, list):
            logger.log(level, f"  Message {i}: {role} - [multipart content]")
            for j, part in enumerate(content):
                part_type = part.get("type", "unknown")
                if part_type == "text":
                    text = part.get("text", "")
                    text_preview = text[:100] + "..." if len(text) > 100 else text
                    logger.log(level, f"    Part {j}: {part_type} - {text_preview}")
                else:
                    logger.log(level, f"    Part {j}: {part_type}")
