"""
Integration tests for the tamper detection experiment.
"""

import os
import tempfile
import pathlib
from unittest import mock

import pytest
import pandas as pd

from src.core.experiment import ExperimentRunner
from src.llm.dummy_llm import DummyLLM
from src.tasks.tamper_detection.task_handler import TamperDetectionTask


class TestTamperDetectionExperiment:
    """Integration tests for the tamper detection experiment."""

    def setup_method(self):
        """Set up test fixtures."""
        # Create temporary directory for output
        self.temp_dir = tempfile.TemporaryDirectory()
        self.output_dir = pathlib.Path(self.temp_dir.name)

        # Create a test config
        self.config = {
            "project": {
                "name": "tamper_detection_test",
                "version": "0.1.0",
            },
            "paths": {
                "data": "data",
                "output": str(self.output_dir),
            },
            "llm": {
                "type": "DummyLLM",
                "version_name": "dummy",
                "response": "No",  # Will alternate between "No" and "Yes"
            },
            "prompt_optimiser": {
                "type": "none",
            },
            "task": {
                "type": "tamper_detection",
                "font_semantics": "font",
                "num_images": 5,
            },
        }

    def teardown_method(self):
        """Tear down test fixtures."""
        self.temp_dir.cleanup()

    def test_data_path_configuration(self):
        """Test that the data path configuration works correctly."""
        # Update the config with a custom data path
        self.config["paths"]["data"] = "/custom/config/data/path"

        # Create an experiment runner
        runner = ExperimentRunner(
            "data_path_test", config=self.config, output_dir=self.output_dir
        )

        # Define a test function to check the data path
        def check_data_path(config):
            """Check that the data path is correctly configured."""
            # Get the data path from the config
            data_path = config["paths"]["data"]

            # Check that it matches the expected value
            assert data_path == "/custom/config/data/path"

            # Instead of creating a real task, we'll verify the config is correctly passed
            # to the config manager
            from src.core.config import config_manager

            # Update the config manager's config with our custom path
            config_manager.set("paths.data", "/custom/config/data/path")

            # Verify the config manager returns the correct path
            actual_data_path = config_manager.get("paths.data", "default_value")
            assert actual_data_path == "/custom/config/data/path"

            return {"data_path": data_path, "actual_data_path": actual_data_path}

        # Add the step to the experiment
        runner.add_step("check_data_path", check_data_path, config=self.config)

        # Mock the _save_results method to avoid file operations
        with mock.patch.object(runner, "_save_results"):
            # Run the experiment
            results = runner.run()

            # Check the results
            assert "check_data_path" in results
            assert results["check_data_path"]["data_path"] == "/custom/config/data/path"
            assert (
                results["check_data_path"]["actual_data_path"]
                == "/custom/config/data/path"
            )

    def test_data_path_cli_override(self):
        """Test that the command-line data path override works correctly."""
        # Set up a config with one data path
        self.config["paths"]["data"] = "/config/data/path"

        # Create an experiment runner
        runner = ExperimentRunner(
            "data_path_cli_test", config=self.config, output_dir=self.output_dir
        )

        # Define a test function to check the data path with CLI override
        def check_data_path_override(config):
            """Check that the data path is correctly overridden by CLI args."""
            # Create mock args
            import argparse

            mock_args = argparse.Namespace(data_path="/cli/override/data/path")

            # Create a new ConfigManager to test the override
            from src.core.config import ConfigManager

            config_manager = ConfigManager(
                config_dir=pathlib.Path(self.temp_dir.name) / "config",
                schema_dir=pathlib.Path(self.temp_dir.name) / "config" / "schemas",
            )

            # Apply CLI overrides
            config_manager._apply_cli_overrides(config, mock_args)

            # Check that the data path was overridden
            assert config["paths"]["data"] == "/cli/override/data/path"

            return {
                "original_path": "/config/data/path",
                "overridden_path": config["paths"]["data"],
            }

        # Add the step to the experiment
        runner.add_step(
            "check_data_path_override", check_data_path_override, config=self.config
        )

        # Mock the _save_results method to avoid file operations
        with mock.patch.object(runner, "_save_results"):
            # Run the experiment
            results = runner.run()

            # Check the results
            assert "check_data_path_override" in results
            assert (
                results["check_data_path_override"]["original_path"]
                == "/config/data/path"
            )
            assert (
                results["check_data_path_override"]["overridden_path"]
                == "/cli/override/data/path"
            )
