"""
Tests for the registry module.
"""

import pytest

from src.core.registry import Registry


class TestRegistry:
    """Tests for the Registry class."""

    def test_register_and_get(self):
        """Test registering and retrieving a component."""
        # Create a registry
        registry = Registry()

        # Define a test class
        class TestComponent:
            def __init__(self, value=None):
                self.value = value

        # Register the test class
        registry.register("test")(TestComponent)

        # Get the registered class
        retrieved_class = registry.get("test")

        # Check that the retrieved class is the same as the registered class
        assert retrieved_class is TestComponent

    def test_create(self):
        """Test creating an instance of a registered component."""
        # Create a registry
        registry = Registry()

        # Define a test class
        class TestComponent:
            def __init__(self, value=None):
                self.value = value

        # Register the test class
        registry.register("test")(TestComponent)

        # Create an instance of the registered class
        instance = registry.create("test", value=42)

        # Check that the instance is of the correct type
        assert isinstance(instance, TestComponent)
        # Check that the instance has the correct value
        assert instance.value == 42

    def test_list_registered(self):
        """Test listing registered components."""
        # Create a registry
        registry = Registry()

        # Define test classes
        class TestComponent1:
            pass

        class TestComponent2:
            pass

        # Register the test classes
        registry.register("test1")(TestComponent1)
        registry.register("test2")(TestComponent2)

        # List registered components
        registered = registry.list_registered()

        # Check that the list contains the registered components
        assert "test1" in registered
        assert "test2" in registered
        assert len(registered) == 2

    def test_register_duplicate(self):
        """Test registering a component with a duplicate name."""
        # Create a registry
        registry = Registry()

        # Define test classes
        class TestComponent1:
            pass

        class TestComponent2:
            pass

        # Register the first test class
        registry.register("test")(TestComponent1)

        # Try to register the second test class with the same name
        with pytest.raises(ValueError):
            registry.register("test")(TestComponent2)

    def test_get_nonexistent(self):
        """Test getting a component that doesn't exist."""
        # Create a registry
        registry = Registry()

        # Try to get a component that doesn't exist
        with pytest.raises(ValueError):
            registry.get("nonexistent")

    def test_create_nonexistent(self):
        """Test creating an instance of a component that doesn't exist."""
        # Create a registry
        registry = Registry()

        # Try to create an instance of a component that doesn't exist
        with pytest.raises(ValueError):
            registry.create("nonexistent")