"""
Tests for the dummy LLM implementation.
"""

import time
from unittest import mock

import pytest

# Mock the OpenAI module
mock_openai = mock.MagicMock()
mock.patch.dict('sys.modules', {'openai': mock_openai}).start()

from src.llm.dummy_llm import DummyLLM


class TestDummyLLM:
    """Tests for the DummyLLM class."""

    def test_init(self):
        """Test initialisation of DummyLLM."""
        # Test with default parameters
        llm = DummyLLM()
        assert llm.version_name == "dummy"
        assert llm.response == "No"
        assert llm.delay == 0.0
        assert llm.kwargs == {}
        
        # Test with custom parameters
        llm = DummyLLM(
            version_name="custom",
            response="Yes",
            delay=1.0,
            extra_param="value"
        )
        assert llm.version_name == "custom"
        assert llm.response == "Yes"
        assert llm.delay == 1.0
        assert llm.kwargs == {"extra_param": "value"}

    def test_generate(self):
        """Test generating text with DummyLLM."""
        # Create a dummy LLM
        llm = DummyLLM(response="Test response")
        
        # Generate text
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {"role": "user", "content": "Hello!"},
        ]
        response = llm.generate(messages)
        
        # Check that the response is the fixed response
        assert response == "Test response"

    def test_generate_with_delay(self):
        """Test generating text with a delay."""
        # Create a dummy LLM with a small delay
        delay = 0.1
        llm = DummyLLM(response="Test response", delay=delay)
        
        # Generate text and measure the time
        messages = [{"role": "user", "content": "Hello!"}]
        start_time = time.time()
        response = llm.generate(messages)
        end_time = time.time()
        
        # Check that the response is correct
        assert response == "Test response"
        
        # Check that the delay was applied (with some tolerance)
        assert end_time - start_time >= delay

    def test_model_info(self):
        """Test getting model information."""
        # Create a dummy LLM with custom parameters
        llm = DummyLLM(
            version_name="custom",
            response="Yes",
            delay=1.0,
            extra_param="value"
        )
        
        # Get the model info
        info = llm.model_info
        
        # Check the info
        assert info["name"] == "DummyLLM"
        assert info["version_name"] == "custom"
        assert info["response"] == "Yes"
        assert info["delay"] == 1.0
        assert info["extra_param"] == "value"