"""
Tests for the CLI utility module.
"""

import argparse
import logging
from unittest import mock

from src.utils.cli import default_parse_args, configure_logging


class TestCLI:
    """Tests for the CLI utility functions."""

    def test_default_parse_args(self):
        """Test the default_parse_args function."""
        # Test with default arguments
        with mock.patch(
            "argparse.ArgumentParser.parse_args",
            return_value=argparse.Namespace(
                config="test_config",
                data_path=None,
                verbose=False,
                quiet=False,
                log_file=None,
            ),
        ):
            args = default_parse_args(default_config="test_config")
            assert args.config == "test_config"
            assert args.data_path is None
            assert not args.verbose
            assert not args.quiet
            assert args.log_file is None

    def test_data_path_argument(self):
        """Test that the data_path argument is correctly added and parsed."""
        # Test with data_path specified
        with mock.patch(
            "argparse.ArgumentParser.parse_args",
            return_value=argparse.Namespace(
                config="test_config",
                data_path="/custom/data/path",
                verbose=False,
                quiet=False,
                log_file=None,
            ),
        ):
            args = default_parse_args(default_config="test_config")
            assert args.config == "test_config"
            assert args.data_path == "/custom/data/path"
            assert not args.verbose
            assert not args.quiet
            assert args.log_file is None

    def test_parse_args_with_extra_args(self):
        """Test the default_parse_args function with extra arguments."""

        # Define a function to add extra arguments
        def add_extra_args(parser):
            parser.add_argument("--extra", type=str, help="Extra argument")

        # Test with extra arguments
        with mock.patch(
            "argparse.ArgumentParser.parse_args",
            return_value=argparse.Namespace(
                config="test_config",
                data_path="/custom/data/path",
                verbose=False,
                quiet=False,
                log_file=None,
                extra="extra_value",
            ),
        ):
            args = default_parse_args(
                default_config="test_config", extra_args_fn=add_extra_args
            )
            assert args.config == "test_config"
            assert args.data_path == "/custom/data/path"
            assert args.extra == "extra_value"

    def test_logging_configuration(self):
        """Test that logging is configured correctly based on arguments."""
        # Test with verbose=True
        args_verbose = argparse.Namespace(
            config="test_config",
            data_path="/custom/data/path",
            verbose=True,
            quiet=False,
            log_file=None,
        )

        with mock.patch(
            "src.utils.cli.configure_logging_from_args"
        ) as mock_configure_logging_from_args, mock.patch(
            "src.utils.cli.initialise_logging_system"
        ) as mock_initialise_logging_system, mock.patch(
            "logging.getLogger"
        ) as mock_get_logger:

            # Mock the logger and its handlers
            mock_logger = mock.Mock()
            mock_logger.handlers = []  # No existing file handlers
            mock_get_logger.return_value = mock_logger

            # Mock the config returned by configure_logging_from_args
            mock_config = mock.Mock()
            mock_config.level = 10  # DEBUG level
            mock_config.log_file = None
            mock_config.format_string = (
                "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
            )
            mock_config.date_format = "%Y-%m-%d %H:%M:%S"
            mock_config.console_output = True
            mock_config.project_filter = True
            mock_config.apply_dict_config = mock.Mock()
            mock_configure_logging_from_args.return_value = mock_config

            configure_logging(args_verbose)

            # Check that configure_logging_from_args was called with the correct arguments
            mock_configure_logging_from_args.assert_called_once_with(args_verbose)
            # Check that initialise_logging_system was called
            mock_initialise_logging_system.assert_called_once()

        # Test with quiet=True
        args_quiet = argparse.Namespace(
            config="test_config",
            data_path="/custom/data/path",
            verbose=False,
            quiet=True,
            log_file="test.log",
        )

        with mock.patch(
            "src.utils.cli.configure_logging_from_args"
        ) as mock_configure_logging_from_args, mock.patch(
            "src.utils.cli.initialise_logging_system"
        ) as mock_initialise_logging_system, mock.patch(
            "logging.getLogger"
        ) as mock_get_logger:

            # Mock the logger and its handlers
            mock_logger = mock.Mock()
            mock_logger.handlers = []  # No existing file handlers
            mock_get_logger.return_value = mock_logger

            # Mock the config returned by configure_logging_from_args
            mock_config = mock.Mock()
            mock_config.level = 30  # WARNING level
            mock_config.log_file = "test.log"
            mock_config.format_string = (
                "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
            )
            mock_config.date_format = "%Y-%m-%d %H:%M:%S"
            mock_config.console_output = True
            mock_config.project_filter = True
            mock_config.apply_dict_config = mock.Mock()
            mock_configure_logging_from_args.return_value = mock_config

            configure_logging(args_quiet)

            # Check that configure_logging_from_args was called with the correct arguments
            mock_configure_logging_from_args.assert_called_once_with(args_quiet)
            # Check that initialise_logging_system was called
            mock_initialise_logging_system.assert_called_once()
            # Check that apply_dict_config was called since log_file is specified
            mock_config.apply_dict_config.assert_called_once()

        # Test with existing file handlers (should not call setup_logger)
        args_existing = argparse.Namespace(
            config="test_config",
            verbose=False,
            quiet=False,
            log_file=None,
        )

        with mock.patch(
            "src.utils.cli.configure_logging_from_args"
        ) as mock_configure_logging_from_args, mock.patch(
            "src.utils.cli.initialise_logging_system"
        ) as mock_initialise_logging_system, mock.patch(
            "logging.getLogger"
        ) as mock_get_logger:

            # Mock existing file handler
            mock_file_handler = mock.Mock(spec=logging.FileHandler)
            mock_file_handler.baseFilename = "/path/to/existing.log"
            mock_logger = mock.Mock()
            mock_logger.handlers = [mock_file_handler]
            mock_get_logger.return_value = mock_logger

            # Mock the config
            mock_config = mock.Mock()
            mock_config.level = 20  # INFO level
            mock_config.log_file = None
            mock_config.format_string = (
                "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
            )
            mock_config.date_format = "%Y-%m-%d %H:%M:%S"
            mock_config.console_output = True
            mock_config.project_filter = True
            mock_config.apply_dict_config = mock.Mock()
            mock_configure_logging_from_args.return_value = mock_config

            configure_logging(args_existing)

            # Check that configure_logging_from_args was called with the correct arguments
            mock_configure_logging_from_args.assert_called_once_with(args_existing)
            # Check that initialise_logging_system was called
            mock_initialise_logging_system.assert_called_once()
