"""
Tests for decorator utilities
"""

import logging
from unittest.mock import patch, MagicMock

from src.utils.decorator_utils import with_logger


def test_with_logger_decorator_injects_logger():
    """Test that the with_logger decorator injects a logger into the function's scope."""

    # Define a test function using the decorator
    @with_logger
    def test_func():
        return logger  # This logger is injected by the decorator

    # Mock the functions used by the decorator
    mock_logger = MagicMock(spec=logging.Logger)
    mock_config = MagicMock()
    mock_config.copy.return_value = mock_config

    with patch(
        "src.utils.decorator_utils.get_effective_config", return_value=mock_config
    ), patch(
        "src.utils.decorator_utils._get_configured_logger", return_value=mock_logger
    ):
        # Call the function
        result = test_func()

        # Verify that the result is the mock logger
        assert result is mock_logger


def test_with_logger_decorator_uses_provided_logger():
    """Test that the with_logger decorator uses a provided logger."""

    # Define a test function using the decorator
    @with_logger
    def test_func():
        return logger  # This logger is injected by the decorator

    # Create a mock logger to pass to the function
    provided_logger = MagicMock(spec=logging.Logger)

    # Call the function with the provided logger
    result = test_func(logger=provided_logger)

    # Verify that the result is the provided logger
    assert result is provided_logger


def test_with_logger_decorator_works_with_kwargs():
    """Test that the with_logger decorator works with other kwargs."""

    # Define a test function using the decorator
    @with_logger
    def test_func(param1):
        return param1, logger  # logger is injected by the decorator

    # Mock the functions used by the decorator
    mock_logger = MagicMock(spec=logging.Logger)
    mock_config = MagicMock()
    mock_config.copy.return_value = mock_config

    with patch(
        "src.utils.decorator_utils.get_effective_config", return_value=mock_config
    ), patch(
        "src.utils.decorator_utils._get_configured_logger", return_value=mock_logger
    ):
        # Call the function with a parameter
        param1, result_logger = test_func(param1="value1")

        # Verify the parameter was passed correctly
        assert param1 == "value1"
        # Verify that the logger is the mock logger
        assert result_logger is mock_logger
