"""
Tests for the prompt utilities.
"""

import pytest

from src.utils.prompt_utils import (
    extract_system_prompt,
    extract_user_prompts,
    create_system_message,
    create_user_message,
    create_assistant_message,
    create_user_message_with_image,
)


class TestPromptUtils:
    """Tests for the prompt utilities."""

    def test_extract_system_prompt(self):
        """Test extracting the system prompt from messages."""
        # Test with a simple system prompt
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {"role": "user", "content": "Hello!"},
        ]
        assert extract_system_prompt(messages) == "You are a helpful assistant."
        
        # Test with a system prompt with multiple content parts
        messages = [
            {
                "role": "system",
                "content": [
                    {"type": "text", "text": "You are a helpful assistant."},
                    {"type": "text", "text": "Be concise."},
                ],
            },
            {"role": "user", "content": "Hello!"},
        ]
        assert extract_system_prompt(messages) == "You are a helpful assistant. Be concise."
        
        # Test with no system prompt
        messages = [
            {"role": "user", "content": "Hello!"},
            {"role": "assistant", "content": "Hi there!"},
        ]
        with pytest.raises(ValueError):
            extract_system_prompt(messages)
        
        # Test with empty system prompt
        messages = [
            {"role": "system", "content": ""},
            {"role": "user", "content": "Hello!"},
        ]
        assert extract_system_prompt(messages) == ""
        
        # Test with non-text content parts
        messages = [
            {
                "role": "system",
                "content": [
                    {"type": "text", "text": "You are a helpful assistant."},
                    {"type": "image", "url": "http://example.com/image.jpg"},
                ],
            },
            {"role": "user", "content": "Hello!"},
        ]
        assert extract_system_prompt(messages) == "You are a helpful assistant."

    def test_extract_user_prompts(self):
        """Test extracting user prompts from messages."""
        # Test with simple user prompts
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {"role": "user", "content": "Hello!"},
            {"role": "assistant", "content": "Hi there!"},
            {"role": "user", "content": "How are you?"},
        ]
        assert extract_user_prompts(messages) == ["Hello!", "How are you?"]
        
        # Test with user prompts with multiple content parts
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": "Hello!"},
                    {"type": "text", "text": "How are you?"},
                ],
            },
            {"role": "assistant", "content": "Hi there!"},
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": "What's the weather like?"},
                ],
            },
        ]
        assert extract_user_prompts(messages) == ["Hello! How are you?", "What's the weather like?"]
        
        # Test with no user prompts
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {"role": "assistant", "content": "Hi there!"},
        ]
        assert extract_user_prompts(messages) == []
        
        # Test with empty user prompts
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {"role": "user", "content": ""},
            {"role": "assistant", "content": "Hi there!"},
        ]
        assert extract_user_prompts(messages) == [""]
        
        # Test with non-text content parts
        messages = [
            {"role": "system", "content": "You are a helpful assistant."},
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": "Hello!"},
                    {"type": "image", "url": "http://example.com/image.jpg"},
                ],
            },
        ]
        assert extract_user_prompts(messages) == ["Hello!"]

    def test_create_system_message(self):
        """Test creating a system message."""
        message = create_system_message("You are a helpful assistant.")
        assert message == {"role": "system", "content": "You are a helpful assistant."}
        
        # Test with empty content
        message = create_system_message("")
        assert message == {"role": "system", "content": ""}

    def test_create_user_message(self):
        """Test creating a user message."""
        message = create_user_message("Hello!")
        assert message == {"role": "user", "content": "Hello!"}
        
        # Test with empty content
        message = create_user_message("")
        assert message == {"role": "user", "content": ""}

    def test_create_assistant_message(self):
        """Test creating an assistant message."""
        message = create_assistant_message("Hi there!")
        assert message == {"role": "assistant", "content": "Hi there!"}
        
        # Test with empty content
        message = create_assistant_message("")
        assert message == {"role": "assistant", "content": ""}

    def test_create_user_message_with_image(self):
        """Test creating a user message with an image."""
        message = create_user_message_with_image(
            "What's in this image?",
            "http://example.com/image.jpg"
        )
        assert message == {
            "role": "user",
            "content": [
                {"type": "text", "text": "What's in this image?"},
                {"type": "image_url", "image_url": {"url": "http://example.com/image.jpg"}},
            ],
        }
        
        # Test with empty text
        message = create_user_message_with_image(
            "",
            "http://example.com/image.jpg"
        )
        assert message == {
            "role": "user",
            "content": [
                {"type": "text", "text": ""},
                {"type": "image_url", "image_url": {"url": "http://example.com/image.jpg"}},
            ],
        }