"""
Simple Experiment Example

This script demonstrates how to use the core infrastructure to run a simple experiment.
"""

import sys
import pathlib
import time
from pathlib import Path

# Add the project root to the Python path
project_root = pathlib.Path(__file__).parent.parent.resolve()
sys.path.insert(0, str(project_root))

# Import logging utils first to set up global logger before other imports
from src.utils.logging_utils import setup_logger
from src.utils.decorator_utils import with_logger

# Now import other modules
from src.core import (
    config_manager,
    ExperimentRunner,
    llm_registry,
    task_registry,
)
from src.utils.cli import default_parse_args, configure_logging

# Logger will be set up after CLI args are parsed
logger = None
log_path = None


@with_logger
def init_llm(config):
    """Initialise the LLM."""
    logger.info(f"Initialising LLM: {config['llm']['default']}")

    # In a real implementation, this would use the LLM registry
    llm_name = config["llm"]["default"]
    llm_config = config["llm"][llm_name]
    logger.info(f"LLM configuration: {llm_config}")

    llm = llm_registry.create(llm_name, **llm_config)

    return llm


@with_logger
def init_task(config, init_llm=None, **kwargs):
    """initialise the task."""
    logger.info(f"initialising task: {config['task']['default']}")

    # In a real implementation, this would use the task registry
    task_name = config["task"]["default"]
    task_config = config["task"][task_name]
    logger.info(f"Task configuration: {task_config}")

    task = task_registry.create(task_name, **task_config)

    return task


@with_logger
def run_task(init_llm, init_task, **kwargs):
    """Run the task with the base prompt."""
    logger.info("Running task with base prompt")

    # In a real implementation, this would run the task with the base prompt

    # Simulate running the task
    result_df, score = init_task.run(init_llm)
    logger.info(f"Base prompt score: {score:.4f}")

    return result_df, score


def main():
    """Run the experiment."""
    global logger, log_path

    # Parse command-line arguments first to get verbosity setting
    args = default_parse_args(
        description="Run a simple experiment",
    )
    # Configure logging
    configure_logging(args)

    # Set up experiment-specific logger AFTER CLI configuration
    experiment_name = "simple_experiment"
    logger, log_path = setup_logger(experiment_name, args=args)

    # Set up output directory
    timestamp = time.strftime("%Y%m%d-%H%M%S")
    output_dir = Path("output")
    log_dir = output_dir / f"{experiment_name}_{timestamp}"
    log_dir.mkdir(parents=True, exist_ok=True)

    # Use the experiment logger for the rest of the script
    logger.info(f"Starting {experiment_name}")

    # Load the experiment configuration
    logger.info(f"Loading configuration: {args.config}")
    config = config_manager.load_config(args.config)
    logger.info(f"Loaded config: {config}")

    # Create the experiment runner
    experiment_name = config["experiment"]["name"]
    logger.info(f"Creating experiment runner: {experiment_name}")
    experiment = ExperimentRunner(
        name=experiment_name,
        config=config,
        output_dir=log_dir.parent,  # Use the parent directory of our log_dir
    )

    # Add the experiment steps and run the experiment
    experiment.add_step("init_llm", init_llm, config=config)
    experiment.add_step("init_task", init_task, config=config)
    experiment.add_step("run_task", run_task)
    results = experiment.run()
    print(results["run_task"][0])
    print(results["run_task"][1])

    results["run_task"][0].to_csv("results.csv", index=False)
    # Log the results
    logger.info("Experiment Results:")
    # logger.info(f"Base Score: {results['evaluate_results']['base_score']:.4f}")
    # logger.info(
    #     f"optimised Score: {results['evaluate_results']['optimised_score']:.4f}"
    # )
    # logger.info(f"Improvement: {results['evaluate_results']['improvement']:.4f}")
    logger.info(f"{experiment_name} completed successfully")


if __name__ == "__main__":
    main()
