"""
Simple Experiment Example

This script demonstrates how to use the core infrastructure to run a simple experiment.
"""

import sys
import pathlib
import time
from pathlib import Path

# Add the project root to the Python path
project_root = pathlib.Path(__file__).parent.parent.resolve()
sys.path.insert(0, str(project_root))

# Import logging utils first to set up global logger before other imports
from src.utils.logging_utils import setup_logger
from src.utils.decorator_utils import with_logger

# Now import other modules
from src.core import (
    config_manager,
    ExperimentRunner,
)
from src.utils.cli import default_parse_args, configure_logging

# Logger will be set up after CLI args are parsed
logger = None
log_path = None


@with_logger
def init_llm(config):
    """Initialise the LLM."""
    logger.info(f"Initialising LLM: {config['llm']['default']}")

    # In a real implementation, this would use the LLM registry
    llm_name = config["llm"]["default"]
    llm_config = config["llm"][llm_name]
    logger.info(f"LLM configuration: {llm_config}")

    return {
        "name": llm_name,
        "config": llm_config,
    }


@with_logger
def init_task(config, init_llm=None, **kwargs):
    """initialise the task."""
    logger.info(f"initialising task: {config['task']['default']}")

    # In a real implementation, this would use the task registry
    task_name = config["task"]["default"]
    task_config = config["task"][task_name]
    logger.info(f"Task configuration: {task_config}")

    return {
        "name": task_name,
        "config": task_config,
    }


@with_logger
def init_prompt_optimiser(config, init_task=None, init_llm=None, **kwargs):
    """initialise the prompt optimiser."""
    logger.info(
        f"initialising prompt optimiser: {config['prompt_optimiser']['default']}"
    )

    # In a real implementation, this would use the prompt optimiser registry
    optimiser_name = config["prompt_optimiser"]["default"]
    optimiser_config = config["prompt_optimiser"][optimiser_name]
    logger.info(f"Prompt optimiser configuration: {optimiser_config}")

    return {
        "name": optimiser_name,
        "config": optimiser_config,
    }


@with_logger
def run_base_prompt(init_llm, init_task, init_prompt_optimiser, **kwargs):
    """Run the task with the base prompt."""
    logger.info("Running task with base prompt")

    # In a real implementation, this would run the task with the base prompt

    # Log the inputs
    logger.info(f"Using LLM: {init_llm['name']}")
    logger.info(f"Using task: {init_task['name']}")

    # Simulate running the task
    score = 0.75
    logger.info(f"Base prompt score: {score:.4f}")

    return {
        "results": "Base prompt results",
        "score": score,
    }


@with_logger
def run_optimised_prompt(
    init_llm, init_task, init_prompt_optimiser, run_base_prompt=None, **kwargs
):
    """Run the task with the optimised prompt."""
    logger.info("Running task with optimised prompt")

    # In a real implementation, this would run the task with the optimised prompt

    # Log the inputs
    logger.info(f"Using LLM: {init_llm['name']}")
    logger.info(f"Using task: {init_task['name']}")
    logger.info(f"Using optimiser: {init_prompt_optimiser['name']}")

    # Simulate running the task with optimised prompt
    score = 0.85
    logger.info(f"optimised prompt score: {score:.4f}")

    return {
        "results": "optimised prompt results",
        "score": score,
    }


@with_logger
def evaluate_results(run_base_prompt, run_optimised_prompt, **kwargs):
    """Evaluate the results."""
    logger.info("Evaluating results")

    # In a real implementation, this would evaluate the results
    base_score = run_base_prompt["score"]
    optimised_score = run_optimised_prompt["score"]
    improvement = optimised_score - base_score
    improvement_percent = (improvement / base_score) * 100 if base_score != 0 else 0

    logger.info(f"Base Score: {base_score:.4f}")
    logger.info(f"optimised Score: {optimised_score:.4f}")
    logger.info(f"Improvement: {improvement:.4f} ({improvement_percent:.2f}%)")

    return {
        "base_score": base_score,
        "optimised_score": optimised_score,
        "improvement": improvement,
        "improvement_percent": improvement_percent,
    }


def main():
    """Run the experiment."""
    global logger, log_path

    # Parse command-line arguments first to get verbosity setting
    args = default_parse_args(
        description="Run a simple experiment",
    )
    # Configure logging
    configure_logging(args)

    # Set up experiment-specific logger AFTER CLI configuration
    experiment_name = "simple_experiment"
    logger, log_path = setup_logger(experiment_name, args=args)

    # Set up output directory
    timestamp = time.strftime("%Y%m%d-%H%M%S")
    output_dir = Path("output")
    log_dir = output_dir / f"{experiment_name}_{timestamp}"
    log_dir.mkdir(parents=True, exist_ok=True)

    # Use the experiment logger for the rest of the script
    logger.info(f"Starting {experiment_name}")

    # Load the experiment configuration
    logger.info(f"Loading configuration: {args.config}")
    config = config_manager.load_config(args.config)
    logger.info(f"Loaded config: {config}")

    # Create the experiment runner
    experiment_name = config["experiment"]["name"]
    logger.info(f"Creating experiment runner: {experiment_name}")
    experiment = ExperimentRunner(
        name=experiment_name,
        config=config,
        output_dir=log_dir.parent,  # Use the parent directory of our log_dir
    )

    # Add the experiment steps and run the experiment
    experiment.add_step("init_llm", init_llm, config=config)
    experiment.add_step("init_task", init_task, config=config)
    experiment.add_step("init_prompt_optimiser", init_prompt_optimiser, config=config)
    experiment.add_step("run_base_prompt", run_base_prompt)
    experiment.add_step("run_optimised_prompt", run_optimised_prompt)
    experiment.add_step("evaluate_results", evaluate_results)
    results = experiment.run()

    # Log the results
    logger.info("Experiment Results:")
    logger.info(f"Base Score: {results['evaluate_results']['base_score']:.4f}")
    logger.info(
        f"optimised Score: {results['evaluate_results']['optimised_score']:.4f}"
    )
    logger.info(f"Improvement: {results['evaluate_results']['improvement']:.4f}")
    logger.info(f"{experiment_name} completed successfully")


if __name__ == "__main__":
    main()
