"""
Tamper Detection Experiment

This script demonstrates how to use the prompt optimisation framework for
the tamper detection task.
"""

import sys
import pathlib
import logging
import numpy as np

# Add the project root to the Python path
project_root = pathlib.Path(__file__).parent.parent.resolve()
sys.path.insert(0, str(project_root))

# Import logging utils first to set up global logger before other imports
from src.utils.logging_utils import setup_global_logger, setup_experiment_logger, get_logger
from src.utils.decorator_utils import with_logger

# Set up global logger to capture logs from all imports
setup_global_logger()

from src.core import (
    config_manager,
    ExperimentRunner,
    llm_registry,
    task_registry,
    prompt_optimiser_registry,
    embeddings_registry,
)
from src.utils.cli import default_parse_args
from src.embeddings.embeddings_analyzer import create_embeddings_analyzer

# Set up logger for this script
logger = get_logger(__name__)


@with_logger
def init_llm(config):
    """initialise the LLM."""
    logger.info(f"Initialising LLM: {config['llm']['default']}")
    
    # Get the LLM configuration
    llm_name = config["llm"]["default"]
    llm_config = config["llm"][llm_name]
    
    logger.info(f"LLM configuration: {llm_config}")
    
    # Create the LLM instance
    llm = llm_registry.create(llm_name, **llm_config)
    
    logger.info(f"LLM initialised: {llm.model_info.get('name')} {llm.model_info.get('version_name')}")
    
    return llm


@with_logger
def init_task(config, init_llm=None, **kwargs):
    """initialise the task."""
    logger.info(f"Initialising task: {config['task']['default']}")
    
    # Get the task configuration
    task_name = config["task"]["default"]
    task_config = config["task"][task_name]
    
    logger.info(f"Task configuration: {task_config}")
    
    # Create the task instance
    task = task_registry.create(task_name, **task_config)
    
    logger.info(f"Task initialised: {task_name}")
    
    return task


@with_logger
def init_prompt_optimiser(config, init_task=None, init_llm=None, **kwargs):
    """initialise the prompt optimiser if specified in the config."""
    # Check if prompt_optimiser is in the config
    if "prompt_optimiser" not in config:
        logger.info("No prompt optimiser specified in config, skipping initialisation")
        return None
    
    logger.info(f"Initialising prompt optimiser: {config['prompt_optimiser']['default']}")
    
    # Get the prompt optimiser configuration
    optimiser_name = config["prompt_optimiser"]["default"]
    optimiser_config = config["prompt_optimiser"][optimiser_name]
    
    logger.info(f"Prompt optimiser configuration: {optimiser_config}")
    
    # Create the prompt optimiser instance
    optimiser = prompt_optimiser_registry.create(optimiser_name, **optimiser_config)
    
    logger.info(f"Prompt optimiser initialised: {optimiser_name}")
    
    return optimiser


@with_logger
def run_base_prompt(llm=None, task=None, prompt_optimiser=None, init_llm=None, init_task=None, init_prompt_optimiser=None, **kwargs):
    """Run the task with the base prompt."""
    logger.info("Running task with base prompt")
    
    # Use the parameters from init_* steps if the direct parameters are None
    llm = llm or init_llm
    task = task or init_task
    prompt_optimiser = prompt_optimiser or init_prompt_optimiser
    
    # Check if we have the required parameters
    if not llm:
        logger.error("No LLM provided")
        return {"error": "No LLM provided", "score": 0.0}
    
    if not task:
        logger.error("No task provided")
        return {"error": "No task provided", "score": 0.0}
    
    # Run the task with the base prompt
    logger.info("Executing task.run() with base prompt")
    results_df, score = task.run(llm)
    
    logger.info(f"Base prompt score: {score:.4f}")
    logger.info(f"Results dataframe shape: {results_df.shape}")
    
    return {
        "results_df": results_df,
        "score": score,
    }

@with_logger
def run_pre_optimised_prompt(config, llm=None, task=None, prompt_optimiser=None, init_llm=None, init_task=None, init_prompt_optimiser=None, **kwargs):
    """Run the task with the pre-optimised prompt."""
    logger.info("RUNNING PRE OPTIMISED")
    if not config["pre_optimised_prompts"]["enable_all_pre_optimised_prompts"]:
        logger.info("Pre-optimised prompts are not enabled, skipping this step")
        return {"skipped": True, "reason": "Pre-optimised prompts not enabled", "score": 0.0}

    logger.info("Running task with pre-optimised prompts")
    
    # Use the parameters from init_* steps if the direct parameters are None
    llm = llm or init_llm
    task = task or init_task
    prompt_optimiser = prompt_optimiser or init_prompt_optimiser
    
    # Check if we have the required parameters
    if not llm:
        logger.error("No LLM provided")
        return {"error": "No LLM provided", "score": 0.0}
    
    if not task:
        logger.error("No task provided")
        return {"error": "No task provided", "score": 0.0}
    
    # Update the task's prompt template with the pre-optimised prompt if manual optimisation is enabled
    if config["pre_optimised_prompts"]["enable_manual_optimised_prompt"]:
        manual_optimised_prompt = config["pre_optimised_prompts"]["tone"]["manual_optimised_prompt"]
        logger.info("Using manual pre-optimised prompt")
        logger.info(f"Pre-optimised prompt length: {len(manual_optimised_prompt)} characters")
        
        # Create a copy of the task to avoid modifying the original
        import copy
        task_copy = copy.deepcopy(task)
        task_copy.prompt_msg_template[0]["content"] = manual_optimised_prompt
        logger.info("Updated task prompt template with pre-optimised prompt")
        
        # Run the task with the pre-optimised prompt
        logger.info("Executing task.run() with pre-optimised prompt")
        results_df, score = task_copy.run(llm)
    else:
        # Run the task with the base prompt
        logger.info("No manual pre-optimised prompt specified, using base prompt")
        logger.info("Executing task.run() with base prompt")
        results_df, score = task.run(llm)
    
    logger.info(f"Pre-optimised prompt score: {score:.4f}")
    logger.info(f"Results dataframe shape: {results_df.shape}")
    
    return {
        "results_df": results_df,
        "score": score,
    }


@with_logger
def optimise_prompt(llm=None, task=None, prompt_optimiser=None, base_results=None,
                   init_llm=None, init_task=None, init_prompt_optimiser=None, **kwargs):
    """Optimise the prompt if an optimiser is available."""
    # Use the parameters from init_* steps if the direct parameters are None
    llm = llm or init_llm
    task = task or init_task
    prompt_optimiser = prompt_optimiser or init_prompt_optimiser
    
    # Check if we have the required parameters
    if not llm:
        logger.error("No LLM provided")
        return {"error": "No LLM provided", "base_prompt": "", "optimised_prompt": ""}
    
    if not task:
        logger.error("No task provided")
        return {"error": "No task provided", "base_prompt": "", "optimised_prompt": ""}
    
    # Get the base prompt
    base_prompt = task.prompt_msg_template[0]["content"]
    logger.info(f"Base prompt length: {len(base_prompt)} characters")
    
    # If no prompt optimiser is provided, return the base prompt as the optimised prompt
    if not prompt_optimiser:
        logger.info("No prompt optimiser provided, using base prompt as optimised prompt")
        return {
            "base_prompt": base_prompt,
            "optimised_prompt": base_prompt,
        }
    
    logger.info("Optimising prompt")
    
    # Get the feedback function
    logger.info("Getting feedback function from task")
    feedback_function = task.get_feedback_function(llm=llm)
    
    # optimise the prompt
    logger.info("Starting prompt optimisation process")
    optimised_prompt = prompt_optimiser.optimise(base_prompt, feedback_function)
    
    logger.info("Prompt optimisation complete")
    logger.info(f"optimised prompt length: {len(optimised_prompt)} characters")
    
    return {
        "base_prompt": base_prompt,
        "optimised_prompt": optimised_prompt,
    }


@with_logger
def run_optimised_prompt(llm=None, task=None, prompt_optimiser=None, optimise_prompt=None,
                        run_base_prompt=None, init_llm=None, init_task=None, init_prompt_optimiser=None, **kwargs):
    """Run the task with the optimised prompt."""
    logger.info("Running task with optimised prompt")
    
    # Use the parameters from init_* steps if the direct parameters are None
    llm = llm or init_llm
    task = task or init_task
    prompt_optimiser = prompt_optimiser or init_prompt_optimiser
    
    # Check if we have the required parameters
    if not llm:
        logger.error("No LLM provided")
        return {"error": "No LLM provided", "score": 0.0}
    
    if not task:
        logger.error("No task provided")
        return {"error": "No task provided", "score": 0.0}
    
    if not optimise_prompt:
        logger.error("No optimised prompt provided")
        return {"error": "No optimised prompt provided", "score": 0.0}
    
    # Check if the optimised prompt is different from the base prompt
    if optimise_prompt["optimised_prompt"] == optimise_prompt["base_prompt"]:
        logger.info("Optimised prompt is the same as base prompt (no optimisation performed)")
        # If we're not using an optimiser, just return the base results
        if not prompt_optimiser and run_base_prompt:
            logger.info("No optimiser used, returning base prompt results")
            return {
                "results_df": run_base_prompt["results_df"],
                "score": run_base_prompt["score"],
            }
    
    # Update the task's prompt template with the optimised prompt
    task_copy = task
    task_copy.prompt_msg_template[0]["content"] = optimise_prompt["optimised_prompt"]
    logger.info("Updated task prompt template with optimised prompt")
    
    # Run the task with the optimised prompt
    logger.info("Executing task.run() with optimised prompt")
    results_df, score = task_copy.run(llm)
    
    logger.info(f"optimised prompt score: {score:.4f}")
    logger.info(f"Results dataframe shape: {results_df.shape}")
    
    return {
        "results_df": results_df,
        "score": score,
    }


@with_logger
def evaluate_results(run_base_prompt=None, run_pre_optimised_prompt=None, optimise_prompt=None, run_optimised_prompt=None, **kwargs):
    """Evaluate the results."""
    logger.info("Evaluating results")
    
    # Check if we have the required parameters
    if not run_base_prompt:
        logger.error("No base prompt results provided")
        return {"error": "No base prompt results provided"}
    
    if not run_pre_optimised_prompt:
        logger.error("No pre-optimised prompt results provided")
        return {"error": "No pre-optimised prompt results provided"}
    
    if not optimise_prompt:
        logger.error("No optimised prompt provided")
        return {"error": "No optimised prompt provided"}
    
    if not run_optimised_prompt:
        logger.error("No optimised prompt results provided")
        return {"error": "No optimised prompt results provided"}
    
    # Check for errors in previous steps
    if "error" in run_base_prompt:
        logger.error(f"Error in base prompt run: {run_base_prompt['error']}")
        return {"error": f"Error in base prompt run: {run_base_prompt['error']}"}
    
    if "error" in run_pre_optimised_prompt:
        logger.error(f"Error in pre-optimised prompt run: {run_pre_optimised_prompt['error']}")
        return {"error": f"Error in pre-optimised prompt run: {run_pre_optimised_prompt['error']}"}
    
    # Handle case where pre-optimised prompts were skipped
    if "skipped" in run_pre_optimised_prompt:
        logger.info("Pre-optimised prompts were skipped, using base score as fallback")
        pre_optimised_score = run_base_prompt["score"]
    else:
        pre_optimised_score = run_pre_optimised_prompt["score"]
    
    if "error" in optimise_prompt:
        logger.error(f"Error in prompt optimisation: {optimise_prompt['error']}")
        return {"error": f"Error in prompt optimisation: {optimise_prompt['error']}"}
    
    if "error" in run_optimised_prompt:
        logger.error(f"Error in optimised prompt run: {run_optimised_prompt['error']}")
        return {"error": f"Error in optimised prompt run: {run_optimised_prompt['error']}"}
    
    # Calculate improvement
    base_score = run_base_prompt["score"]
    optimised_score = run_optimised_prompt["score"]

    improvement = optimised_score - base_score
    improvement_percent = (improvement / base_score) * 100 if base_score != 0 else 0
    
    logger.info(f"Base Score: {base_score:.4f}")
    logger.info(f"optimised Score: {optimised_score:.4f}")
    logger.info(f"Improvement: {improvement:.4f} ({improvement_percent:.2f}%)")
    logger.info(f"Pre-optimised Score: {pre_optimised_score:.4f}")
    
    # Check if optimisation was performed
    if optimise_prompt["base_prompt"] == optimise_prompt["optimised_prompt"]:
        logger.info("No optimisation was performed (base prompt = optimised prompt)")
    else:
        # Log detailed comparison
        logger.info("Detailed comparison:")
        logger.info(f"  Base prompt length: {len(optimise_prompt['base_prompt'])} characters")
        logger.info(f"  optimised prompt length: {len(optimise_prompt['optimised_prompt'])} characters")
        # logger.info(f"  Pre-optimised prompt length: {len(optimise_prompt['base_prompt'])} characters")
    
    return {
        "base_score": base_score,
        "optimised_score": optimised_score,
        "improvement": improvement,
        "improvement_percent": improvement_percent,
        "pre_optimised_prompt_score": pre_optimised_score,
    }


@with_logger
def init_embeddings(config, **kwargs):
    """Initialize the embeddings model if specified in the config."""
    # Check if embeddings is in the config
    if "embeddings" not in config:
        logger.info("No embeddings specified in config, skipping initialization")
        return None
    
    embeddings_config = config["embeddings"]
    
    if not embeddings_config.get('enabled', False):
        logger.info("Embeddings analysis is disabled in configuration")
        return None
    
    logger.info(f"Initializing embeddings: {embeddings_config['default']}")
    
    # Get the embeddings configuration
    embeddings_name = embeddings_config["default"]
    embeddings_model_config = embeddings_config[embeddings_name]
    
    logger.info(f"Embeddings configuration: {embeddings_model_config}")
    
    # For embeddings, we need to provide input texts, but we'll do that later
    # For now, just store the config for later use
    return {
        "name": embeddings_name,
        "config": embeddings_model_config,
        "enabled": True
    }


@with_logger
def generate_embeddings_analysis(
    run_base_prompt=None,
    run_optimised_prompt=None,
    optimise_prompt=None,
    init_embeddings=None,
    init_prompt_optimiser=None,
    config=None,
    **kwargs
):
    """Generate embeddings analysis from experiment results."""
    logger.info("Starting embeddings analysis")
    
    # Check if embeddings is enabled
    if not init_embeddings or not init_embeddings.get('enabled', False):
        logger.info("Embeddings analysis not enabled, skipping")
        return {"skipped": True, "reason": "Embeddings analysis not enabled"}
    
    # Check if we have the required results
    if not optimise_prompt:
        logger.warning("No prompt optimization results available for embeddings analysis")
        return {"error": "No prompt optimization results available"}
    
    # Extract prompts for analysis
    prompts = []
    labels = []
    
    # Get base prompt
    if 'base_prompt' in optimise_prompt:
        prompts.append(optimise_prompt['base_prompt'])
        labels.append("Base Prompt")
    
    # Get optimized prompt (if different)
    if 'optimised_prompt' in optimise_prompt:
        optimized_prompt = optimise_prompt['optimised_prompt']
        if optimized_prompt != optimise_prompt.get('base_prompt', ''):
            prompts.append(optimized_prompt)
            labels.append("Optimized Prompt")
    
    # Get all generated prompts from the tone optimizer if available
    if init_prompt_optimiser and hasattr(init_prompt_optimiser, 'random_initialised_prompts'):
        logger.info("Retrieving generated prompts from tone optimizer")
        try:
            generated_prompts = init_prompt_optimiser.random_initialised_prompts
            logger.info(generated_prompts)
            logger.info(f"Found {len(generated_prompts)} randomly generated prompts in random_initialised_prompts")
            
            # Add unique generated prompts (avoid duplicates)
            for i, generated_prompt in enumerate(generated_prompts):
                if generated_prompt not in prompts:
                    prompts.append(generated_prompt)
                    labels.append(f"Randomly initialised Prompt {i+1}")
            
            logger.info(f"Added {len(generated_prompts)} randomly generated prompts to embeddings analysis")
        except Exception as e:
            logger.warning(f"Could not retrieve generated prompts: {str(e)}")
    
    if len(prompts) < 1:
        logger.error("No prompts found for embeddings analysis")
        return {"error": "No prompts found for analysis"}
    
    # If we only have one prompt, duplicate it for comparison
    if len(prompts) == 1:
        prompts.append(prompts[0])
        labels.append("Base Prompt (Copy)")
        logger.info("Only one unique prompt found, duplicating for comparison")
    
    if config["pre_optimised_prompts"]["enable_manual_optimised_prompt"]:
        manual_optimised_prompt = config["pre_optimised_prompts"]["tone"]["manual_optimised_prompt"]
        prompts.append(manual_optimised_prompt)
        labels.append("Pre Optimised Prompt Manual")

    try:
        # Create embeddings instance with the prompts
        embeddings_name = init_embeddings["name"]
        embeddings_config = init_embeddings["config"].copy()
        
        logger.info(f"Creating embeddings instance: {embeddings_name}")
        embeddings_instance = embeddings_registry.create(
            embeddings_name,
            input=prompts,
            **embeddings_config
        )
        
        # Generate embeddings and similarity analysis
        logger.info("Generating embeddings and heatmap")
        fig, ax = embeddings_instance.generate(prompts)
        
        # Get similarity matrix for additional metrics
        embeddings_data = embeddings_instance.get_embeddings(prompts)
        similarity_df = embeddings_instance.similarity_matrix(embeddings_data)
        
        # Calculate similarity metrics
        similarity_metrics = {}
        if similarity_df.shape[0] >= 2:
            # Get off-diagonal values (excluding self-similarity)
            mask = np.ones(similarity_df.shape, dtype=bool)
            np.fill_diagonal(mask, False)
            off_diagonal_values = similarity_df.values[mask]
            
            similarity_metrics['mean_similarity'] = float(np.mean(off_diagonal_values))
            similarity_metrics['max_similarity'] = float(np.max(off_diagonal_values))
            similarity_metrics['min_similarity'] = float(np.min(off_diagonal_values))
            similarity_metrics['std_similarity'] = float(np.std(off_diagonal_values))
            
            # If we have exactly 2 prompts, get the direct similarity
            if similarity_df.shape[0] == 2:
                similarity_metrics['prompt_similarity'] = float(similarity_df.iloc[0, 1])
        
        # Customize the plot with labels
        if len(labels) == len(prompts):
            ax.set_xticklabels(labels, rotation=45, ha='right')
            ax.set_yticklabels(labels, rotation=0)
        
        logger.info("Embeddings analysis completed successfully")
        logger.info(f"Analyzed {len(prompts)} prompts")
        
        # Log similarity metrics
        if similarity_metrics:
            logger.info("Similarity metrics:")
            for key, value in similarity_metrics.items():
                logger.info(f"  {key}: {value:.4f}")
        
        return {
            "prompts": prompts,
            "labels": labels,
            "similarity_matrix": similarity_df,
            "similarity_metrics": similarity_metrics,
            "figure": fig,
            "axes": ax,
            "embeddings_instance": embeddings_instance
        }
        
    except Exception as e:
        logger.error(f"Error during embeddings analysis: {str(e)}", exc_info=True)
        return {"error": str(e)}


@with_logger
def save_embeddings_results(
    generate_embeddings_analysis=None,
    config=None,
    **kwargs
):
    """Save embeddings analysis results to configured output paths."""
    logger.info("Saving embeddings analysis results")
    
    if not generate_embeddings_analysis:
        logger.info("No embeddings analysis results to save")
        return {"skipped": True}
    
    if "error" in generate_embeddings_analysis or "skipped" in generate_embeddings_analysis:
        logger.info("Skipping embeddings results saving due to previous step issues")
        return generate_embeddings_analysis
    
    try:
        # Get experiment name from config
        experiment_name = config.get("experiment", {}).get("name", "experiment") if config else "experiment"
        
        # The figure and similarity matrix are available from the analysis
        fig = generate_embeddings_analysis.get("figure")
        similarity_df = generate_embeddings_analysis.get("similarity_matrix")
        
        if fig is None or similarity_df is None:
            logger.error("Missing figure or similarity matrix for saving")
            return {"error": "Missing required data for saving"}
        
        # Get configured output paths
        output_config = config.get("output", {}) if config else {}
        
        saved_files = {}
        
        # Save heatmap to configured path
        if "embeddings_heatmap" in output_config:
            heatmap_path = pathlib.Path(output_config["embeddings_heatmap"])
            heatmap_path.parent.mkdir(parents=True, exist_ok=True)
            
            try:
                fig.savefig(heatmap_path, bbox_inches='tight', dpi=300)
                saved_files['heatmap'] = str(heatmap_path)
                logger.info(f"Saved heatmap to configured path: {heatmap_path}")
            except Exception as e:
                logger.error(f"Failed to save heatmap: {str(e)}")
        
        # Save similarity matrix to configured path
        if "embeddings_similarity" in output_config:
            similarity_path = pathlib.Path(output_config["embeddings_similarity"])
            similarity_path.parent.mkdir(parents=True, exist_ok=True)
            
            try:
                similarity_df.to_csv(similarity_path, index=True)
                saved_files['similarity_matrix'] = str(similarity_path)
                logger.info(f"Saved similarity matrix to configured path: {similarity_path}")
            except Exception as e:
                logger.error(f"Failed to save similarity matrix: {str(e)}")
        
        # If no configured paths, save to default location
        if not saved_files:
            logger.info("No configured output paths found, using default location")
            output_dir = pathlib.Path("output") / experiment_name
            output_dir.mkdir(parents=True, exist_ok=True)
            
            # Save heatmap
            heatmap_path = output_dir / "embeddings_heatmap.png"
            try:
                fig.savefig(heatmap_path, bbox_inches='tight', dpi=300)
                saved_files['heatmap'] = str(heatmap_path)
                logger.info(f"Saved heatmap to default path: {heatmap_path}")
            except Exception as e:
                logger.error(f"Failed to save heatmap: {str(e)}")
            
            # Save similarity matrix
            similarity_path = output_dir / "similarity_matrix.csv"
            try:
                similarity_df.to_csv(similarity_path, index=True)
                saved_files['similarity_matrix'] = str(similarity_path)
                logger.info(f"Saved similarity matrix to default path: {similarity_path}")
            except Exception as e:
                logger.error(f"Failed to save similarity matrix: {str(e)}")
        
        logger.info(f"Successfully saved {len(saved_files)} embeddings files")
        
        return {
            "saved_files": saved_files,
            "experiment_name": experiment_name,
            "success": True
        }
        
    except Exception as e:
        logger.error(f"Error saving embeddings results: {str(e)}", exc_info=True)
        return {"error": str(e)}


def main():
    """Run the experiment."""
    # Default experiment name, can be overridden by command-line arguments
    experiment_name = "tamper_detection_no_opt"

    # Parse command-line arguments first to get verbosity setting
    args = default_parse_args(
        description="Run tamper detection experiment",
        default_config=experiment_name
    )
    
    # Set up experiment-specific logger and get the output directory
    experiment_logger, log_dir = setup_experiment_logger(
        experiment_name=experiment_name,
        args=args
    )
    
    # Use the experiment logger for the rest of the script
    experiment_logger.info("Starting tamper detection experiment")

    # Load the experiment configuration
    experiment_logger.info(f"Loading configuration: {args.config}")
    config = config_manager.load_config(args.config, args=args)
    experiment_logger.info(f"Loaded config: {config}")

    # Create the experiment runner
    experiment_name = config["experiment"]["name"]
    experiment_logger.info(f"Creating experiment runner: {experiment_name}")
    experiment = ExperimentRunner(
        name=experiment_name,
        config=config,
        log_level=logging.DEBUG,  # Always use DEBUG level for now
        output_dir=log_dir.parent  # Use the parent directory of our log_dir
    )
    
    # Add the experiment steps
    experiment.add_step("init_llm", init_llm, config=config)
    experiment.add_step("init_task", init_task, config=config)
    experiment.add_step("init_prompt_optimiser", init_prompt_optimiser, config=config)
    experiment.add_step("init_embeddings", init_embeddings, config=config)
    experiment.add_step("run_base_prompt", run_base_prompt)

    # ? Adding preoptimised prompts
    experiment.add_step("run_pre_optimised_prompt", run_pre_optimised_prompt, config=config)

    experiment.add_step("optimise_prompt", optimise_prompt)
    experiment.add_step("run_optimised_prompt", run_optimised_prompt)
    experiment.add_step("evaluate_results", evaluate_results)
    experiment.add_step("generate_embeddings_analysis", generate_embeddings_analysis, config=config)
    experiment.add_step("save_embeddings_results", save_embeddings_results, config=config)
    
    # Run the experiment
    results = experiment.run()
    
    # Log the results
    logger.info("\nExperiment Results:")
    logger.info(f"Base Score: {results['evaluate_results']['base_score']:.4f}")
    logger.info(f"optimised Score: {results['evaluate_results']['optimised_score']:.4f}")
    logger.info(f"Improvement: {results['evaluate_results']['improvement']:.4f} ({results['evaluate_results']['improvement_percent']:.2f}%)")
    
    logger.info("Experiment completed successfully")


if __name__ == "__main__":
    main()