"""
Configuration Management System

This module provides a centralized configuration management system that can
load, validate, and provide access to configuration settings from YAML files.
"""

import argparse
import pathlib
import re
from typing import Any, Dict, Optional, Union

import jsonschema
import yaml

from src.utils.decorator_utils import with_logger


class ConfigManager:
    """
    Configuration manager for loading and validating configuration files.

    This class handles loading configuration from YAML files, validating them
    against JSON schemas, and providing access to the configuration settings.
    """

    @with_logger
    def __init__(
        self,
        config_dir: Optional[Union[str, pathlib.Path]] = None,
        schema_dir: Optional[Union[str, pathlib.Path]] = None,
        default_config: Optional[str] = None,
    ):
        """
        Initialise the configuration manager.

        Args:
            config_dir: Directory containing configuration files
            schema_dir: Directory containing schema files
            default_config: Name of the default configuration file
        """
        # logger.info("Initialising configuration manager")

        # Get the project root directory
        self.project_root = pathlib.Path(__file__).parent.parent.parent.resolve()

        # Set the configuration directory
        if config_dir is None:
            self.config_dir = self.project_root / "config"
        else:
            self.config_dir = pathlib.Path(config_dir)

        # logger.info(f"Configuration directory: {self.config_dir}")

        # Set the schema directory
        if schema_dir is None:
            self.schema_dir = self.config_dir / "schemas"
        else:
            self.schema_dir = pathlib.Path(schema_dir)

        # logger.info(f"Schema directory: {self.schema_dir}")

        # Set the default configuration file
        self.default_config = default_config or "default.yaml"
        # logger.info(f"Default configuration file: {self.default_config}")

        # initialise the configuration dictionary
        self.config: Dict[str, Any] = {}

        # Load the default configuration
        self._load_default_config()

    @with_logger
    def _load_default_config(self) -> None:
        """
        Load the default configuration file.

        Raises:
            FileNotFoundError: If the default configuration file is not found
        """
        default_config_path = self.config_dir / self.default_config
        # logger.info(f"Loading default configuration from: {default_config_path}")

        # Create default config if it doesn't exist
        if not default_config_path.exists():
            # logger.warning(
            #     f"Default configuration file not found, creating: {default_config_path}"
            # )
            self._create_default_config(default_config_path)

        # Load the default configuration
        try:
            with open(default_config_path, "r") as f:
                self.config = yaml.safe_load(f) or {}
            # logger.info(
            #     f"Loaded default configuration with {len(self.config)} top-level keys"
            # )
        except Exception:
            # logger.error(
            #     f"Error loading default configuration: {str(e)}", exc_info=True
            # )
            raise

    @with_logger
    def _create_default_config(self, config_path: pathlib.Path) -> None:
        """
        Create a default configuration file.

        Args:
            config_path: Path to the configuration file to create
        """
        # logger.info(f"Creating default configuration file: {config_path}")

        # Create the directory if it doesn't exist
        config_path.parent.mkdir(parents=True, exist_ok=True)

        # Create a basic default configuration
        default_config = {
            "project": {
                "name": "prompt_optimisation",
                "version": "0.1.0",
            },
            "paths": {
                "data": str(self.project_root / "data"),
                "output": str(self.project_root / "output"),
            },
            "logging": {
                "level": "INFO",
                "format": "%(asctime)s - %(name)s - %(levelname)s - %(message)s",
                "file_logging": True,
                "log_dir": str(self.project_root / "logs"),
            },
        }

        # Write the default configuration to the file
        try:
            with open(config_path, "w") as f:
                yaml.dump(default_config, f, default_flow_style=False)
            # logger.info("Default configuration file created successfully")
        except Exception:
            # logger.error(
            #     f"Error creating default configuration file: {str(e)}", exc_info=True
            # )
            raise

    def load_config(
        self, config_name: str, args: Optional[Any] = None
    ) -> Dict[str, Any]:
        """
        Load a configuration file, merge it with the default configuration,
        and validate it against the appropriate schema.

        Args:
            config_name: Name of the configuration file to load
            args: Optional command-line arguments to override configuration

        Returns:
            The merged and validated configuration dictionary

        Raises:
            FileNotFoundError: If the configuration file is not found
            jsonschema.exceptions.ValidationError: If the configuration is invalid
        """
        # Determine the configuration file path
        if not config_name.endswith(".yaml") and not config_name.endswith(".yml"):
            config_name += ".yaml"

        config_path = self.config_dir / "experiments" / config_name

        # Check if the configuration file exists
        if not config_path.exists():
            raise FileNotFoundError(f"Configuration file not found: {config_path}")

        # Load the configuration file
        try:
            with open(config_path, "r") as f:
                config = yaml.safe_load(f) or {}
        except Exception as e:
            raise Exception(f"Failed loading configuration file: {str(e)})")

        # Merge with the default configuration
        merged_config = self._merge_configs(self.config, config)

        # Check for command-line arguments that might override configuration
        self._apply_cli_overrides(merged_config, args)

        # Validate the configuration against the experiment schema
        try:
            self.validate_config(merged_config, "experiment")
        except FileNotFoundError:
            pass
        except jsonschema.exceptions.ValidationError as e:
            raise Exception(f"Configuration validation error: {e}")

        # Update the internal configuration with the merged configuration
        self.config = merged_config

        return merged_config

    @with_logger
    def _merge_configs(
        self, base_config: Dict[str, Any], override_config: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Merge two configuration dictionaries.

        Args:
            base_config: The base configuration dictionary
            override_config: The configuration dictionary to override with

        Returns:
            The merged configuration dictionary
        """
        merged = base_config.copy()

        # logger.debug(f"Merging configs:")
        # logger.debug(f"Base config: {base_config}")
        # logger.debug(f"Override config: {override_config}")

        for key, value in override_config.items():
            # logger.debug(f"Processing key: {key}, value: {value}")
            # If the value is a dictionary and the key exists in the base config
            if (
                isinstance(value, dict)
                and key in merged
                and isinstance(merged[key], dict)
            ):
                # logger.debug(f"Recursively merging dictionaries for key: {key}")
                # logger.debug(f"Base value: {merged[key]}")
                # logger.debug(f"Override value: {value}")
                # Recursively merge the dictionaries
                merged[key] = self._merge_configs(merged[key], value)
                # logger.debug(f"Merged result for key {key}: {merged[key]}")
            else:
                # Otherwise, override the value
                # logger.debug(f"Directly overriding value for key: {key}")
                # logger.debug(f"Old value: {merged.get(key)}")
                # logger.debug(f"New value: {value}")
                merged[key] = value

        # logger.debug(f"Final merged result: {merged}")
        return merged

    @with_logger
    def validate_config(self, config: Dict[str, Any], schema_name: str) -> None:
        """
        Validate a configuration against a JSON schema.

        Args:
            config: The configuration dictionary to validate
            schema_name: The name of the schema to validate against

        Raises:
            FileNotFoundError: If the schema file is not found
            jsonschema.exceptions.ValidationError: If the configuration is invalid
        """
        logger.debug(f"Validating configuration against schema: {schema_name}")

        # Determine the schema file path
        if not schema_name.endswith(".json"):
            schema_name += ".json"

        schema_path = self.schema_dir / schema_name
        logger.debug(f"Schema file path: {schema_path}")

        # Check if the schema file exists
        if not schema_path.exists():
            logger.error(f"Schema file not found: {schema_path}")
            raise FileNotFoundError(f"Schema file not found: {schema_path}")

        # Load the schema file
        try:
            with open(schema_path, "r") as f:
                schema = yaml.safe_load(f)
        except Exception as e:
            logger.error(f"Error loading schema file: {str(e)}", exc_info=True)
            raise

        # Validate the configuration against the schema
        try:
            jsonschema.validate(config, schema)
        except jsonschema.exceptions.ValidationError as e:
            logger.error(f"Configuration validation error: {str(e)}")
            raise

    def get(self, key: str, default: Any = None) -> Any:
        """
        Get a configuration value by key.

        Args:
            key: The key to get the value for (can be a dotted path)
            default: The default value to return if the key is not found

        Returns:
            The configuration value, or the default value if not found
        """
        # Split the key into parts
        parts = key.split(".")

        # Start with the entire configuration
        value = self.config

        # Traverse the configuration dictionary
        for part in parts:
            if isinstance(value, dict) and part in value:
                value = value[part]
            else:
                return default

        return value

    def set(self, key: str, value: Any) -> None:
        """
        Set a configuration value by key.

        Args:
            key: The key to set the value for (can be a dotted path)
            value: The value to set
        """
        # Split the key into parts
        parts = key.split(".")

        # Start with the entire configuration
        config = self.config

        # Traverse the configuration dictionary
        for i, part in enumerate(parts[:-1]):
            if part not in config:
                config[part] = {}
            elif not isinstance(config[part], dict):
                config[part] = {}

            config = config[part]

        # Set the value
        config[parts[-1]] = value

    @with_logger
    def save(self, config_name: Optional[str] = None) -> None:
        """
        Save the current configuration to a file.

        Args:
            config_name: The name of the configuration file to save to
                         (defaults to the default configuration file)
        """
        # Determine the configuration file path
        if config_name is None:
            config_path = self.config_dir / self.default_config
            # logger.info(f"Saving configuration to default file: {config_path}")
        else:
            if not config_name.endswith(".yaml") and not config_name.endswith(".yml"):
                config_name += ".yaml"

            config_path = self.config_dir / "experiments" / config_name
            # logger.info(f"Saving configuration to file: {config_path}")

        # Create the directory if it doesn't exist
        config_path.parent.mkdir(parents=True, exist_ok=True)

        # Write the configuration to the file
        try:
            with open(config_path, "w") as f:
                yaml.dump(self.config, f, default_flow_style=False)
            # logger.info(f"Configuration saved successfully to: {config_path}")
        except Exception:
            # logger.error(f"Error saving configuration: {str(e)}", exc_info=True)
            raise

    @with_logger
    def _apply_cli_overrides(
        self, config: Dict[str, Any], args: Optional[argparse.Namespace] = None
    ) -> None:
        """
        Apply command-line argument overrides to the configuration.

        This method checks for command-line arguments that might override
        configuration settings, particularly the data path.

        Args:
            config: The configuration dictionary to modify
            args: Optional pre-parsed command-line arguments
        """
        import argparse

        try:
            # If args are provided directly, use them
            if args is not None:
                if hasattr(args, "data_path") and args.data_path:
                    # logger.info(
                    #     f"Overriding data path from command line: {args.data_path}"
                    # )
                    if "paths" not in config:
                        config["paths"] = {}
                    config["paths"]["data"] = args.data_path
                return

            # Otherwise, try to find args in the current stack frames
            import sys

            # Only process if we're running as a script (not during import)
            if not sys.argv or sys.argv[0].endswith("__main__.py"):
                return

            # Check if argparse has already parsed arguments
            try:
                for frame in sys._current_frames().values():
                    if "args" in frame.f_locals and isinstance(
                        frame.f_locals["args"], argparse.Namespace
                    ):
                        args = frame.f_locals["args"]
                        if hasattr(args, "data_path") and args.data_path:
                            # logger.info(
                            #     f"Overriding data path from command line: {args.data_path}"
                            # )
                            if "paths" not in config:
                                config["paths"] = {}
                            config["paths"]["data"] = args.data_path
                        break
            except Exception:
                # Don't fail if we can't access frame locals
                pass
        except Exception as e:
            raise e
            # Don't fail if we can't get command-line arguments
            # logger.warning(f"Could not process command-line arguments: {str(e)}")

    @with_logger
    def resolve_variables(self, config: Dict[str, Any]) -> Dict[str, Any]:
        """
        Resolve variable interpolations in configuration values.

        Supports syntax like ${components.llm.default} to reference configuration values.

        Args:
            config: Configuration dictionary to process

        Returns:
            Configuration dictionary with variables resolved
        """
        logger.debug("Resolving variables in configuration")
        return self._resolve_variables_recursive(config, config)

    def _resolve_variables_recursive(
        self, value: Any, root_config: Dict[str, Any]
    ) -> Any:
        """
        Recursively resolve variables in configuration values.

        Args:
            value: The value to process (can be dict, list, string, etc.)
            root_config: The root configuration for variable resolution

        Returns:
            The value with variables resolved
        """
        if isinstance(value, dict):
            return {
                k: self._resolve_variables_recursive(v, root_config)
                for k, v in value.items()
            }
        elif isinstance(value, list):
            return [
                self._resolve_variables_recursive(item, root_config) for item in value
            ]
        elif isinstance(value, str):
            return self._resolve_string_variables(value, root_config)
        else:
            return value

    @with_logger
    def _resolve_string_variables(self, value: str, root_config: Dict[str, Any]) -> str:
        """
        Resolve variable interpolations in a string value.

        Args:
            value: String that may contain variable references like ${path.to.value}
            root_config: The root configuration for variable resolution

        Returns:
            String with variables resolved
        """
        if "${" not in value:
            return value

        # Find all variable references
        pattern = r"\$\{([^}]+)\}"
        matches = re.findall(pattern, value)

        resolved_value = value
        for match in matches:
            var_path = match.strip()
            try:
                resolved = self._get_nested_config_value(var_path, root_config)
                placeholder = "${" + match + "}"
                resolved_value = resolved_value.replace(placeholder, str(resolved))
            except KeyError:
                # Leave unresolved variables as-is and log a warning
                logger.warning(f"Variable not found: {var_path}")

        return resolved_value

    def _get_nested_config_value(self, path: str, config: Dict[str, Any]) -> Any:
        """
        Get a nested value from the configuration using dot notation.

        Args:
            path: Dot-separated path like "components.llm.default"
            config: Configuration dictionary to search in

        Returns:
            The value at the specified path

        Raises:
            KeyError: If the path is not found
        """
        parts = path.split(".")
        current = config

        for part in parts:
            if isinstance(current, dict) and part in current:
                current = current[part]
            else:
                raise KeyError(f"Path not found: {path}")

        return current

    @with_logger
    def validate_step_based_config(self, config: Dict[str, Any]) -> bool:
        """
        Validate a step-based configuration.

        Args:
            config: Configuration dictionary to validate

        Returns:
            True if valid, False otherwise
        """
        try:
            # Check if it's a step-based configuration
            if "steps" not in config:
                logger.debug(
                    "Configuration does not contain steps - using legacy validation"
                )
                return True

            # Validate against the updated schema
            self.validate_config(config, "experiment")

            # Additional step-based validation
            steps = config["steps"]
            step_names = set()

            for step in steps:
                step_name = step.get("name")
                if not step_name:
                    logger.error("Step missing required 'name' field")
                    return False

                if step_name in step_names:
                    logger.error(f"Duplicate step name: {step_name}")
                    return False

                step_names.add(step_name)

                # Validate dependencies exist
                depends_on = step.get("depends_on", [])
                for dep in depends_on:
                    if dep not in step_names and dep not in [
                        s.get("name") for s in steps
                    ]:
                        logger.error(
                            f"Step '{step_name}' depends on unknown step '{dep}'"
                        )
                        return False
            return True

        except Exception as e:
            logger.error(f"Step-based configuration validation failed: {str(e)}")
            return False


# Create a singleton instance of the configuration manager
config_manager = ConfigManager()
