"""
Registry Pattern Implementation

This module provides a generic registry pattern implementation that can be used
to register and retrieve different components in the system.
"""

from typing import Any, Callable, Dict, Type, TypeVar

from src.utils.decorator_utils import with_logger

T = TypeVar("T")


class Registry:
    """
    A generic registry for components.

    This class implements the registry pattern, allowing components to be
    registered by name and retrieved later. It's designed to be used as a base
    class for specific registries.
    """

    @with_logger
    def __init__(self):
        self._registry: Dict[str, Type[T]] = {}

    def register(self, name: str) -> Callable[[Type[T]], Type[T]]:
        """
        Register a component with the given name.

        Args:
            name: The name to register the component under

        Returns:
            A decorator function that registers the component
        """

        def decorator(cls: Type[T]) -> Type[T]:
            if name in self._registry:
                raise ValueError(f"Component '{name}' is already registered")
            else:
                self._registry[name] = cls
            return cls

        return decorator

    def get(self, name: str) -> Type[T]:
        """
        Get a component by name.

        Args:
            name: The name of the component to retrieve

        Returns:
            The registered component class

        Raises:
            ValueError: If the component is not registered
        """

        @with_logger
        def _get_component(name: str) -> Type[T]:
            if name not in self._registry:
                logger.error(f"Component '{name}' is not registered")
                raise ValueError(f"Component '{name}' is not registered")
            return self._registry[name]

        return _get_component(name)

    def create(self, name: str, **kwargs: Any) -> T:
        """
        Create an instance of a component by name.

        Args:
            name: The name of the component to create
            **kwargs: Arguments to pass to the component constructor

        Returns:
            An instance of the component

        Raises:
            ValueError: If the component is not registered
        """

        @with_logger
        def _create_instance(name: str, **kwargs: Any) -> T:
            component_cls = self.get(name)
            logger.debug(f"Creating instance of '{name}': {component_cls.__name__}")
            try:
                instance = component_cls(**kwargs)
                return instance
            except Exception as e:
                logger.error(
                    f"Error creating instance of '{name}': {str(e)}", exc_info=True
                )
                raise

        return _create_instance(name, **kwargs)

    def list_registered(self) -> list[str]:
        """
        List all registered component names.

        Returns:
            A list of registered component names
        """

        @with_logger
        def _list_components() -> list[str]:
            components = list(self._registry.keys())
            return components

        return _list_components()


# Create specific registries for different component types
class LLMRegistry(Registry):
    """Registry for Language Model components."""

    pass


class TaskRegistry(Registry):
    """Registry for Task components."""

    pass


class PromptoptimiserRegistry(Registry):
    """Registry for Prompt optimiser components."""

    pass


class EvaluatorRegistry(Registry):
    """Registry for Evaluator components."""

    pass


class EmbeddingsRegistry(Registry):
    """Registry for Embeddings Model components."""

    pass


# Singleton instances for each registry
@with_logger
def _create_registries():
    """Create singleton registry instances."""
    global \
        llm_registry, \
        task_registry, \
        prompt_optimiser_registry, \
        evaluator_registry, \
        embeddings_registry

    llm_registry = LLMRegistry()
    task_registry = TaskRegistry()
    prompt_optimiser_registry = PromptoptimiserRegistry()
    evaluator_registry = EvaluatorRegistry()
    embeddings_registry = EmbeddingsRegistry()


# Create the registries
_create_registries()
