"""
Dummy Language Model Implementation

This module provides a dummy implementation of the LLM interface for testing purposes.
"""

import time
from typing import Any, Dict, List

from src.core.registry import llm_registry
from src.llm.base import LLMInterface


@llm_registry.register("DummyLLM")
class DummyLLM(LLMInterface):
    """
    Dummy language model implementation.

    This class provides a dummy implementation of the LLM interface for testing purposes.
    It always returns a fixed response.
    """

    def __init__(
        self,
        version_name: str = "dummy",
        response: str = "No",
        delay: float = 0.0,
        **kwargs: Any
    ):
        """
        initialise the dummy language model.

        Args:
            version_name: The name of the model version to use
            response: The fixed response to return
            delay: The delay in seconds before returning the response
            **kwargs: Additional keyword arguments for the model
        """
        self.version_name = version_name
        self.response = response
        self.delay = delay
        self.kwargs = kwargs

    def generate(self, messages: List[Dict[str, Any]], **kwargs: Any) -> str:
        """
        Generate text based on the provided messages.

        Args:
            messages: A list of message dictionaries, each with 'role' and 'content' keys
            **kwargs: Additional keyword arguments for the generation process

        Returns:
            The fixed response
        """
        # Simulate processing delay
        if self.delay > 0:
            time.sleep(self.delay)

        return self.response

    @property
    def model_info(self) -> Dict[str, Any]:
        """
        Get information about the model.

        Returns:
            A dictionary containing information about the model
        """
        return {
            "name": "DummyLLM",
            "version_name": self.version_name,
            "response": self.response,
            "delay": self.delay,
            **self.kwargs,
        }
