"""
Base Prompt optimiser Interface

This module defines the base interface for prompt optimisers.
"""

from abc import ABC, abstractmethod
from typing import Any, Dict, List

from src.llm.base import LLMInterface
from src.tasks.base import TaskInterface
from src.utils.decorator_utils import with_logger


class PromptOptimiserInterface(ABC):
    """
    Interface for prompt optimisers.

    This abstract class defines the interface that all prompt optimiser
    implementations must adhere to.
    """

    @abstractmethod
    @with_logger
    def set_message_template(
        self,
        message_template_lst: List[Dict[str, str]],
    ) -> None:
        """
        Set the message template list for the prompt optimiser.

        Args:
            message_template_lst: The message template list to set
        """
        pass

    @abstractmethod
    @with_logger
    def optimise(
        self,
        task: TaskInterface,
        llm: LLMInterface,
        **kwargs: Any,
    ) -> None:
        """
        optimise a prompt using a feedback function.

        Args:
            base_prompt: The base prompt to optimise
            feedback_function: A function that takes a prompt and returns a score
            **kwargs: Additional keyword arguments for the optimisation process

        Returns:
            The optimised prompt
        """
        pass

    @abstractmethod
    @with_logger
    def apply(
        self,
        **kwargs: Any,
    ) -> List[Dict[str, Any]]:
        """
        Apply the optimised prompt transformation to a base prompt.

        Args:
            base_prompt: The base prompt to transform
            **kwargs: Additional keyword arguments for the transformation process

        Returns:
            The transformed prompt
        """
        pass
