"""
No Prompt optimiser Implementation

This module provides an empty implementation of the prompt optimiser interface. No optimisation
is performed and is intended for use as a placeholder.
"""

import copy
from typing import Any, Dict, List

from src.core.registry import prompt_optimiser_registry
from src.llm.base import LLMInterface
from src.prompt_optimisation.base import PromptOptimiserInterface
from src.tasks.base import TaskInterface
from src.utils.decorator_utils import with_logger


@prompt_optimiser_registry.register("none")
class NonePromptOptimiser(PromptOptimiserInterface):
    """
    None prompt optimiser implementation.

    This class provides an implementation of the prompt optimiser interface that
    does not perform any optimisation.
    """

    @with_logger
    def __init__(
        self,
        **kwargs: Any,
    ):
        """
        Initialise the PSAO prompt optimiser.

        Args:
            prompt: The prompt to use for optimisation
            **kwargs: Additional keyword arguments
        """
        self.msg_template_lst = []

    @with_logger
    def set_message_template(
        self,
        message_template_lst: List[Dict[str, Any]],
    ) -> None:
        """
        Initialise the optimiser from a message template.
        """
        self.msg_template_lst = copy.deepcopy(message_template_lst)

    @with_logger
    def optimise(
        self,
        task: TaskInterface,
        llm: LLMInterface,
        **kwargs: Any,
    ) -> List[Dict[str, Any]]:
        """
        Optimise a prompt using a feedback function.

        Args:
            base_prompt: The base prompt to optimise
            feedback_function: A function that takes a prompt and returns a score
            **kwargs: Additional keyword arguments for the optimisation process

        Returns:
            The optimised prompt
        """

        logger.debug("NonePromptOptimiser: Returning original prompt")

        # Log optimisation statistics
        return self.msg_template_lst

    @with_logger
    def apply(
        self,
        **kwargs: Any,
    ) -> List[Dict[str, Any]]:
        """
        Apply the optimised prompt transformation to a base prompt.

        Args:
            **kwargs: Additional keyword arguments for the transformation process

        Returns:
            The transformed prompt message list
        """
        return self.msg_template_lst
