"""
Command Line Interface Utilities

This module provides common CLI utilities for the prompt optimisation framework.
"""

import argparse
import logging
from typing import Callable, Optional

from src.utils.logging_config import (
    configure_logging_from_args,
    initialise_logging_system,
)


def default_parse_args(
    description: str = "Run a prompt optimisation experiment",
    default_config: str = "default",
    extra_args_fn: Optional[Callable[[argparse.ArgumentParser], None]] = None,
) -> argparse.Namespace:
    """
    Parse command-line arguments with standard options and optional extensions.

    Args:
        description: Description of the script for help text
        default_config: Default configuration name to use
        extra_args_fn: Optional function to add extra arguments to the parser

    Returns:
        Parsed command-line arguments
    """
    parser = argparse.ArgumentParser(description=description)

    # Add standard arguments
    parser.add_argument(
        "--config",
        type=str,
        default=default_config,
        help="Name of the experiment configuration file",
    )

    # Add data path argument
    parser.add_argument(
        "--data-path",
        type=str,
        help="Path to the data directory (overrides config settings)",
    )

    # Add logging-related arguments
    parser.add_argument(
        "--verbose",
        "-v",
        action="store_true",
        help="Enable verbose (DEBUG level) logging",
    )

    parser.add_argument(
        "--quiet",
        "-q",
        action="store_true",
        help="Enable quiet (WARNING level) logging",
    )

    parser.add_argument(
        "--log-file",
        type=str,
        help="Path to log file (if not specified, logs to console only)",
    )

    # Allow caller to add extra arguments
    if extra_args_fn:
        extra_args_fn(parser)

    args = parser.parse_args()

    return args


def configure_logging(args: argparse.Namespace) -> None:
    """
    Configure logging based on command-line arguments.

    This function now uses the centralized logging configuration system
    with dictConfig to eliminate handler duplication and PlaceHolder objects.

    Args:
        args: Parsed command-line arguments
    """
    # Configure global logging from CLI arguments
    config = configure_logging_from_args(args)

    # Initialise the centralized logging system
    # This will apply the dictConfig and set up the proper hierarchy
    initialise_logging_system()

    # If the configuration was updated with a log file, re-apply the config
    if config.log_file:
        config.apply_dict_config()

    # Log configuration messages
    # In quiet mode, these will only appear in the log file (if specified)
    # but not on the console due to console handler level being WARNING
    logger = logging.getLogger(__name__)

    # Only log these messages if not in quiet mode, or if there's a log file
    # In quiet mode without a log file, these messages would be lost anyway
    if not (hasattr(args, "quiet") and args.quiet and not config.log_file):
        logger.info("Centralised logging system configured")
        if config.log_file:
            logger.info(f"Logging to file: {config.log_file}")
        else:
            logger.info("Console logging enabled")

        # Log the arguments
        logger.info(f"Command-line arguments: {args}")
