"""
Decorator Utilities

This module provides decorators for common patterns used throughout the project.
"""

import functools
import logging
from pathlib import Path
from typing import Callable, Any, TypeVar, cast, Optional, Union

from src.utils.logging_config import (
    get_effective_config,
    LoggingConfig,
    ensure_logging_initialised,
)

F = TypeVar("F", bound=Callable[..., Any])


def _generate_logger_name(func: Callable, args: tuple) -> str:
    """
    Generate a context-aware logger name for a function.

    Args:
        func: The function being decorated
        args: Arguments passed to the function (to detect class context)

    Returns:
        Generated logger name
    """
    module_name = func.__module__

    # Check if this is a method (first arg is 'self' or 'cls')
    if args and hasattr(args[0], "__class__"):
        # This is likely a method call
        class_name = args[0].__class__.__name__
        method_name = func.__name__
        return f"{module_name}.{class_name}.{method_name}"

    # Check if this is a nested function by inspecting the qualname
    if "." in func.__qualname__:
        # This is likely a nested function, use a more specific logger name
        return f"{module_name}.{func.__qualname__}"
    else:
        # This is a top-level function, use the module name + function name
        return f"{module_name}.{func.__name__}"


def _get_configured_logger(name: str, config: LoggingConfig) -> logging.Logger:
    """
    Get or create a logger with the specified configuration using centralized approach.

    Args:
        name: Logger name
        config: Logging configuration

    Returns:
        Configured logger instance
    """
    # Ensure the centralized logging system is initialised
    ensure_logging_initialised()

    # Get the logger using the centralized approach
    logger = logging.getLogger(name)

    # Set the logger level based on config
    logger.setLevel(config.level)

    # Ensure propagation is enabled for centralized handling
    logger.propagate = True

    # Remove any existing handlers (centralized approach)
    for handler in logger.handlers[:]:
        logger.removeHandler(handler)

    return logger


def _cleanup_logger_injection(func: Callable, original_logger: Any) -> None:
    """
    Clean up logger injection from function globals.

    Args:
        func: The function to clean up
        original_logger: The original logger value (or None)
    """
    func_globals = func.__globals__

    if original_logger is not None:
        func_globals["logger"] = original_logger
    else:
        # Remove the logger from globals if it wasn't there before
        if "logger" in func_globals:
            del func_globals["logger"]


def with_logger(
    level: Optional[int] = None,
    log_file: Optional[Union[str, Path]] = None,
    logger_name: Optional[str] = None,
    console_output: Optional[bool] = None,
):
    """
    Enhanced decorator that ensures a logger is available for the function.

    This decorator creates a logger using configuration from the global logging
    config, with optional per-function overrides. It supports both direct decoration
    and parameterized decoration.

    Args:
        level: Override global log level for this function
        log_file: Override global log file for this function
        logger_name: Override auto-generated logger name
        console_output: Override console output setting

    Returns:
        The decorated function or decorator function

    Examples:
        # Simple decoration (uses global config)
        @with_logger
        def my_function():
            logger.info("Function called")

        # Parameterized decoration (with overrides)
        @with_logger(level=logging.DEBUG, log_file="debug.log")
        def debug_function():
            logger.debug("Debug message")

        # Custom logger name
        @with_logger(logger_name="custom.logger")
        def custom_function():
            logger.info("Custom logger")
    """

    def decorator(func: F) -> F:
        # Get the function's globals
        func_globals = func.__globals__

        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            # Get the effective configuration
            base_config = get_effective_config()

            # Create a copy and apply overrides
            config = base_config.copy()
            if level is not None:
                config.level = level
            if log_file is not None:
                config.log_file = (
                    Path(log_file) if isinstance(log_file, str) else log_file
                )
            if console_output is not None:
                config.console_output = console_output

            # Determine logger name
            if logger_name:
                name = logger_name
            else:
                name = _generate_logger_name(func, args)

            # Create or get the logger
            # First check if a logger was provided in kwargs
            if "logger" in kwargs:
                # Use the provided logger (for backward compatibility)
                logger_instance = kwargs.pop("logger")
            else:
                # Create a new logger with current config
                logger_instance = _get_configured_logger(name, config)

            # Store the original logger if it exists in globals
            original_logger = func_globals.get("logger", None)

            # Set the logger in the function's globals
            func_globals["logger"] = logger_instance

            try:
                # Call the original function
                return func(*args, **kwargs)
            finally:
                # Restore the original logger or remove it
                _cleanup_logger_injection(func, original_logger)

        return cast(F, wrapper)

    # Support both @with_logger and @with_logger() syntax
    if callable(level):
        # Direct decoration: @with_logger
        func = level
        level = None
        return decorator(func)
    else:
        # Parameterized decoration: @with_logger(...)
        return decorator


# Type annotation for static analysers
# This helps VS Code understand that 'logger' will be available in decorated functions
if False:  # This block is never executed, it's only for static analysis

    def with_logger(func: F) -> F:
        """Type stub for static analysis."""
        import logging

        logger: logging.Logger  # This tells type checkers that logger exists
        return func
