"""
Tests for the config module.
"""

import json
import tempfile
import pathlib
from unittest import mock

import pytest
import yaml
import jsonschema

from src.core.config import ConfigManager


class TestConfigManager:
    """Tests for the ConfigManager class."""

    def setup_method(self):
        """Set up test fixtures."""
        # Create temporary directories for config and schema
        self.temp_dir = tempfile.TemporaryDirectory()
        self.config_dir = pathlib.Path(self.temp_dir.name) / "config"
        self.schema_dir = self.config_dir / "schemas"

        # Create the directories
        self.config_dir.mkdir(parents=True)
        self.schema_dir.mkdir(parents=True)

        # Create a default config file
        self.default_config = {
            "project": {
                "name": "test_project",
                "version": "0.1.0",
            },
            "paths": {
                "data": "data",
                "output": "output",
            },
            "logging": {
                "level": "INFO",
                "format": "%(asctime)s - %(name)s - %(levelname)s - %(message)s",
            },
        }

        with open(self.config_dir / "default.yaml", "w") as f:
            yaml.dump(self.default_config, f)

        # Create a test schema
        self.test_schema = {
            "type": "object",
            "properties": {"test": {"type": "string"}},
            "required": ["test"],
        }

        with open(self.schema_dir / "test.json", "w") as f:
            json.dump(self.test_schema, f)

    def teardown_method(self):
        """Tear down test fixtures."""
        self.temp_dir.cleanup()

    def test_init(self):
        """Test initialisation of ConfigManager."""
        # Test with default parameters
        with mock.patch(
            "pathlib.Path.resolve", return_value=pathlib.Path("/fake/path")
        ), mock.patch("pathlib.Path.mkdir"), mock.patch(
            "builtins.open", mock.mock_open()
        ), mock.patch(
            "yaml.dump"
        ):
            config_manager = ConfigManager()
            assert config_manager.project_root == pathlib.Path("/fake/path")
            assert config_manager.config_dir == pathlib.Path("/fake/path") / "config"
            assert (
                config_manager.schema_dir
                == pathlib.Path("/fake/path") / "config" / "schemas"
            )
            assert config_manager.default_config == "default.yaml"

        # Test with custom parameters
        config_manager = ConfigManager(
            config_dir=self.config_dir,
            schema_dir=self.schema_dir,
            default_config="custom.yaml",
        )
        assert config_manager.config_dir == self.config_dir
        assert config_manager.schema_dir == self.schema_dir
        assert config_manager.default_config == "custom.yaml"

    def test_load_default_config(self):
        """Test loading the default configuration."""
        # Create a config manager with our test directories
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Check that the default config was loaded
        assert config_manager.config == self.default_config

    def test_load_config(self):
        """Test loading a configuration file."""
        # Create a test config file
        test_config = {
            "project": {
                "name": "override_project",
                "version": "0.1.0",
            },
            "extra": {
                "value": 42,
            },
        }

        # Create experiments directory
        experiments_dir = self.config_dir / "experiments"
        experiments_dir.mkdir(exist_ok=True)

        with open(experiments_dir / "test.yaml", "w") as f:
            yaml.dump(test_config, f)

        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Load the test config
        loaded_config = config_manager.load_config("test")

        # Check that the config was merged correctly
        assert loaded_config["project"]["name"] == "override_project"
        assert loaded_config["project"]["version"] == "0.1.0"
        assert loaded_config["extra"]["value"] == 42

        # Test loading a non-existent config
        with pytest.raises(FileNotFoundError):
            config_manager.load_config("nonexistent")

    def test_merge_configs(self):
        """Test merging configuration dictionaries."""
        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Define base and override configs
        base_config = {
            "a": 1,
            "b": {
                "c": 2,
                "d": 3,
            },
            "e": [4, 5],
        }

        override_config = {
            "a": 10,
            "b": {
                "c": 20,
                "f": 30,
            },
            "g": 40,
        }

        # Merge the configs
        merged = config_manager._merge_configs(base_config, override_config)

        # Check the merged result
        assert merged["a"] == 10
        assert merged["b"]["c"] == 20
        assert merged["b"]["d"] == 3
        assert merged["b"]["f"] == 30
        assert merged["e"] == [4, 5]
        assert merged["g"] == 40

    def test_validate_config(self):
        """Test validating a configuration against a schema."""
        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Valid config
        valid_config = {"test": "value"}
        config_manager.validate_config(valid_config, "test")

        # Invalid config
        invalid_config = {"not_test": "value"}
        with pytest.raises(jsonschema.exceptions.ValidationError):
            config_manager.validate_config(invalid_config, "test")

        # Non-existent schema
        with pytest.raises(FileNotFoundError):
            config_manager.validate_config(valid_config, "nonexistent")

    def test_get(self):
        """Test getting a configuration value by key."""
        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Test getting existing values
        assert config_manager.get("project.name") == "test_project"
        assert config_manager.get("paths.data") == "data"

        # Test getting non-existent values
        assert config_manager.get("nonexistent") is None
        assert config_manager.get("nonexistent", "default") == "default"
        assert config_manager.get("project.nonexistent") is None
        assert config_manager.get("project.nonexistent", "default") == "default"

    def test_set(self):
        """Test setting a configuration value by key."""
        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Test setting existing values
        config_manager.set("project.name", "new_name")
        assert config_manager.get("project.name") == "new_name"

        # Test setting new values
        config_manager.set("new.key", "value")
        assert config_manager.get("new.key") == "value"

        # Test setting nested values
        config_manager.set("nested.deeply.key", "nested_value")
        assert config_manager.get("nested.deeply.key") == "nested_value"

        # Test overriding a non-dict value with a dict
        config_manager.set("project.name.subkey", "subvalue")
        assert config_manager.get("project.name.subkey") == "subvalue"

    def test_save(self):
        """Test saving the configuration to a file."""
        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Modify the config
        config_manager.set("project.name", "saved_project")

        # Save to the default config
        config_manager.save()

        # Check that the file was saved
        with open(self.config_dir / "default.yaml", "r") as f:
            saved_config = yaml.safe_load(f)

        assert saved_config["project"]["name"] == "saved_project"

        # Save to a custom config
        config_manager.set("project.name", "custom_project")
        config_manager.save("custom")

        # Check that the file was saved
        with open(self.config_dir / "experiments" / "custom.yaml", "r") as f:
            custom_config = yaml.safe_load(f)

        assert custom_config["project"]["name"] == "custom_project"

    def test_apply_cli_overrides(self):
        """Test applying command-line argument overrides to the configuration."""
        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Create a test config
        test_config = {"paths": {"data": "original_data_path"}}

        # Create mock args
        import argparse

        mock_args = argparse.Namespace(data_path="/custom/data/path")

        # Apply CLI overrides with explicit args
        config_manager._apply_cli_overrides(test_config, mock_args)

        # Check that the data path was overridden
        assert test_config["paths"]["data"] == "/custom/data/path"

    def test_load_config_with_cli_override(self):
        """Test loading a configuration with command-line overrides."""
        # Create experiments directory and a test config file
        experiments_dir = self.config_dir / "experiments"
        experiments_dir.mkdir(exist_ok=True)

        test_config = {"paths": {"data": "config_data_path"}}

        with open(experiments_dir / "test_cli.yaml", "w") as f:
            yaml.dump(test_config, f)

        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Create mock args
        import argparse

        mock_args = argparse.Namespace(data_path="/cli/data/path")

        # Load the config with CLI overrides
        loaded_config = config_manager.load_config("test_cli", args=mock_args)

        # Check that the data path was overridden by the CLI argument
        assert loaded_config["paths"]["data"] == "/cli/data/path"

    def test_data_path_priority(self):
        """Test the priority order for data path resolution."""
        # Create experiments directory and a test config file
        experiments_dir = self.config_dir / "experiments"
        experiments_dir.mkdir(exist_ok=True)

        # Config with custom data path
        test_config = {"paths": {"data": "experiment_data_path"}}

        with open(experiments_dir / "test_priority.yaml", "w") as f:
            yaml.dump(test_config, f)

        # Create a config manager
        config_manager = ConfigManager(
            config_dir=self.config_dir, schema_dir=self.schema_dir
        )

        # Test 1: No CLI override - should use experiment config
        loaded_config = config_manager.load_config("test_priority")
        assert loaded_config["paths"]["data"] == "experiment_data_path"

        # Test 2: With CLI override - should use CLI value
        import argparse

        mock_args = argparse.Namespace(data_path="/cli/override/path")

        loaded_config = config_manager.load_config("test_priority", args=mock_args)

        assert loaded_config["paths"]["data"] == "/cli/override/path"
