"""
OpenAI Language Model Implementation

This module provides an implementation of the LLM interface for OpenAI models.
"""

import time
from typing import Any, Dict, List, Optional

from openai import OpenAI

from src.core.registry import llm_registry
from src.llm.base import LLMInterface
from src.utils.decorator_utils import with_logger
from src.utils.retry_utils import RetryConfig, create_retry_decorator


@llm_registry.register("OpenAI")
class OpenAILLM(LLMInterface):
    """
    OpenAI language model implementation.

    This class provides an implementation of the LLM interface for OpenAI models.
    """

    @with_logger
    def __init__(
        self,
        version_name: str,
        temperature: float = 0.0,
        top_p: float = 1.0,
        retry_config: Optional[RetryConfig] = None,
        **kwargs: Any,
    ):
        """
        Initialise the OpenAI language model.

        Args:
            version_name: The name of the model version to use
            temperature: The temperature to use for generation
            top_p: The top_p value to use for generation
            retry_config: Configuration for retry behavior
            **kwargs: Additional keyword arguments for the model
        """
        super().__init__(retry_config)
        logger.info(f"Initialising OpenAI LLM with model: {version_name}")

        self.version_name = version_name
        self.temperature = temperature
        self.top_p = top_p
        self.kwargs = kwargs

        # Initialise the OpenAI client
        base_url, api_key = self.load_api_info()
        self.configure_environment(base_url, api_key)
        logger.info(f"Using base URL: {base_url}")

        try:
            self.client = OpenAI(
                base_url=base_url,
                api_key=api_key,
            )
            logger.info("OpenAI client initialised successfully")
        except Exception as e:
            logger.error(f"Failed to initialise OpenAI client: {str(e)}", exc_info=True)
            raise

        # Set up retry decorator
        self.retry_decorator = create_retry_decorator(
            self.retry_config,
            logger=logger,
        )

    @with_logger
    def generate(self, messages: List[Dict[str, Any]], **kwargs: Any) -> str:
        """
        Generate text based on the provided messages.

        Args:
            messages: A list of message dictionaries, each with 'role' and 'content' keys
            **kwargs: Additional keyword arguments for the generation process

        Returns:
            The generated text
        """
        # Log the request using the base class method
        self.log_request(messages, **kwargs)

        # Merge default kwargs with provided kwargs
        generation_kwargs = {
            "temperature": self.temperature,
            "top_p": self.top_p,
        }
        generation_kwargs.update(kwargs)

        start_time = time.time()

        def _generate_with_retry():
            """Internal function for API call with retry logic."""
            stream = self.client.beta.chat.completions.parse(
                model=self.version_name,
                messages=messages,
                **generation_kwargs,
            )
            return stream.choices[0].message.content

        try:
            # Execute the API call with retry logic
            for attempt in self.retry_decorator:
                with attempt:
                    response = _generate_with_retry()

            # Log the response using the base class method
            duration = time.time() - start_time
            self.log_response(response, duration)

            return response

        except Exception as e:
            logger.error(
                f"Error during text generation after retries: {str(e)}", exc_info=True
            )
            raise

    @property
    def model_info(self) -> Dict[str, Any]:
        """
        Get information about the model.

        Returns:
            A dictionary containing information about the model
        """
        return {
            "name": "OpenAI",
            "version_name": self.version_name,
            "temperature": self.temperature,
            "top_p": self.top_p,
            **self.kwargs,
        }
