"""
Centralized Logging Configuration

This module provides a centralized configuration system for logging throughout the project.
It supports configuration from CLI arguments, environment variables, and programmatic updates.
"""

import logging
import logging.config
import os
from dataclasses import dataclass
from pathlib import Path
from typing import Optional, Dict, Any
import threading

# Thread-local storage for configuration
_thread_local = threading.local()

# Standard logging configuration using dictConfig
LOGGING_CONFIG = {
    "version": 1,
    "disable_existing_loggers": False,
    "formatters": {
        "standard": {
            "format": "%(asctime)s - %(name)s - %(levelname)s - %(message)s",
            "datefmt": "%Y-%m-%d %H:%M:%S",
        },
        "detailed": {
            "format": "%(asctime)s - %(name)s - %(levelname)s - %(module)s:%(lineno)d - %(message)s",
            "datefmt": "%Y-%m-%d %H:%M:%S",
        },
    },
    "filters": {
        "project_filter": {
            "()": "src.utils.logging_utils.ProjectLogFilter",
        }
    },
    "handlers": {
        "console": {
            "class": "logging.StreamHandler",
            "level": "INFO",
            "formatter": "standard",
            "stream": "ext://sys.stdout",
            "filters": ["project_filter"],
        }
    },
    "loggers": {
        "src": {"level": "INFO", "propagate": True, "handlers": []},
        "src.core": {"level": "NOTSET", "propagate": True, "handlers": []},
        "src.tasks": {"level": "NOTSET", "propagate": True, "handlers": []},
        "src.llm": {"level": "NOTSET", "propagate": True, "handlers": []},
        "src.embeddings": {"level": "NOTSET", "propagate": True, "handlers": []},
        "src.evaluation": {"level": "NOTSET", "propagate": True, "handlers": []},
        "src.prompt_optimisation": {
            "level": "NOTSET",
            "propagate": True,
            "handlers": [],
        },
        "src.utils": {"level": "NOTSET", "propagate": True, "handlers": []},
        "examples": {"level": "INFO", "propagate": True, "handlers": []},
        "__main__": {"level": "INFO", "propagate": True, "handlers": []},
    },
    "root": {"level": "INFO", "handlers": ["console"]},
}


@dataclass
class LoggingConfig:
    """
    Centralized logging configuration.

    This class holds all logging configuration parameters and provides methods
    to create configurations from various sources.
    """

    level: int = logging.INFO
    log_file: Optional[Path] = None
    console_output: bool = True
    format_string: str = "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
    date_format: str = "%Y-%m-%d %H:%M:%S"
    project_filter: bool = True
    experiment_name: Optional[str] = None
    experiment_output_dir: Optional[Path] = None

    # Additional configuration options
    propagate: bool = True
    file_level: Optional[int] = None  # Separate level for file handler
    console_level: Optional[int] = None  # Separate level for console handler

    @classmethod
    def from_args(cls, args) -> "LoggingConfig":
        """
        Create configuration from CLI arguments.

        Args:
            args: Parsed command-line arguments (argparse.Namespace)

        Returns:
            LoggingConfig instance
        """
        config = cls()

        # Set log level based on verbose/quiet flags
        if hasattr(args, "verbose") and args.verbose:
            config.level = logging.DEBUG
            config.console_level = logging.DEBUG
            config.file_level = logging.DEBUG
        elif hasattr(args, "quiet") and args.quiet:
            # In quiet mode, console shows only WARNING+, but file gets all INFO+ messages
            config.level = logging.INFO  # Keep general level at INFO for file logging
            config.console_level = logging.WARNING  # Console only shows WARNING+
            config.file_level = logging.INFO  # File gets all INFO+ messages

        # Set log file if specified
        if hasattr(args, "log_file") and args.log_file:
            config.log_file = Path(args.log_file)

        return config

    @classmethod
    def from_env(cls) -> "LoggingConfig":
        """
        Create configuration from environment variables.

        Supported environment variables:
        - LOGGING_LEVEL: Log level (DEBUG, INFO, WARNING, ERROR, CRITICAL)
        - LOGGING_FILE: Path to log file
        - LOGGING_FORMAT: Log format string
        - LOGGING_CONSOLE: Enable/disable console output (true/false)

        Returns:
            LoggingConfig instance
        """
        config = cls()

        # Set log level from environment
        level_str = os.environ.get("LOGGING_LEVEL", "").upper()
        if level_str:
            level_map = {
                "DEBUG": logging.DEBUG,
                "INFO": logging.INFO,
                "WARNING": logging.WARNING,
                "ERROR": logging.ERROR,
                "CRITICAL": logging.CRITICAL,
            }
            if level_str in level_map:
                config.level = level_map[level_str]

        # Set log file from environment
        log_file = os.environ.get("LOGGING_FILE")
        if log_file:
            config.log_file = Path(log_file)

        # Set format from environment
        log_format = os.environ.get("LOGGING_FORMAT")
        if log_format:
            config.format_string = log_format

        # Set console output from environment
        console_str = os.environ.get("LOGGING_CONSOLE", "").lower()
        if console_str in ("false", "0", "no", "off"):
            config.console_output = False

        return config

    @classmethod
    def from_dict(cls, config_dict: Dict[str, Any]) -> "LoggingConfig":
        """
        Create configuration from a dictionary.

        Args:
            config_dict: Dictionary containing configuration parameters

        Returns:
            LoggingConfig instance
        """
        config = cls()

        # Map string levels to logging constants
        level = config_dict.get("level", "INFO")
        if isinstance(level, str):
            level_map = {
                "DEBUG": logging.DEBUG,
                "INFO": logging.INFO,
                "WARNING": logging.WARNING,
                "ERROR": logging.ERROR,
                "CRITICAL": logging.CRITICAL,
            }
            config.level = level_map.get(level.upper(), logging.INFO)
        else:
            config.level = level

        # Set other parameters
        if "log_file" in config_dict:
            config.log_file = Path(config_dict["log_file"])

        config.console_output = config_dict.get("console_output", True)
        config.format_string = config_dict.get("format_string", config.format_string)
        config.date_format = config_dict.get("date_format", config.date_format)
        config.project_filter = config_dict.get("project_filter", True)

        return config

    def update_from_experiment(
        self, experiment_name: str, output_dir: Optional[Path] = None
    ):
        """
        Update configuration for experiment-specific logging.

        Args:
            experiment_name: Name of the experiment
            output_dir: Output directory for experiment logs
        """
        self.experiment_name = experiment_name
        if output_dir:
            self.experiment_output_dir = Path(output_dir)

    def get_effective_level(self, context: str = "default") -> int:
        """
        Get the effective log level for a given context.

        Args:
            context: Context for logging ('console', 'file', or 'default')

        Returns:
            Effective log level
        """
        if context == "console" and self.console_level is not None:
            return self.console_level
        elif context == "file" and self.file_level is not None:
            return self.file_level
        else:
            return self.level

    def copy(self) -> "LoggingConfig":
        """Create a copy of this configuration."""
        return LoggingConfig(
            level=self.level,
            log_file=self.log_file,
            console_output=self.console_output,
            format_string=self.format_string,
            date_format=self.date_format,
            project_filter=self.project_filter,
            experiment_name=self.experiment_name,
            experiment_output_dir=self.experiment_output_dir,
            propagate=self.propagate,
            file_level=self.file_level,
            console_level=self.console_level,
        )

    def to_dict_config(self) -> Dict[str, Any]:
        """
        Generate a dictConfig-compatible configuration dictionary.

        Returns:
            Dictionary suitable for logging.config.dictConfig()
        """
        # Start with the base configuration
        config = LOGGING_CONFIG.copy()

        # Update formatters based on current settings
        config["formatters"]["standard"]["format"] = self.format_string
        config["formatters"]["standard"]["datefmt"] = self.date_format
        config["formatters"]["detailed"]["format"] = self.format_string.replace(
            "%(message)s", "%(module)s:%(lineno)d - %(message)s"
        )
        config["formatters"]["detailed"]["datefmt"] = self.date_format

        # Update console handler level
        config["handlers"]["console"]["level"] = logging.getLevelName(
            self.get_effective_level("console")
        )

        # Add file handler if specified
        if self.log_file:
            config["handlers"]["file"] = {
                "class": "logging.FileHandler",
                "level": logging.getLevelName(self.get_effective_level("file")),
                "formatter": "detailed",
                "filename": str(self.log_file),
                "filters": ["project_filter"],
            }
            # Add file handler to root logger
            config["root"]["handlers"].append("file")

        # Update root logger level
        config["root"]["level"] = logging.getLevelName(self.level)

        # Update logger levels based on configuration
        for logger_name in config["loggers"]:
            if logger_name == "src":
                config["loggers"][logger_name]["level"] = logging.getLevelName(
                    self.level
                )
            # All other loggers inherit from their parents (NOTSET)

        # Remove console handler if console output is disabled
        if not self.console_output:
            config["root"]["handlers"] = [
                h for h in config["root"]["handlers"] if h != "console"
            ]
            if "console" in config["handlers"]:
                del config["handlers"]["console"]

        return config

    def apply_dict_config(self) -> None:
        """
        Apply this configuration using logging.config.dictConfig().

        This method initialises the entire logger hierarchy and eliminates
        PlaceHolder objects by ensuring all parent loggers are created.
        """
        config_dict = self.to_dict_config()

        # Apply the configuration
        logging.config.dictConfig(config_dict)

        # Ensure all logger hierarchy is properly initialised
        self._initialise_logger_hierarchy()

    def _initialise_logger_hierarchy(self) -> None:
        """
        Initialise the complete logger hierarchy to prevent PlaceHolder objects.

        This method explicitly creates all parent loggers in the hierarchy
        to ensure proper inheritance and eliminate PlaceHolder objects.
        """
        # Define the complete hierarchy
        hierarchy = [
            "src",
            "src.core",
            "src.core.experiment",
            "src.tasks",
            "src.llm",
            "src.embeddings",
            "src.evaluation",
            "src.prompt_optimisation",
            "src.utils",
            "examples",
            "__main__",
        ]

        # Create all loggers in the hierarchy
        for logger_name in hierarchy:
            logger = logging.getLogger(logger_name)
            # Ensure proper configuration
            if logger_name == "src":
                logger.setLevel(self.level)
            else:
                logger.setLevel(logging.NOTSET)  # Inherit from parent

            # Ensure propagation is enabled for proper hierarchy
            logger.propagate = True

            # Remove any existing handlers (centralized approach)
            for handler in logger.handlers[:]:
                logger.removeHandler(handler)


# Global configuration instance
_global_logging_config = LoggingConfig()
_config_lock = threading.Lock()

# Flag to track if logging has been initialised
_logging_initialised = False
_init_lock = threading.Lock()


def initialise_logging_system() -> None:
    """
    Initialise the centralized logging system.

    This function should be called once at application startup to set up
    the standardized logging architecture and eliminate PlaceHolder objects.
    """
    global _logging_initialised

    with _init_lock:
        if _logging_initialised:
            return

        # Get the current global configuration
        config = get_global_logging_config()

        # Apply the dictConfig to initialise the hierarchy
        config.apply_dict_config()

        _logging_initialised = True


def ensure_logging_initialised() -> None:
    """
    Ensure the logging system is initialised.

    This is a safe method that can be called multiple times.
    """
    if not _logging_initialised:
        initialise_logging_system()


def get_global_logging_config() -> LoggingConfig:
    """
    Get the global logging configuration.

    Returns:
        Current global logging configuration
    """
    with _config_lock:
        return _global_logging_config.copy()


def set_global_logging_config(config: LoggingConfig) -> None:
    """
    Set the global logging configuration.

    Args:
        config: New logging configuration
    """
    global _global_logging_config
    with _config_lock:
        _global_logging_config = config.copy()


def update_global_logging_config(**kwargs) -> None:
    """
    Update specific parameters in the global logging configuration.

    Args:
        **kwargs: Configuration parameters to update
    """
    global _global_logging_config
    with _config_lock:
        for key, value in kwargs.items():
            if hasattr(_global_logging_config, key):
                setattr(_global_logging_config, key, value)


def configure_logging_from_args(args) -> LoggingConfig:
    """
    Configure global logging from CLI arguments.

    Args:
        args: Parsed command-line arguments

    Returns:
        The configured LoggingConfig
    """
    config = LoggingConfig.from_args(args)
    set_global_logging_config(config)
    return config


def configure_logging_from_env() -> LoggingConfig:
    """
    Configure global logging from environment variables.

    Returns:
        The configured LoggingConfig
    """
    config = LoggingConfig.from_env()
    set_global_logging_config(config)
    return config


def configure_logging_from_dict(config_dict: Dict[str, Any]) -> LoggingConfig:
    """
    Configure global logging from a dictionary.

    Args:
        config_dict: Dictionary containing configuration parameters

    Returns:
        The configured LoggingConfig
    """
    config = LoggingConfig.from_dict(config_dict)
    set_global_logging_config(config)
    return config


def get_thread_local_config() -> Optional[LoggingConfig]:
    """
    Get thread-local logging configuration if it exists.

    Returns:
        Thread-local configuration or None
    """
    return getattr(_thread_local, "config", None)


def set_thread_local_config(config: Optional[LoggingConfig]) -> None:
    """
    Set thread-local logging configuration.

    Args:
        config: Configuration to set, or None to clear
    """
    _thread_local.config = config


def get_effective_config() -> LoggingConfig:
    """
    Get the effective logging configuration.

    Returns thread-local config if available, otherwise global config.

    Returns:
        Effective logging configuration
    """
    thread_config = get_thread_local_config()
    if thread_config is not None:
        return thread_config
    return get_global_logging_config()
