import numpy as np


def calculate_2dft(inp):
    ft = np.fft.fft2(inp)
    return np.fft.fftshift(ft)


def calculate_2dift(inp, mode="real"):
    ift = np.fft.ifftshift(inp)
    ift = np.fft.ifft2(ift)

    if mode == "real":
        return ift.real.astype(np.float32)
    elif mode == "abs":
        return np.abs(ift).astype(np.float32)
    else:
        return ift
    

def calculate_distance_from_centre(coords, centre):
    # Distance from centre is sqrt(x^2 + y^2)
    return np.sqrt(
        (coords[0] - centre) ** 2 + (coords[1] - centre) ** 2
    )


def find_symmetric_coordinates(coords, centre):
    return (centre + (centre - coords[0]),
            centre + (centre - coords[1]))


def get_coords_left_half(h, centre):
    assert h % 2 != 0

    coords_left_half = (
        (i_row, j_col) for i_row in range(h) for j_col in range(centre+1)
    )

    coords_centre_down = (
        (i_row, centre) for i_row in range(centre+1, h)
    )

    return list(set(coords_left_half) - set(coords_centre_down))


def get_lowpass_filter(h, threshold=0.25):
    assert h % 2 != 0

    centre = int((h - 1) / 2)

    coords_left_half = get_coords_left_half(h, centre)
    coords_left_half = sorted(coords_left_half,
                              key=lambda x: calculate_distance_from_centre(x, centre))

    harmonics_N = len(coords_left_half)
    harmonics_limit = int(np.ceil(harmonics_N * threshold))

    lowpass_filter = np.zeros((h, h), dtype=np.float32)
    for cc in coords_left_half[:harmonics_limit]:
        symm_cc = find_symmetric_coordinates(cc, centre)
        lowpass_filter[cc] = 1.0
        lowpass_filter[symm_cc] = 1.0

    diameter = int(np.sum(lowpass_filter[:, centre]))
    assert diameter % 2 != 0

    return lowpass_filter.astype(bool), centre, int((diameter - 1) / 2)
