import json
import base64
import os

def load_templates(path):
    with open(path, 'r') as file:
        return json.load(file)

def read_netlists(path):
    netlists = {}
    with open(path, 'r') as file:
        content = file.read().split('TEMPLATE_ID ')
        for entry in content[1:]:  # Skip the first split as it's empty
            parts = entry.strip().split('---')
            template_id = parts[0].strip()
            netlist = parts[1].strip() if len(parts) > 1 else None
            netlists[template_id] = netlist
    return netlists

def encode_image(path):
    with open(path, 'rb') as image_file:
        return base64.b64encode(image_file.read()).decode('utf-8')

def process_templates(templates, netlists, diagrams_path):
    for template in templates:
        template_id = str(template['template_id'])  # Ensure the ID is treated as a string
        template['netlist'] = netlists.get(template_id, None)
        if 'images' in template:
            for image_name in list(template['images'].keys()):
                image_path = os.path.join(diagrams_path, image_name + '.png')
                if os.path.exists(image_path):
                    template['images'][image_name] = encode_image(image_path)
                else:
                    template['images'][image_name] = None
    return templates

def save_templates(templates, output_path):
    with open(output_path, 'w') as file:
        json.dump(templates, file, indent=4)

# Paths to the files and directories
templates_path = 'templates.json'
netlists_path = 'netlists.txt'
diagrams_path = '../diagrams'  # Adjusted relative path
output_path = 'templates_complete.json'

# Main processing steps
templates = load_templates(templates_path)
netlists = read_netlists(netlists_path)
processed_templates = process_templates(templates, netlists, diagrams_path)
save_templates(processed_templates, output_path)

print("Templates have been processed and saved to", output_path)
