import json
import re

# Load the JSON file
with open('data.json', 'r') as file:
    data = json.load(file)

# HTML template
html_template = """
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Data Display</title>
    <style>
        body {{
            font-family: Arial, sans-serif;
            margin: 20px;
        }}
        .data-point {{
            margin-bottom: 40px;
        }}
        .key {{
            font-weight: bold;
        }}
        img {{
            height: 400px;  /* Set a uniform height for all images */
            width: auto;
        }}
        pre {{
            background: #f4f4f4;
            padding: 10px;
            border-radius: 5px;
            white-space: pre-wrap;  /* Ensure that preformatted text wraps properly */
        }}
        code {{
            background: #f4f4f4;
            padding: 2px 4px;
            border-radius: 3px;
            line-height: 1.5;
        }}
        .horizontal-line {{
            border-top: 1px solid #ccc;
            margin: 20px 0;
        }}
    </style>
    <link rel="stylesheet" href="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.2.0/styles/default.min.css">
    <script src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.2.0/highlight.min.js"></script>
    <script>hljs.highlightAll();</script>
    <script type="text/x-mathjax-config">
        MathJax.Hub.Config({{
            tex2jax: {{
                inlineMath: [['$', '$'], ['\\\\(', '\\\\)']],
                displayMath: [['$$', '$$'], ['\\\\[', '\\\\]']],
                processEscapes: true
            }}
        }});
    </script>
    <script type="text/javascript" async
        src="https://cdnjs.cloudflare.com/ajax/libs/mathjax/2.7.7/MathJax.js?config=TeX-MML-AM_CHTML">
    </script>
</head>
<body>
    {content}
</body>
</html>
"""

# Function to replace special characters with HTML entities
def replace_special_chars(text):
    replacements = {
        '\\omega': '&#969;',
        '\\alpha': '&#945;',
        '\\pi': '&#960;',
        '\\langle': '&lang;',
        '\\rangle': '&rang;',
        '\\mu': '&#956;',
    }
    for key, value in replacements.items():
        text = text.replace(key, value)
    return text

# Function to process code blocks within a string
def process_code_blocks(text):
    # Handle ```python ... ``` and ``` ... ```
    code_block_pattern = re.compile(r'```(?:python)?(.*?)```', re.DOTALL)
    text = code_block_pattern.sub(r'<pre><code>\1</code></pre>', text)
    # Replace special characters within code blocks
    text = re.sub(r'<pre><code>(.*?)</code></pre>', lambda m: f'<pre><code>{replace_special_chars(m.group(1))}</code></pre>', text, flags=re.DOTALL)
    return text

# Function to process netlist
def process_netlist(netlist):
    if netlist is not None:
        return process_code_blocks(netlist)

# Function to convert newlines to <br>
def convert_newlines(text):
    if text:
        return text.replace('\r\n', '<br>').replace('\n', '<br>')
    return text

# Function to process each data point
def process_data(data):
    content = ""
    for entry in data:
        content += '<div class="data-point">'
        content += f'<div><span class="key">Template ID:</span> {entry["template_id"]}</div>'
        content += f'<div><span class="key">ID:</span> {entry["id"]}</div>'
        content += f'<div><span class="key">Category:</span> {entry["category"]}</div>'
        content += f'<div><span class="key">Level:</span> {entry["level"]}</div>'
        content += f'<div><span class="key">Source:</span> {entry["source"]}</div>'

        # Display images
        content += '<div><span class="key">Images:</span></div>'
        for key, img_data in entry["images"].items():
            content += f'<div><img src="data:image/png;base64,{img_data}" alt="{key}"></div>'

        # Process and display netlist
        netlist_processed = process_netlist(entry["netlist"])
        content += f'<div><span class="key">Netlist:</span> {netlist_processed}</div>'

        # Convert newlines in problem and answer
        problem_converted = convert_newlines(entry["problem"])
        answer_converted = convert_newlines(entry["answer"])

        content += f'<div><span class="key">Problem:</span> {problem_converted}</div>'

        # Process full solution for code blocks and convert newlines
        full_solution_processed = convert_newlines(process_code_blocks(entry["full_solution"]))
        content += f'<div><span class="key">Full Solution:</span> {full_solution_processed}</div>'

        content += f'<div><span class="key">Answer:</span> {answer_converted}</div>'
        content += '</div><div class="horizontal-line"></div>'
    return content

# Generate the HTML content
html_content = process_data(data)
full_html = html_template.format(content=html_content)

# Save the HTML content to a file with UTF-8 encoding
with open('view-data.html', 'w', encoding='utf-8') as file:
    file.write(full_html)

print("HTML file has been created successfully.")
