# Copyright 2024 The Brax Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# pylint:disable=g-multiple-import
"""Trains a reacher to reach a target.

Based on the OpenAI Gym MuJoCo Reacher environment.
"""

from typing import Tuple

from brax import base
from brax import math
from brax.envs.base import PipelineEnv, State
from brax.io import mjcf
from etils import epath
import jax
from jax import numpy as jp
import jax.random as jrnd
import jax.numpy as jnp
import flax.core as fcore
import flax.struct as fstruct
from utils.jax_util import GenericEnvState
from brax.base import State as PipelineState
from os import path
from typing import Literal
from dataclasses import dataclass


@dataclass
class Cfg:
    variant: Literal["normal", "quick"]


@fstruct.dataclass
class Sys:
    # dynamic system state that changes after every `env.step`
    pipeline_state: PipelineState

    # static system state that depicts fixed paramemters
    static_system: mjcf.System
    target_q: jnp.ndarray


EnvState = GenericEnvState[Sys]


class Reacher(PipelineEnv):

    def __init__(
        self,
        cfg: Cfg,
    ):
        sys = mjcf.load(path.dirname(__file__) + "/assets/reacher.xml")

        n_frames = 2

        # kwargs["n_frames"] = kwargs.get("n_frames", n_frames)

        super().__init__(sys=sys, backend="mjx", n_frames=n_frames, debug=False)

        q = self.sys.init_q
        qd = jnp.zeros((self.sys.qd_size(),))

        self.variant: Literal["normal", "quick"] = cfg.variant
        assert self.variant in ["normal", "quick"]
        self.example_pipeline_state = self.pipeline_init(q, qd)
        self.n_obs = self._get_obs(self.example_pipeline_state).shape[0]
        self.n_act = self.sys.act_size()

        print(f"in reacher: system geom_size is: {self.sys.geom_size}")

    def gen_tree(self):
        in_axes = jax.tree_map(lambda _: jnp.array(jnp.nan), self.sys)
        in_axes = in_axes.tree_replace({"geom_size": jnp.array(0)})

        return EnvState(
            sys=Sys(
                pipeline_state=jax.tree_map(
                    lambda _: jnp.array(0), self.example_pipeline_state
                ),
                static_system=in_axes,
                target_q=jnp.array(0),
            ),
            obs=jnp.array(0),
            acts=jnp.array(0),
            rwds=jnp.array(0),
            terms=jnp.array(0),
            truncs=jnp.array(0),
            infos=fcore.freeze(dict()),
        )

    def make(self, key: jax.Array, rang: jax.Array):
        low, high = rang[0], rang[1]

        key, radius_key = jrnd.split(key)
        radius_ratio = jrnd.uniform(
            radius_key,
            self.sys.geom_size[[6, 7], :1].shape,
            minval=low,
            maxval=high,
        )
        # _old_length = self.sys.geom_size[[6], 1:2]
        key, target = self._random_target(key)

        _new_sys = self.sys.replace(
            geom_size=self.sys.geom_size.at[[6, 7], :1].multiply(radius_ratio),
        )

        env_state = EnvState(
            sys=Sys(
                pipeline_state=self.example_pipeline_state,
                static_system=_new_sys,
                target_q=target,
            ),
            obs=self._get_obs(self.example_pipeline_state),
            acts=jnp.zeros((self.n_act,)),
            rwds=jnp.zeros(()),
            terms=jnp.array(0),
            truncs=jnp.array(0),
            infos=fcore.freeze(dict()),
        )
        return self.reset(key, env_state)

    def reset(self, keys: jax.Array, env_state: EnvState):
        static_sys = env_state.sys.static_system
        key, key1, key2 = jax.random.split(keys, 3)

        q = static_sys.init_q + jax.random.uniform(
            key1, (self.sys.q_size(),), minval=-0.1, maxval=0.1
        )
        qd = jax.random.uniform(
            key2, (self.sys.qd_size(),), minval=-0.005, maxval=0.005
        )

        # set the target q, qd
        q = q.at[2:].set(env_state.sys.target_q)
        qd = qd.at[2:].set(0)

        pipeline_state = self.pipeline_init(q, qd)

        done = jnp.array(0)
        return EnvState(
            sys=Sys(
                pipeline_state=pipeline_state,
                static_system=static_sys,
                target_q=env_state.sys.target_q,
            ),
            obs=self._get_obs(pipeline_state),
            acts=jnp.zeros((self.n_act,)),
            rwds=jnp.zeros(()),
            terms=done,
            truncs=jnp.zeros_like(done),
            infos=env_state.infos,
        )

    def step(self, state: EnvState, action: jax.Array) -> State:
        pipeline_state = self._my_pipeline_step(state, action)
        obs = self._get_obs(pipeline_state)

        # vector from tip to target is last 3 entries of obs vector
        if self.variant == "normal":
            reward_dist = -math.safe_norm(obs[-3:])
            reward_ctrl = -jp.square(action).sum()
            reward = reward_dist + reward_ctrl
            done = jnp.array(0)
        else:
            reward = jnp.array(-1.0)
            dist = math.safe_norm(obs[-3:])
            done = jnp.where(dist < 0.02, jnp.array(1), jnp.array(0))

        # state.metrics.update(
        #     reward_dist=reward_dist,
        #     reward_ctrl=reward_ctrl,
        # )

        # return state.replace(pipeline_state=pipeline_state, obs=obs, reward=reward)
        return state.replace(
            sys=state.sys.replace(pipeline_state=pipeline_state),
            obs=obs,
            acts=action,
            rwds=reward,
            terms=done,
        )

    def _get_obs(self, pipeline_state: base.State) -> jax.Array:
        """Returns egocentric observation of target and arm body."""
        theta = pipeline_state.q[:2]
        target_pos = pipeline_state.x.pos[2]
        tip_pos = (
            pipeline_state.x.take(1)
            .do(base.Transform.create(pos=jp.array([0.11, 0, 0])))
            .pos
        )
        # tip_vel, instead of pipeline_state.qd[:2], leads to more sensible policies
        # for a randomly initialized policy network
        tip_vel = (
            base.Transform.create(pos=jp.array([0.11, 0, 0]))
            .do(pipeline_state.xd.take(1))
            .vel
        )
        tip_to_target = tip_pos - target_pos

        return jp.concatenate(
            [
                jp.cos(theta),
                jp.sin(theta),
                pipeline_state.q[2:],  # target x, y
                tip_vel[:2],
                tip_to_target,
            ]
        )

    def _random_target(self, key: jax.Array) -> Tuple[jax.Array, jax.Array]:
        """Returns a target location in a random circle slightly above xy plane."""
        key, key1, key2 = jax.random.split(key, 3)
        dist = 0.15 * jax.random.uniform(key1)
        ang = jp.pi * 2.0 * jax.random.uniform(key2)
        target_x = dist * jp.cos(ang)
        target_y = dist * jp.sin(ang)
        return key, jp.array([target_x, target_y])

    def _my_pipeline_step(self, env_state: EnvState, action: jax.Array) -> base.State:
        """Takes a physics step using the physics pipeline."""

        def f(carry, _):
            pipe_state, sys = carry
            return (
                (
                    self._pipeline.step(sys, pipe_state, action, self._debug),
                    sys,
                ),
                None,
            )

        carry, _ = jax.lax.scan(
            f,
            (
                env_state.sys.pipeline_state,
                env_state.sys.static_system,
            ),
            (),
            self._n_frames,
        )

        return carry[0]

    @property
    def state_dim(self):
        return self.n_obs

    @property
    def act_dim(self):
        return self.n_act


def refill_cfg(cfg):
    from omegaconf.omegaconf import OmegaConf

    dcfg = {}

    dcfg["obs_is_concrete"] = False
    dcfg["obs_dim"] = 11
    dcfg["act_is_concrete"] = False
    dcfg["act_dim"] = 1

    if cfg.env.kind == "quick":
        dcfg["rwd_is_concrete"] = True
        dcfg["rwd_nums"] = 1
    else:
        dcfg["rwd_is_concrete"] = False
        dcfg["rwd_dim"] = 1

    return cfg, dcfg
