# Copyright 2024 The Brax Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# pylint:disable=g-multiple-import
"""An inverted pendulum environment."""

from brax import base
from brax.envs.base import PipelineEnv, State
from brax.io import mjcf
from etils import epath
import jax
from jax import numpy as jp
import jax.numpy as jnp
from brax.base import State as PipelineState
import flax.struct as fstruct
from utils.jax_util import GenericEnvState
from os import path
import jax.random as jrnd
import flax.core as fcore
from typing import Literal


@fstruct.dataclass
class Sys:
    # dynamic system state that changes after every `env.step`
    pipeline_state: PipelineState

    # static system state that depicts fixed paramemters
    static_system: mjcf.System


EnvState = GenericEnvState[Sys]


class InvertedPendulum(PipelineEnv):

    def __init__(self):
        sys = mjcf.load(path.dirname(__file__) + "/assets/inverted_pendulum.xml")

        n_frames = 2
        self._debug = False

        super().__init__(sys=sys, backend="mjx", n_frames=n_frames, debug=self._debug)

        q = self.sys.init_q
        qd = jnp.zeros((self.sys.qd_size(),))

        self.example_pipeline_state = self.pipeline_init(q, qd)
        self.n_obs = self._get_obs(self.example_pipeline_state).shape[0]
        self.n_act = self.sys.act_size()

        print(f"in cartpole: system geom_size is: {self.sys.geom_size}")

    def gen_tree(self):
        in_axes = jax.tree_map(lambda _: jnp.array(jnp.nan), self.sys)
        in_axes = in_axes.tree_replace({"geom_size": jnp.array(0)})

        return EnvState(
            sys=Sys(
                pipeline_state=jax.tree_map(
                    lambda _: jnp.array(0), self.example_pipeline_state
                ),
                static_system=in_axes,
            ),
            obs=jnp.array(0),
            acts=jnp.array(0),
            rwds=jnp.array(0),
            terms=jnp.array(0),
            truncs=jnp.array(0),
            infos=fcore.freeze(dict()),
        )

    def make(self, key: jax.Array, rang: jax.Array):
        low, high = rang[0], rang[1]

        key, radius_key = jrnd.split(key)
        radius_ratio = jrnd.uniform(
            radius_key,
            self.sys.geom_size[[2], :1].shape,
            minval=low,
            maxval=high,
        )

        _new_sys = self.sys.replace(
            geom_size=self.sys.geom_size.at[[2], :1].multiply(radius_ratio)
        )

        env_state = EnvState(
            sys=Sys(
                pipeline_state=self.example_pipeline_state,
                static_system=_new_sys,
            ),
            obs=self._get_obs(self.example_pipeline_state),
            acts=jnp.zeros((self.n_act,)),
            rwds=jnp.zeros(()),
            terms=jnp.array(0),
            truncs=jnp.array(0),
            infos=fcore.freeze(dict()),
        )
        return self.reset(key, env_state)

    def reset(self, keys: jax.Array, env_state: EnvState):
        """Resets the environment to an initial state."""

        _, key1, key2 = jax.random.split(keys, 3)
        static_sys = env_state.sys.static_system

        q = static_sys.init_q + jax.random.uniform(
            key1, (self.sys.q_size(),), minval=-0.01, maxval=0.01
        )
        qd = jax.random.uniform(key2, (self.sys.qd_size(),), minval=-0.01, maxval=0.01)

        pipeline_state = self.pipeline_init(q, qd)
        done = jnp.array(0)
        return EnvState(
            sys=Sys(
                pipeline_state=pipeline_state,
                static_system=static_sys,
            ),
            obs=self._get_obs(pipeline_state),
            acts=jnp.zeros((self.n_act,)),
            rwds=jnp.zeros(()),
            terms=done,
            truncs=jnp.zeros_like(done),
            infos=env_state.infos,
        )

    def step(self, state: EnvState, action: jax.Array) -> State:
        """Run one timestep of the environment's dynamics."""
        pipeline_state = self._my_pipeline_step(state, action * 3)
        obs = self._get_obs(pipeline_state)
        reward = jnp.ones(())  # (1.0)
        done = jp.where(jp.abs(obs[1]) > 0.2, 1, 0)
        return state.replace(
            sys=state.sys.replace(pipeline_state=pipeline_state),
            obs=obs,
            acts=action,
            rwds=reward,
            terms=done,
        )

    def _my_pipeline_step(self, env_state: EnvState, action: jax.Array) -> base.State:
        """Takes a physics step using the physics pipeline."""

        def f(carry, _):
            pipe_state, sys = carry
            return (
                (
                    self._pipeline.step(sys, pipe_state, action, self._debug),
                    sys,
                ),
                None,
            )

        carry, _ = jax.lax.scan(
            f,
            (
                env_state.sys.pipeline_state,
                env_state.sys.static_system,
            ),
            (),
            self._n_frames,
        )

        return carry[0]

    def _get_obs(self, pipeline_state: base.State) -> jax.Array:
        """Observe cartpole body position and velocities."""
        return jp.concatenate([pipeline_state.q, pipeline_state.qd])

    @property
    def state_dim(self):
        return self.n_obs

    @property
    def act_dim(self):
        return self.n_act


def refill_cfg(cfg):
    from omegaconf.omegaconf import OmegaConf

    dcfg = {}

    dcfg["obs_is_concrete"] = False
    dcfg["obs_dim"] = 4
    dcfg["act_is_concrete"] = False
    dcfg["act_dim"] = 1
    dcfg["rwd_is_concrete"] = False
    dcfg['rwd_dim'] = 1

    return cfg, dcfg
