from pathlib import Path
from collections import OrderedDict

import torch
import torch.nn as nn
import torch.nn.init as init
import torch.nn.functional as F
import torchvision.models as tvm
from torch import nn

from models.base_model import SequentialModel


class ResNet(SequentialModel):
    """ResNet tuned to work on ImageNet and CIFAR10."""

    def __init__(
        self,
        width: int,
        dims: list = [[3, 224, 224], 1000],
        pretrained: bool = True,
        path_to_cifar_weights=Path("/local/home/author/run-data/resnet-cifar10"),
        **kwargs,
    ):
        super(ResNet, self).__init__(**kwargs)

        self.dims = dims

        if self.dims == [[3, 224, 224], 1000]:
            # ImageNet case
            if width == 18:
                weights = None
                if pretrained:
                    weights = tvm.ResNet18_Weights.IMAGENET1K_V1
                model = tvm.resnet18(weights=weights)
            elif width == 34:
                weights = None
                if pretrained:
                    weights = tvm.ResNet34_Weights.IMAGENET1K_V1
                model = tvm.resnet34(weights=weights)
            elif width == 50:
                weights = None
                if pretrained:
                    weights = tvm.ResNet50_Weights.IMAGENET1K_V1
                model = tvm.resnet50(weights=weights)
            elif width == 101:
                weights = None
                if pretrained:
                    weights = tvm.ResNet101_Weights.IMAGENET1K_V1
                model = tvm.resnet101(weights=weights)
            elif width == 152:
                weights = None
                if pretrained:
                    weights = tvm.ResNet152_Weights.IMAGENET1K_V1
                model = tvm.resnet152(weights=weights)
            else:
                raise NameError("This ResNet width is not supported.")

        elif self.dims == [[3, 32, 32], 10]:
            # CIFAR-10 case
            if width == 20:
                model = CIFAR10_ResNet(BasicBlock, [3, 3, 3])
                if pretrained:
                    model = load_cifar_weights(model, path_to_cifar_weights, "resnet20-12fca82f.th")
            elif width == 32:
                model = CIFAR10_ResNet(BasicBlock, [5, 5, 5])
                if pretrained:
                    model = load_cifar_weights(model, path_to_cifar_weights, "resnet32-d509ac18.th")
            elif width == 44:
                model = CIFAR10_ResNet(BasicBlock, [7, 7, 7])
                if pretrained:
                    model = load_cifar_weights(model, path_to_cifar_weights, "resnet44-014dd654.th")
            elif width == 56:
                model = CIFAR10_ResNet(BasicBlock, [9, 9, 9])
                if pretrained:
                    model = load_cifar_weights(model, path_to_cifar_weights, "resnet56-4bfd9763.th")
            elif width == 110:
                model = CIFAR10_ResNet(BasicBlock, [18, 18, 18])
                if pretrained:
                    model = load_cifar_weights(
                        model, path_to_cifar_weights, "resnet110-1d1ed7c2.th"
                    )
            elif width == 1202:
                model = CIFAR10_ResNet(BasicBlock, [200, 200, 200])
                if pretrained:
                    model = load_cifar_weights(
                        model, path_to_cifar_weights, "resnet1202-f3b1deed.th"
                    )
            else:
                raise NameError("This ResNet width is not supported.")

        else:
            raise ValueError(f"No ResNet implementation found for dimensions {dims}")

        layers = list(model._modules.values())
        # insert the flatten layer before FC so that the sequential wrapper works
        layers.insert(-1, nn.Flatten())
        self.layers = nn.Sequential(*layers)
        self.layer_input_shapes = self.get_layer_input_shapes()


def load_cifar_weights(model, path, weights_name):
    weights_ = torch.load(
        path / weights_name,
        map_location=torch.device("cpu"),
    )["state_dict"]
    # remove "module." prefix
    weights = {}
    for k, v in weights_.items():
        k = ".".join(k.split(".")[1:])
        weights[k] = v

    weigths = OrderedDict(weights)
    model.load_state_dict(weights)

    return model


# Code below is based on the implementation of Yerlan Idelbayev
# https://github.com/akamaster/pytorch_resnet_cifar10


def _weights_init(m):
    classname = m.__class__.__name__
    # print(classname)
    if isinstance(m, nn.Linear) or isinstance(m, nn.Conv2d):
        init.kaiming_normal_(m.weight)


class LambdaLayer(nn.Module):
    def __init__(self, lambd):
        super(LambdaLayer, self).__init__()
        self.lambd = lambd

    def forward(self, x):
        return self.lambd(x)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, option="A"):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False
        )
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            if option == "A":
                """
                For CIFAR10 ResNet paper uses option A.
                """
                self.shortcut = LambdaLayer(
                    lambda x: F.pad(
                        x[:, :, ::2, ::2], (0, 0, 0, 0, planes // 4, planes // 4), "constant", 0
                    )
                )
            elif option == "B":
                self.shortcut = nn.Sequential(
                    nn.Conv2d(
                        in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False
                    ),
                    nn.BatchNorm2d(self.expansion * planes),
                )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class CIFAR10_ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(CIFAR10_ResNet, self).__init__()
        self.in_planes = 16

        self.conv1 = nn.Conv2d(3, 16, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(16)
        self.relu = nn.ReLU()
        self.layer1 = self._make_layer(block, 16, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 32, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 64, num_blocks[2], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d(output_size=(1, 1))
        self.linear = nn.Linear(64, num_classes)

        self.apply(_weights_init)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.avgpool(out)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out
