import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.utils.checkpoint as checkpoint
import numpy as np
from timm.models.layers import DropPath, to_2tuple, trunc_normal_

from .SA_layer import *
from .CA_layer import *

# Attention:
class CABlock(torch.nn.Module):
    def __init__(self, embed_dim, next_embed_dim, kd, nh, ar, mr, mlp_activation, attention_activation, resolution, drop_path, input_dynamic=False, output_dynamic=False):
        super().__init__()
        self.c_attn = Channel_via_MSA(embed_dim, embed_dim, dim_ratio=int(mr), num_heads=nh, qkv_bias=False,
                                activation=mlp_activation, attn_drop=0.,
                                proj_drop=0.)
        # if next_embed_dim == embed_dim:
        self.c_attn = Channel_via_Residual(self.c_attn, drop_path, mr = mr)

        self.s_conv = Spatial_via_Conv(embed_dim, embed_dim, kernel_size=3,
                                        act_layer=torch.nn.GELU, depth=2,
                                        residual_block=True, drop_path=0)
        self.mr = mr
        if mr > 0:
            h = int(embed_dim * mr)
            self.mlp = Residual(torch.nn.Sequential(
                Linear_BN(embed_dim, h, resolution=resolution),
                mlp_activation(),
                Linear_BN(h, embed_dim, bn_weight_init=0, resolution=resolution),
                ), drop_path)
        self.norm_sc = torch.nn.LayerNorm(embed_dim)
        self.norm_mlp = torch.nn.LayerNorm(embed_dim)

    def forward(self, x, H_x, W_x, out_attn=None):
        x = self.c_attn(x, out_attn, H_x, W_x)
        x = self.s_conv(self.norm_sc(x), H_x, W_x)
        if self.mr > 0:
            x = self.mlp(self.norm_mlp(x), H_x, W_x)
        return x

class SABlock(nn.Module):
    def __init__(self, dim, num_heads, window_size=7, shift_size=0,
                 mlp_ratio=4., qkv_bias=True, qk_scale=None, drop=0., attn_drop=0., drop_path=0.,
                 act_layer=nn.GELU, norm_layer=nn.LayerNorm):
        super().__init__()
        self.dim = dim
        self.num_heads = num_heads
        self.window_size = window_size
        self.shift_size = shift_size
        self.mlp_ratio = mlp_ratio
        assert 0 <= self.shift_size < self.window_size, "shift_size must in 0-window_size"

        self.norm1 = norm_layer(dim)
        self.attn = WindowAttention(
            dim, window_size=to_2tuple(self.window_size), num_heads=num_heads,
            qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)

        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        mlp_hidden_dim = int(dim * mlp_ratio)

        self.norm2 = norm_layer(dim)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

        self.H = None
        self.W = None

    def forward(self, x, mask_matrix, H=None, W=None):
        B, L, C = x.shape
        if H is None:
            H, W = self.H, self.W
        assert L == H * W, "input feature has wrong size"

        shortcut = x
        x = self.norm1(x)
        x = x.view(B, H, W, C)
        pad_l = pad_t = 0
        pad_r = (self.window_size - W % self.window_size) % self.window_size
        pad_b = (self.window_size - H % self.window_size) % self.window_size
        x = F.pad(x, (0, 0, pad_l, pad_r, pad_t, pad_b))
        _, Hp, Wp, _ = x.shape

        if self.shift_size > 0:
            shifted_x = torch.roll(x, shifts=(-self.shift_size, -self.shift_size), dims=(1, 2))
            attn_mask = mask_matrix
        else:
            shifted_x = x
            attn_mask = None

        # partition windows
        x_windows = window_partition(shifted_x, self.window_size)  # nW*B, window_size, window_size, C
        x_windows = x_windows.view(-1, self.window_size * self.window_size, C)  # nW*B, window_size*window_size, C

        # W-MSA/SW-MSA
        attn_windows = self.attn(x_windows, mask=attn_mask)  # nW*B, window_size*window_size, C

        # merge windows
        attn_windows = attn_windows.view(-1, self.window_size, self.window_size, C)
        shifted_x = window_reverse(attn_windows, self.window_size, Hp, Wp)  # B H' W' C

        # reverse cyclic shift
        if self.shift_size > 0:
            x = torch.roll(shifted_x, shifts=(self.shift_size, self.shift_size), dims=(1, 2))
        else:
            x = shifted_x

        if pad_r > 0 or pad_b > 0:
            x = x[:, :H, :W, :].contiguous()

        x = x.view(B, H * W, C)

        # FFN
        x = shortcut + self.drop_path(x)
        x = x + self.drop_path(self.mlp(self.norm2(x), H, W))
        return x

class DOTBlock(nn.Module):
    """ A basic Swin Transformer layer for one stage.

    Args:
        dim (int): Number of feature channels
        depth (int): Depths of this stage.
        num_heads (int): Number of attention head.
        window_size (int): Local window size. Default: 7.
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim. Default: 4.
        qkv_bias (bool, optional): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set.
        drop (float, optional): Dropout rate. Default: 0.0
        attn_drop (float, optional): Attention dropout rate. Default: 0.0
        drop_path (float | tuple[float], optional): Stochastic depth rate. Default: 0.0
        norm_layer (nn.Module, optional): Normalization layer. Default: nn.LayerNorm
        downsample (nn.Module | None, optional): Downsample layer at the end of the layer. Default: None
        use_checkpoint (bool): Whether to use checkpointing to save memory. Default: False.
    """

    def __init__(self,
                 dim,
                 depth,
                 num_heads,
                 window_size=7,
                 mlp_ratio=4.,
                 qkv_bias=True,
                 qk_scale=None,
                 drop=0.,
                 attn_drop=0.,
                 drop_path=0.,
                 norm_layer=nn.LayerNorm,
                 use_checkpoint=False,
                 alldepths=2):
        super().__init__()
        self.window_size = window_size
        self.shift_size = window_size // 2
        self.depth = depth
        self.use_checkpoint = use_checkpoint

        # build blocks
        self.blocks = nn.ModuleList([
            SABlock(
                dim=dim,
                num_heads=num_heads,
                window_size=window_size,
                shift_size=0 if (i % 2 == 0) else window_size // 2,
                mlp_ratio=mlp_ratio,
                qkv_bias=qkv_bias,
                qk_scale=qk_scale,
                drop=drop,
                attn_drop=attn_drop,
                drop_path=drop_path[i] if isinstance(drop_path, list) else drop_path,
                norm_layer=norm_layer)
            for i in range(depth)])

        self.cablocks = nn.ModuleList([CABlock(dim, dim, 64, 8, 2, 2, torch.nn.Hardswish, torch.nn.Hardswish, resolution=224, drop_path=0.)
        for i in range(alldepths-depth)])

    def forward(self, x, H, W):
        Hp = int(np.ceil(H / self.window_size)) * self.window_size
        Wp = int(np.ceil(W / self.window_size)) * self.window_size
        img_mask = torch.zeros((1, Hp, Wp, 1), device=x.device)  # 1 Hp Wp 1
        h_slices = (slice(0, -self.window_size),
                    slice(-self.window_size, -self.shift_size),
                    slice(-self.shift_size, None))
        w_slices = (slice(0, -self.window_size),
                    slice(-self.window_size, -self.shift_size),
                    slice(-self.shift_size, None))
        cnt = 0
        for h in h_slices:
            for w in w_slices:
                img_mask[:, h, w, :] = cnt
                cnt += 1

        mask_windows = window_partition(img_mask, self.window_size)  # nW, window_size, window_size, 1
        mask_windows = mask_windows.view(-1, self.window_size * self.window_size)
        attn_mask = mask_windows.unsqueeze(1) - mask_windows.unsqueeze(2)
        attn_mask = attn_mask.masked_fill(attn_mask != 0, float(-100.0)).masked_fill(attn_mask == 0, float(0.0))
        
        for n, blk in enumerate(self.blocks):
            blk.H, blk.W = H, W
            if self.use_checkpoint:
                x = checkpoint.checkpoint(blk, x, attn_mask)
            else:
                x = blk(x, attn_mask)
        for blk in self.cablocks:
            x = blk(x, H, W)
        return x, H, W

# Merging
class PatchMerging(nn.Module):
    """ Patch Merging Layer

    Args:
        dim (int): Number of input channels.
        norm_layer (nn.Module, optional): Normalization layer.  Default: nn.LayerNorm
    """
    def __init__(self, dim, dimout, norm_layer=nn.LayerNorm):
        super().__init__()
        self.dim = dim
        self.dimout = dimout
        self.reduction = nn.Linear(4*dim, dimout, bias=False)
        self.norm = norm_layer(4*dim)

    def forward(self, x, H, W):
        """ Forward function.

        Args:
            x: Input feature, tensor size (B, H*W, C).
            H, W: Spatial resolution of the input feature.
        """
        B, L, C = x.shape
        assert L == H * W, "input feature has wrong size"

        x = x.view(B, H, W, C)

        # padding
        pad_input = (H % 2 == 1) or (W % 2 == 1)
        if pad_input:
            x = F.pad(x, (0, 0, 0, W % 2, 0, H % 2))

        x0 = x[:, 0::2, 0::2, :]  # B H/2 W/2 C
        x1 = x[:, 1::2, 0::2, :]  # B H/2 W/2 C
        x2 = x[:, 0::2, 1::2, :]  # B H/2 W/2 C
        x3 = x[:, 1::2, 1::2, :]  # B H/2 W/2 C
        x = torch.cat([x0, x1, x2, x3], -1)  # B H/2 W/2 4*C
        x = x.view(B, -1, 4 * C)  # B H/2*W/2 4*C

        x = self.norm(x)
        x = self.reduction(x)

        return x
class PatchEmbed(nn.Module):
    def __init__(self, patch_size=4, in_chans=3, embed_dim=96, norm_layer=None):
        super().__init__()
        patch_size = to_2tuple(patch_size)
        self.patch_size = patch_size

        self.in_chans = in_chans
        self.embed_dim = embed_dim

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)
        if norm_layer is not None:
            self.norm = norm_layer(embed_dim)
        else:
            self.norm = None

    def forward(self, x):
        """Forward function."""
        # padding
        _, _, H, W = x.size()

        if self.in_chans == 12:
            x = torch.cat([x[:, :, ::2, ::2],
                       x[:, :, 1::2, ::2],
                       x[:, :, ::2, 1::2],
                       x[:, :, 1::2, 1::2]], 1) 
            _, _, H, W = x.size()

        if W % self.patch_size[1] != 0:
            x = F.pad(x, (0, self.patch_size[1] - W % self.patch_size[1]))
        if H % self.patch_size[0] != 0:
            x = F.pad(x, (0, 0, 0, self.patch_size[0] - H % self.patch_size[0]))

        x = self.proj(x)  # B C Wh Ww
        if self.norm is not None:
            Wh, Ww = x.size(2), x.size(3)
            x = x.flatten(2).transpose(1, 2)
            x = self.norm(x)
            x = x.transpose(1, 2).view(-1, self.embed_dim, Wh, Ww)

        return x
    
# Link:
class SAEBlock(torch.nn.Module):
    def __init__(self, dim, num_heads, mlp_ratio=2., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=torch.nn.Hardswish, norm_layer=nn.LayerNorm):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp_atten(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

        self.upsample_2 = torch.nn.ConvTranspose2d(dim, dim, kernel_size=2, stride=2)
    # up link
    def forward(self, x, HW, last_layer=False):
        low_x, high_x = x[0], x[1]
        low_H_x, low_W_x = HW[0]
        high_H_x, high_W_x = HW[1]
        B, _, C = low_x.shape
        # v2:
        if last_layer:
            # 以低层的HW为准
            upsample_high_x = high_x.permute(0, 2, 1)
            upsample_high_x = self.upsample_2(upsample_high_x.view(B, C, high_H_x, high_W_x)).view(B, C, -1)
            x = low_x + upsample_high_x.permute(0, 2, 1)[:, :low_x.shape[1], :].contiguous()
            # H_x, W_x = low_H_x, low_W_x
        else:
            # 以高层的HW为准
            pooling_low_x = torch.nn.functional.max_pool2d(low_x.permute(0, 2, 1).view(B, C, low_H_x, low_W_x), 1, stride=2).view(B, C, -1)
            x = pooling_low_x.permute(0, 2, 1).contiguous() + high_x
            # H_x, W_x = high_H_x, high_W_x

        x = x + self.drop_path(self.attn(self.norm1(x)))
        x = x + self.drop_path(self.mlp(self.norm2(x)))
        return x, x

class SAAModule(torch.nn.Module):
    def __init__(self, embed_dim, mr, mlp_activation, \
                 resolution, drop_path):
        super().__init__()
        """
        crossType: [cat, add]_[q, a]_[h, l, n]_[t1, t2]
        """
        super().__init__()
        self.c_attn = Channel_via_MSA(embed_dim, embed_dim, dim_ratio=int(mr), num_heads=8, qkv_bias=False,
                                activation=mlp_activation, attn_drop=0.,
                                proj_drop=0.)
        self.c_attn = Channel_via_Residual(self.c_attn, drop_path, mr = mr)
        self.s_conv = Spatial_via_Conv(embed_dim, embed_dim, kernel_size=3,
                                        act_layer=torch.nn.GELU, depth=2,
                                        residual_block=True, drop_path=0)
        self.mr = mr
        if mr > 0:
            h = int(embed_dim * mr)
            self.mlp = Residual(torch.nn.Sequential(
                Linear_BN(embed_dim, h, resolution=resolution),
                mlp_activation(),
                Linear_BN(h, embed_dim, bn_weight_init=0, resolution=resolution),
                ), drop_path)

        self.upsample_2 = torch.nn.ConvTranspose2d(embed_dim, embed_dim, kernel_size=2, stride=2)

        self.norm_sc = torch.nn.LayerNorm(embed_dim)
        self.norm_mlp = torch.nn.LayerNorm(embed_dim)

    def forward(self, x, HW, caadd=False, last_layer=False):            
        low_x, high_x = x[0], x[1]
        low_H_x, low_W_x = HW[0]
        high_H_x, high_W_x = HW[1]
        B, _, C = low_x.shape
        upsample_high_x = high_x.permute(0, 2, 1)
        upsample_high_x = self.upsample_2(upsample_high_x.view(B, C, high_H_x, high_W_x)).view(B, C, -1)
        x = low_x + upsample_high_x.permute(0, 2, 1)[:, :low_x.shape[1], :].contiguous()
        H_x, W_x = low_H_x, low_W_x 
        B, _, C = x.shape
        x = self.c_attn(x, H_x, W_x)
        x = self.s_conv(self.norm_sc(x), H_x, W_x)
        if self.mr > 0:
            x = self.mlp(self.norm_mlp(x), H_x, W_x)

        return x, torch.nn.functional.max_pool2d(x.permute(0, 2, 1).view(B, C, H_x, W_x), 1, stride=2).view(B, C, -1).transpose(-2, -1)


class Mlp_atten(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class Attention(nn.Module):
    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        # NOTE scale factor was wrong in my original version, can set manually to be compat with prev weights
        self.scale = qk_scale or head_dim ** -0.5

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]   # make torchscript happy (cannot use tensor as tuple)

        attn = (q @ k.transpose(-2, -1)) * self.scale
        attn = attn.softmax(dim=-1)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x



class Block(nn.Module):

    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=torch.nn.Hardswish, norm_layer=nn.LayerNorm):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

    def forward(self, x):
        x = x + self.drop_path(self.attn(self.norm1(x)))
        x = x + self.drop_path(self.mlp(self.norm2(x)))
        return x,x


