import ast
from typing import List, Dict, Tuple, Any
from collections import defaultdict

def _extract_logic_paths(source_code: str, max_paths: int = 10) -> List[List[str]]:
    """
    遍历 AST 中的 if/elif/else，提取所有可能的路径条件（每条到达 return 的路径）
    """
    tree = ast.parse(source_code)
    paths: List[List[str]] = []

    def visit(node: ast.AST, conds: List[str]):
        if isinstance(node, ast.Return):
            paths.append(list(conds))
            return

        if isinstance(node, ast.If):
            test_expr = ast.unparse(node.test).strip()
            for child in node.body:
                visit(child, conds + [test_expr])
            for child in node.orelse:
                visit(child, conds + [f"not ({test_expr})"])
        else:
            for child in ast.iter_child_nodes(node):
                visit(child, conds)

    visit(tree, [])
    return paths if paths else [["<unconditional>"]]

def _extract_def_use(source_code: str) -> Tuple[Dict[str, int], Dict[str, List[int]]]:
    """
    提取所有变量的定义行和使用行
    """
    tree = ast.parse(source_code)
    defs: Dict[str, int] = {}
    uses: Dict[str, List[int]] = defaultdict(list)

    class DFVisitor(ast.NodeVisitor):
        def visit_FunctionDef(self, node: ast.FunctionDef):
            for arg in node.args.args:
                defs[arg.arg] = node.lineno
                uses[arg.arg].append(node.lineno)
            self.generic_visit(node)

        def visit_Assign(self, node: ast.Assign):
            for tgt in node.targets:
                if isinstance(tgt, ast.Name):
                    defs[tgt.id] = node.lineno
            self.generic_visit(node)

        def visit_AugAssign(self, node: ast.AugAssign):
            tgt = node.target
            if isinstance(tgt, ast.Name):
                defs[tgt.id] = node.lineno
            self.generic_visit(node)

        def visit_Name(self, node: ast.Name):
            if isinstance(node.ctx, ast.Load):
                uses[node.id].append(node.lineno)

    DFVisitor().visit(tree)
    return defs, uses

def generate_logic_summary(source_code: str) -> List[Dict[str, Any]]:
    """
    每条路径对应一个 dict，包含 path_conditions、defs、uses
    """
    try:
        paths = _extract_logic_paths(source_code)
        defs, uses = _extract_def_use(source_code)
    except SyntaxError as e:
        return [{
            "path_conditions": [f"SyntaxError at line {e.lineno}: {e.msg}"],
            "defs": {},
            "uses": {}
        }]

    summary: List[Dict[str, Any]] = []
    for path in paths:
        path_defs = {}
        path_uses = {}

        for var, def_line in defs.items():
            path_defs[var] = ["参数传入"] if def_line == 1 else [def_line]  # 假设 line 1 为函数头

        for var, use_lines in uses.items():
            path_uses[var] = sorted(use_lines)

        summary.append({
            "path_conditions": path,
            "defs": path_defs,
            "uses": path_uses
        })

    return summary