import ast
from typing import List, Dict, Any
from collections import defaultdict

def _clean_source_code(source_code: str) -> str:
    """Remove surrounding markdown code fences if present."""
    # Check for triple-backtick fences and strip them
    stripped = source_code.lstrip()
    if stripped.startswith("```") and stripped.rstrip().endswith("```"):
        lines = stripped.splitlines()
        # Remove the first and last lines (the fence lines)
        return "\n".join(lines[1:-1])
    return source_code

def _extract_logic_paths(source_code: str, max_paths: int = 50) -> List[Dict[str, Any]]:
    """
    遍历 AST 中的 if/elif/else、for、while、try/except，
    提取所有可能的路径：
      - conditions: 该路径上每个分支的条件或循环/异常描述
      - return: 在该路径上执行的返回表达式（字符串）
    对于 if 的 else 分支，统一标记为 "else"。
    """
    # Clean Markdown fences before parsing
    source_code = _clean_source_code(source_code)
    tree = ast.parse(source_code)
    paths: List[Dict[str, Any]] = []

    def visit(node: ast.AST, conds: List[str]):
        if len(paths) >= max_paths:
            return

        if isinstance(node, ast.Return):
            return_expr = ast.unparse(node.value).strip() if node.value else None
            paths.append({
                "conditions": list(conds),
                "return": return_expr
            })
            return

        if isinstance(node, ast.If):
            test = ast.unparse(node.test).strip()
            for child in node.body:
                visit(child, conds + [test])
            for child in node.orelse:
                visit(child, conds + ["else"])

        elif isinstance(node, ast.For):
            desc = f"for {ast.unparse(node.target).strip()} in {ast.unparse(node.iter).strip()}"
            exit_desc = f"exit {desc}"
            for child in node.body:
                visit(child, conds + [desc])
            for child in node.orelse:
                visit(child, conds + [exit_desc])

        elif isinstance(node, ast.While):
            test = ast.unparse(node.test).strip()
            exit_desc = f"exit {test}"
            for child in node.body:
                visit(child, conds + [test])
            for child in node.orelse:
                visit(child, conds + [exit_desc])

        elif isinstance(node, ast.Try):
            for child in node.body:
                visit(child, conds + ["try"])
            for handler in node.handlers:
                etype = ast.unparse(handler.type).strip() if handler.type else "Exception"
                for child in handler.body:
                    visit(child, conds + [f"except {etype}"])
        
        else:
            for child in ast.iter_child_nodes(node):
                visit(child, conds)

    visit(tree, [])
    return paths if paths else [{"conditions": [], "return": None}]


def generate_logic_summary(source_code: str) -> Dict[str, Any]:
    """
    返回增强版逻辑摘要，包含：
      - paths: 每条可能路径的条件列表与返回表达式
      - path_conditions: 所有唯一的条件描述（可用于概览）
      - definitions: 变量定义所在的行号
      - operations: 每个变量在赋值或增强赋值时对应的完整语句（字符串列表）
      - calls: 所有函数调用的调用表达式及其出现次数
    """
    # Clean Markdown fences
    cleaned = _clean_source_code(source_code)
    try:
        paths = _extract_logic_paths(cleaned)
    except SyntaxError as e:
        return {"error": f"SyntaxError at line {e.lineno}: {e.msg}"}

    tree = ast.parse(cleaned)
    definitions: Dict[str, int] = {}
    operations: Dict[str, List[str]] = defaultdict(list)
    calls: Dict[str, int] = defaultdict(int)

    class Visitor(ast.NodeVisitor):
        def visit_FunctionDef(self, node: ast.FunctionDef):
            for arg in node.args.args:
                definitions[arg.arg] = node.lineno
            self.generic_visit(node)

        def visit_Assign(self, node: ast.Assign):
            stmt = ast.unparse(node).strip()
            for tgt in node.targets:
                if isinstance(tgt, ast.Name):
                    var = tgt.id
                    definitions[var] = node.lineno
                    operations[var].append(stmt)
            self.generic_visit(node)

        def visit_AugAssign(self, node: ast.AugAssign):
            stmt = ast.unparse(node).strip()
            if isinstance(node.target, ast.Name):
                var = node.target.id
                definitions[var] = node.lineno
                operations[var].append(stmt)
            self.generic_visit(node)

        def visit_Call(self, node: ast.Call):
            call_expr = ast.unparse(node).strip()
            calls[call_expr] += 1
            self.generic_visit(node)

    Visitor().visit(tree)

    unique_conditions = sorted({
        cond
        for path in paths
        for cond in path["conditions"]
    })

    return {
        "paths": paths,
        "path_conditions": unique_conditions,
        "definitions": definitions,
        "operations": dict(operations),
        "calls": dict(calls)
    }

if __name__ == "__main__":
    # 示例：带有 Markdown 代码围栏的源代码
    source = "```\ndef reverse_delete(s, c):\n    res = \"\"\n    for i in range(len(s)):\n        if c.find(s[i]) == -1:\n            res += s[i]\n    v = []\n    v.append(res)\n    if res == res[-1]:\n        v.append(\"True\")\n    else:\n        v.append(\"False\")\n    return v\n```"
    import json
    summary = generate_logic_summary(source)
    print(json.dumps(summary, indent=2, ensure_ascii=False))
