import torch
from torch import nn
from torch.nn import functional as F

from XXX.uib.utils.safe_module import SafeModule

from experiments.models.stochastic_dropout import StochasticDropout2d
from experiments.models.stochastic_model import StochasticModel


class DropoutModel(SafeModule):
    C: int

    def __init__(self, C: int):
        super().__init__()

        self.C = C

        self.conv1 = nn.Conv2d(1, 32, 3, 1)
        self.conv2 = nn.Conv2d(32, 64, 3, 1)
        self.dropout1 = nn.Dropout2d(0.25)
        self.dropout2 = nn.Dropout2d(0.5)
        self.fc1 = nn.Linear(9216, 128)
        self.fc2 = nn.Linear(128, C)

    def safe_forward(self, x):
        x = self.conv1(x)
        x = F.relu(x)
        x = self.conv2(x)
        x = F.max_pool2d(x, 2)
        x = self.dropout1(x)
        x = torch.flatten(x, 1)
        x = self.fc1(x)
        x = F.relu(x)
        x = self.dropout2(x)
        x = self.fc2(x)
        # output = F.log_softmax(x, dim=1)
        return x


class DeterministicModel(SafeModule):
    C: int

    def __init__(self, C: int):
        super().__init__()

        self.C = C

        self.conv1 = nn.Conv2d(1, 32, 3, 1)
        self.conv2 = nn.Conv2d(32, 64, 3, 1)
        self.fc1 = nn.Linear(9216, 128)
        self.fc2 = nn.Linear(128, C)

    def safe_forward(self, x):
        x = self.conv1(x)
        x = F.relu(x)
        x = self.conv2(x)
        x = F.max_pool2d(x, 2)
        x = torch.flatten(x, 1)
        x = self.fc1(x)
        x = F.relu(x)
        x = self.fc2(x)
        # output = F.log_softmax(x, dim=1)
        return x


class StochasticDropoutModel(StochasticModel):
    C: int

    def __init__(self, C: int, num_samples):
        super().__init__(num_samples)

        self.C = C

        self.conv1 = nn.Conv2d(1, 32, 3, 1)
        self.conv2 = nn.Conv2d(32, 64, 3, 1)
        self.dropout1 = StochasticDropout2d(0.25)
        self.dropout2 = StochasticDropout2d(0.5)
        self.fc1 = nn.Linear(9216, 128)
        self.fc2 = nn.Linear(128, C)

    def deterministic_forward_impl(self, x: torch.Tensor):
        x = self.conv1(x)
        x = F.relu(x)
        x = self.conv2(x)
        x = F.max_pool2d(x, 2)
        return x

    def stochastic_forward_impl(self, x: torch.Tensor):
        x = self.dropout1(x)
        x = torch.flatten(x, 1)
        x = self.fc1(x)
        x = F.relu(x)
        x = self.dropout2(x)
        x = self.fc2(x)
        # output = F.log_softmax(x, dim=1)
        return x


if __name__ == "__main__":
    from experiments.utils import print_module

    print_module(DropoutModel(10))

    print_module(DeterministicModel(10))

    print_module(StochasticDropoutModel(10))
