from math import fabs, cos, sin, radians, sqrt, acos, degrees
import numpy as np
from enum import Enum


ROLL_THRESHOLD = 1e-2

class RollDirection(Enum):
    LEFT = 1
    RIGHT = 2
    NOROLL = 3


def roll2xy_on_unit_circle(roll):
    return cos(radians(roll)), -sin(radians(roll))


def get_roll_direction(roll_1: float, roll_2: float, roll_threshold: float=ROLL_THRESHOLD):
    if fabs(roll_1 - roll_2) < roll_threshold:
        return RollDirection.NOROLL
    
    x1, y1 = roll2xy_on_unit_circle(roll_1)
    x2, y2 = roll2xy_on_unit_circle(roll_2)

    cross_product = x1 * y2 - y1 * x2

    if cross_product > 0.:
        return RollDirection.LEFT
    else:
        return RollDirection.RIGHT 


def get_roll_deg(roll_1: float, roll_2: float):
    x1, y1 = roll2xy_on_unit_circle(roll_1)
    x2, y2 = roll2xy_on_unit_circle(roll_2)

    inner_product = x1 * x2 + y1 * y2

    tmp = inner_product / (sqrt(x1 * x1 + y1 * y1) * sqrt(x2 * x2 + y2 * y2))
    tmp = np.clip(tmp, a_min=-1., a_max=1.)
    
    res_rad = acos(tmp)

    return degrees(res_rad)