"""imagenet train
"""
import argparse
import os
import random
import shutil
import time
import warnings
from enum import Enum
import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
from torch.optim.lr_scheduler import StepLR
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torchvision.models as models
from torch.utils.data import Subset
#load model
from model import construct_PPNet,Clip_ProtoPNet
from dataset import CUB200_Concept
model_names = sorted(name for name in models.__dict__
    if name.islower() and not name.startswith("__")
    and callable(models.__dict__[name]))
import pdb

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
parser.add_argument('data', metavar='DIR', nargs='?', default='/data/wangjiaqi/datasets/Flower/Flower_splited/',
                    help='path to dataset (default: imagenet)')
parser.add_argument('save_path', metavar='DIR', nargs='?', default='./saved_model/',
                    help='path to dataset (default: imagenet)')
parser.add_argument('-a', '--arch', metavar='ARCH', default='vgg19',
                    choices=model_names,
                    help='model architecture: ' +
                        ' | '.join(model_names) +
                        ' (default: resnet18)')
parser.add_argument('-j', '--workers', default=8, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=300, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--update_G', default=30, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--warm_epochs', default=0, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=64, type=int,
                    metavar='N',
                    help='mini-batch size (default: 256), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--lr', '--learning-rate', default=1e-4, type=float,
                    metavar='LR', help='initial learning rate', dest='lr') #lr_step_size
parser.add_argument('--lr_step_size', default=50, type=int,
                    metavar='LRS', help='initial learning rate')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--G_momentum', default=0.1, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--coeff_logits', default=0, type=float, metavar='coeff',
                    help='ppnet logits')
parser.add_argument('--gpuid',type=str, default='4,5,6,7') # python3 main.py -gpuid=0,1,2,3
parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)',
                    dest='weight_decay')
parser.add_argument('-p', '--print-freq', default=10, type=int,
                    metavar='N', help='q_log.log frequency (default: 10)')
parser.add_argument('--resume', default=None, type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                    help='use pre-trained model')
parser.add_argument('--L1', dest='L1', action='store_true',
                    help='use pre-trained model')
parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int,
                    help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str,
                    help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str,
                    help='distributed backend')
parser.add_argument('--seed', default=None, type=int,
                    help='seed for initializing training. ')
parser.add_argument('--gpu', default=None, type=int,
                    help='GPU id to use.')
parser.add_argument('--times', default="test20240408_4K_gpt_l1", type=str,
                    help='logs.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')
parser.add_argument('--dummy', action='store_true', help="use fake data to benchmark")

best_acc1 = 0
pretrained_ppnet = {"vgg16":"/data/fengyi/wangjiaqi/code/mm-proto-pnet/VL-MA-offline/pretrained_models/cub/prototype_dim512/vgg16/9nopush0.7689.pth",
                    "vgg19":"/data/fengyi/wangjiaqi/code/mm-proto-pnet/VL-MA-offline/pretrained_models/cub/prototype_dim512/vgg19/9nopush0.7822.pth",
                    "resnet34":"/data/fengyi/wangjiaqi/code/mm-proto-pnet/VL-MA-offline/pretrained_models/cub/prototype_dim512/resnet34/9nopush0.7843.pth",
                    "resnet152":"/data/fengyi/wangjiaqi/code/mm-proto-pnet/VL-MA-offline/pretrained_models/cub/prototype_dim512/resnet152/9nopush0.7846.pth",
                    "densenet121":"/data/fengyi/wangjiaqi/code/mm-proto-pnet/VL-MA-offline/pretrained_models/cub/prototype_dim512/densenet121/9nopush0.7810.pth",
                    "densenet161": "/data/fengyi/wangjiaqi/code/mm-proto-pnet/VL-MA-offline/pretrained_models/cub/prototype_dim512/densenet161/9nopush0.8057.pth"}
class Q_log:
    def __init__(self, path):
        self.log_path = os.path.join(path,'training_log.txt')
    def log(self, *args, **kwargs):
        with open(self.log_path, 'a') as f:
            print(*args, **kwargs, file=f)  # 写入文件
        print(*args, **kwargs)  # 正常打印到控制台
    
def main():
    args = parser.parse_args()
    #q_log.log(os.environ['CUDA_VISIBLE_DEVICES'] )
    args.save_path  = os.path.join(args.save_path,args.arch,args.times)
    if not os.path.exists(args.save_path):
        os.makedirs(args.save_path)
    shutil.copy(src=os.path.join(os.getcwd(), __file__), dst=args.save_path ) #__file__.py就是当前的这个文件
    shutil.copy(src=os.path.join(os.getcwd(), 'train_model.py'), dst=args.save_path )
    #shutil.copy(src=os.path.join(os.getcwd(), base_architecture_type + '_features.py'), dst=args.save_path )
    shutil.copy(src=os.path.join(os.getcwd(), 'model.py'), dst=args.save_path )
    global q_log
    q_log = Q_log(args.save_path)
    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)
def get_classes(data_dir):
    all_data = datasets.ImageFolder(data_dir)
    return all_data.classes


def finetune_only(model): 
    
    if hasattr(model,"module"):
        for p in model.module.protopnet.parameters():
            p.requires_grad = True
        #model.module.background.requires_grad = True
        model.module.classifier_c.requires_grad = True
        #model.module.logit_scale.requires_grad = True
    else:
        for p in model.protopnet.parameters():
            p.requires_grad = True

        #model.background.requires_grad = True
        model.classifier_c.requires_grad = True
        #model.logit_scale.requires_grad = True
        
def fc_only(model): 
    
    if hasattr(model,"module"):
        for p in model.module.protopnet.parameters():
            p.requires_grad = False
        #model.module.background.requires_grad = True
        model.module.classifier_c.requires_grad = True
        #model.module.logit_scale.requires_grad = True
    else:
        for p in model.protopnet.parameters():
            p.requires_grad =  False

        #model.background.requires_grad = True
        model.classifier_c.requires_grad = False
        #model.logit_scale.requires_grad = True
def main_worker(gpu, ngpus_per_node, args):
    global best_acc1
    args.gpu = gpu
    traindir = os.path.join(args.data, 'train')
    classes = 200#get_classes(traindir)
    q_log.log("class num:{}".format(classes))
    if args.gpu is not None:
        q_log.log("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
    # create model

    if args.pretrained:
        #q_log.log("=> using pre-trained model '{}'".format(args.arch))
        #model = models.__dict__[args.arch](pretrained=True)
        #model = base_architecture_to_model[args.arch](pretrained=True,num_classes=len(classes))
        ppnet_model = construct_PPNet(base_architecture=args.arch,
                            pretrained=True, img_size=224,
                            prototype_shape=(2000,512,1,1),
                            num_classes=200,
                            prototype_activation_function="log",
                            add_on_layers_type = "regular")
        q_log.log("=> load pre-trained ppnet model '{}'".format(args.arch))
        loaded_model = torch.load(pretrained_ppnet[args.arch])
        # 从加载的模型中提取状态字典
        loaded_state_dict = loaded_model.state_dict()
        ppnet_model.load_state_dict(loaded_state_dict, strict=False)
        
        model = Clip_ProtoPNet(ppnet_model,200,1,args.G_momentum )
        
        
    else:
        q_log.log("=> creating model '{}'".format(args.arch))
        model = construct_PPNet(base_architecture=args.arch,
                    pretrained=False, img_size=224,
                    prototype_shape=(len(classes)*10,512,1,1),
                    num_classes=len(classes),
                    prototype_activation_function="log",
                    add_on_layers_type = "regular")

    finetune_only(model)
    #fc_only(model)
    # if not torch.cuda.is_available():
    #     q_log.log('using CPU, this will be slow')
    # elif args.distributed:
    #     # For multiprocessing distributed, DistributedDataParallel constructor
    #     # should always set the single device scope, otherwise,
    #     # DistributedDataParallel will use all available devices.
    #     if args.gpu is not None:
    #         torch.cuda.set_device(args.gpu)
    #         model.cuda(args.gpu) #分不到不同GPU了
    #         # When using a single GPU per process and per
    #         # DistributedDataParallel, we need to divide the batch size
    #         # ourselves based on the total number of GPUs of the current node.
    #         args.batch_size = int(args.batch_size / ngpus_per_node)
    #         args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
    #         model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu],find_unused_parameters=True)
    #     else:
    #         model.cuda()
    #         # DistributedDataParallel will divide and allocate batch_size to all
    #         # available GPUs if device_ids are not set
    #         model = torch.nn.parallel.DistributedDataParallel(model,find_unused_parameters=True)
    # elif args.gpu is not None:
    #     torch.cuda.set_device(args.gpu)
    #     model = model.cuda(args.gpu)
    # else:
    #     # # DataParallel will divide and allocate batch_size to all available GPUs
    #     # if args.arch.startswith('alexnet') or args.arch.startswith('vgg'):
    #     #     #model.features = torch.nn.DataParallel(model.features)
    #     #     model.cuda()
    #     # else:
    model = model.cuda()
    #model = torch.nn.DataParallel(model)

    # define loss function (criterion), optimizer, and learning rate scheduler
    criterion = nn.CrossEntropyLoss().cuda(args.gpu)
#     joint_optimizer_specs = [{'params': model.protopnet.features.parameters(), 'lr': 1e-4, 'weight_decay': 1e-3}, # bias are now also being regularized
#  {'params': model.protopnet.add_on_layers.parameters(), 'lr': 3e-3, 'weight_decay': 1e-3},
#  {'params': model.protopnet.prototype_vectors, 'lr': 3e-3},
#  {'params': model.classifier_c.parameters(), 'lr': 1e-4}]  
# # model.classifier_c
    joint_optimizer_specs = [{'params': model.protopnet.features.parameters(), 'lr': 1e-4, 'weight_decay': 1e-3}, # bias are now also being regularized
    {'params': model.protopnet.add_on_layers.parameters(), 'lr': 3e-3, 'weight_decay': 1e-3},
    {'params': model.protopnet.prototype_vectors, 'lr': 3e-3},
    {'params': model.classifier_c.parameters(), 'lr': 1e-4}]  
    """Sets the learning rate to the initial LR decayed by 10 every 30 epochs"""
    #optimizer = torch.optim.AdamW(joint_optimizer_specs, lr=3e-3, eps=1e-07, weight_decay=1e-3)
    #scheduler = torch.optim.lr_scheduler.MultiStepLR(optimizer=optimizer, milestones=args.milestones, gamma=args.gamma)
    #scheduler = StepLR(optimizer, step_size=30, gamma=0.1)
    optimizer = torch.optim.Adam(joint_optimizer_specs, lr = 1e-4, weight_decay = 1e-3)
    scheduler =  torch.optim.lr_scheduler.StepLR(optimizer, step_size=args.lr_step_size, gamma=0.5)
    #scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, T_max=args.epochs-args.warm_epochs, eta_min=1e-6, last_epoch=-1)
    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile("/data/wangjiaqi/VL/Flower102-online/saved_model/" + args.arch + args.resume):
            q_log.log("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load("/data/wangjiaqi/VL/Flower102-online/saved_model/" + args.arch + args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load("/data/wangjiaqi/VL/Flower102-online/saved_model/" + args.arch + args.resume, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            best_acc1 = checkpoint['best_acc1']
            q_log.log("=> loading checkpoint best_acc1:{}".format(best_acc1))
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint['state_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            scheduler.load_state_dict(checkpoint['scheduler'])
            q_log.log("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.resume, checkpoint['epoch']))
        else:
            q_log.log("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    # Data loading code
    if args.dummy:
        q_log.log("=> Dummy data is used!")
        train_dataset = datasets.FakeData(1281167, (3, 224, 224), 1000, transforms.ToTensor())
        val_dataset = datasets.FakeData(50000, (3, 224, 224), 1000, transforms.ToTensor())
    else:
        traindir = os.path.join(args.data, 'train')
        valdir = os.path.join(args.data, 'test')
        normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

            # data augmentation
        # train_transforms = transforms.Compose([
        #     transforms.Resize(size=256),
        #     transforms.RandomHorizontalFlip(),
        #     transforms.ColorJitter(0.1),
        #     transforms.RandomCrop(size=224),
        #     transforms.ToTensor(),
        #     transforms.Normalize(mean=(0.485, 0.456, 0.406), std=(0.229, 0.224, 0.225))
        #     ])
        # test_transforms = transforms.Compose([
        #     transforms.Resize(size=224),
        #     transforms.CenterCrop(size=224),
        #     transforms.ToTensor(),
        #     transforms.Normalize(mean=(0.485, 0.456, 0.406), std=(0.229, 0.224, 0.225))
        #     ])
        img_size = 224
        train_transforms =  transforms.Compose([
            transforms.Resize(size=(img_size, img_size)),
            transforms.RandomOrder([
            transforms.RandomPerspective(distortion_scale=0.2, p = 0.5),
            transforms.ColorJitter((0.6,1.4), (0.6,1.4), (0.6,1.4), (-0.02,0.02)),
            transforms.RandomHorizontalFlip(),
            transforms.RandomAffine(degrees=10, shear=(-2,2),translate=[0.05,0.05]),
            ]),
            transforms.ToTensor(),
            normalize,
        ])
        test_transforms = transforms.Compose([
                            transforms.Resize(size=(img_size, img_size)),
                            transforms.ToTensor(),
                            normalize
                        ])
        train_dataset =  CUB200_Concept(root="/data/fengyi/wangjiaqi/datasets/CUB_VL", arch=args.arch,train=True,transform=train_transforms, cropped = True)
        val_dataset =  CUB200_Concept(root="/data/fengyi/wangjiaqi/datasets/CUB_VL", arch=args.arch,train=False,transform=test_transforms, cropped = True)
    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
        val_sampler = torch.utils.data.distributed.DistributedSampler(val_dataset, shuffle=False, drop_last=True)
    else:
        train_sampler = None
        val_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler)

    val_loader = torch.utils.data.DataLoader(
        val_dataset, batch_size=args.batch_size, shuffle=False,
        num_workers=args.workers, pin_memory=True, sampler=val_sampler)

    if args.evaluate:
        validate(val_loader, model, criterion, args)
        return

    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
            # train for one epoch
            train(train_loader, model, criterion, optimizer, epoch, args)
            # evaluate on validation set
            acc1 = validate(val_loader, model, criterion, args)
            scheduler.step()
            # remember best acc@1 and save checkpoint
            is_best = acc1 > best_acc1
            best_acc1 = max(acc1, best_acc1)

            if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                    and args.rank % ngpus_per_node == 0):
                save_checkpoint({
                    'epoch': epoch + 1,
                    'arch': args.arch,
                    'state_dict': model.state_dict(),
                    'best_acc1': best_acc1,
                    'optimizer' : optimizer.state_dict(),
                    'scheduler' : scheduler.state_dict(),
                }, is_best,args.save_path)
        else:
            # train for one epoch
            train(train_loader, model, criterion, optimizer, epoch, args)
            # evaluate on validation set
            acc1 = validate(val_loader, model, criterion, args)
            scheduler.step()
            # remember best acc@1 and save checkpoint
            is_best = acc1 > best_acc1
            best_acc1 = max(acc1, best_acc1)

            if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                    and args.rank % ngpus_per_node == 0):
                save_checkpoint({
                    'epoch': epoch + 1,
                    'arch': args.arch,
                    'state_dict': model.state_dict(),
                    'best_acc1': best_acc1,
                    'optimizer' : optimizer.state_dict(),
                    'scheduler' : scheduler.state_dict(),
                }, is_best,args.save_path)
            

def ppnet_loss(model,min_distances,label):
    cppnet = model
    ppnet = model.protopnet
    max_dist = (ppnet.prototype_shape[1]
                            * ppnet.prototype_shape[2]
                            * ppnet.prototype_shape[3])# 512*1*1   512*14*14 作用类似于三元组损失的一个绝对路径 而不是相对距离

    # prototypes_of_correct_class is a tensor of shape batch_size * num_prototypes
    # calculate cluster cost
    class_ident = ppnet.prototype_class_identity.cuda()
    class_ident = class_ident[:,label].cuda()
    prototypes_of_correct_class = torch.t(class_ident).cuda()
    inverted_distances, _ = torch.max((max_dist - min_distances) * prototypes_of_correct_class, dim=1)
    cluster_cost = torch.mean(max_dist - inverted_distances)

    # calculate separation cost
    prototypes_of_wrong_class = 1 - prototypes_of_correct_class
    inverted_distances_to_nontarget_prototypes, _ = \
        torch.max((max_dist - min_distances) * prototypes_of_wrong_class, dim=1)
    separation_cost = torch.mean(max_dist - inverted_distances_to_nontarget_prototypes)
    
    l1_mask = 1 - torch.t(cppnet.concept_class_identity).cuda()
    l1 = (cppnet.classifier_c.weight * l1_mask).norm(p=1)
    return cluster_cost, separation_cost, l1


def train(train_loader, model, criterion, optimizer, epoch, args):
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    loss_ce = AverageMeter('Ce', ':.4e')
    loss_cls = AverageMeter('Cls', ':.4e')
    loss_sep = AverageMeter('Sep', ':.4e')
    loss_l1 = AverageMeter('L1', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, loss_ce, loss_cls, loss_sep,loss_l1, top1, top5],
        prefix="Epoch: [{}]".format(epoch))

    # switch to train mode
    model.train()

    end = time.time()
    for i, (images, target, encoded_img) in enumerate(train_loader):
        # measure data loading time
        #pdb.set_trace()  #encoded_img [32,10,512]
        #q_log.log(i)
        model.train()
        data_time.update(time.time() - end)

        #if args.gpu is not None:
        # if torch.cuda.is_available():
        #     images = images.cuda(args.gpu, non_blocking=True)
        #     encoded_img = encoded_img.cuda(args.gpu, non_blocking=True)
        #     target = target.cuda(args.gpu, non_blocking=True)
        images = images.cuda()
        encoded_img = encoded_img.cuda()
        target = target.cuda()
        # compute output
        output_list, min_distances = model(images,target,encoded_img)
        output = args.coeff_logits * output_list[0] + (1-args.coeff_logits) * output_list[1]
        cross_entropy = criterion(output, target)
        cluster_cost, separation_cost, l1_cost = ppnet_loss(model,min_distances,target)
        if args.L1:
            loss = cross_entropy + 0.8 * cluster_cost - 0.08 * separation_cost  + 1e-4 * l1_cost
        else:
            loss = cross_entropy + 0.8 * cluster_cost - 0.08 * separation_cost
        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        loss_ce.update(cross_entropy.item(), images.size(0))
        loss_cls.update(cluster_cost.item(), images.size(0))
        loss_sep.update(separation_cost.item(), images.size(0))
        loss_l1.update(l1_cost.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        if (i + 1) % args.update_G == 0:
            #q_log.log("")
            model.eval()
            with torch.no_grad():
                if hasattr(model,"module"):
                    model.module.update_rotation_matrix()
                else:
                    model.update_rotation_matrix()
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            progress.display(i + 1)

    progress.display_summary()


def validate(val_loader, model, criterion, args):

    def run_validate(loader, base_progress=0):
        with torch.no_grad():
            end = time.time()
            for i, (images, target) in enumerate(loader):
                i = base_progress + i
                # if args.gpu is not None:
                #     images = images.cuda(args.gpu, non_blocking=True)
                # if torch.cuda.is_available():
                #     target = target.cuda(args.gpu, non_blocking=True)
                images = images.cuda()
                target = target.cuda()
                # compute output
                output_list,_ = model(images) #
                output = args.coeff_logits * output_list[0] + (1-args.coeff_logits) * output_list[1]
                loss = criterion(output, target)

                # measure accuracy and record loss
                acc1, acc5 = accuracy(output, target, topk=(1, 5))
                losses.update(loss.item(), images.size(0))
                top1.update(acc1[0], images.size(0))
                top5.update(acc5[0], images.size(0))

                # measure elapsed time
                batch_time.update(time.time() - end)
                end = time.time()

                if i % args.print_freq == 0:
                    progress.display(i + 1)

    batch_time = AverageMeter('Time', ':6.3f', Summary.NONE)
    losses = AverageMeter('Loss', ':.4e', Summary.NONE)
    top1 = AverageMeter('Acc@1', ':6.2f', Summary.AVERAGE)
    top5 = AverageMeter('Acc@5', ':6.2f', Summary.AVERAGE)
    progress = ProgressMeter(
        len(val_loader) + (args.distributed and (len(val_loader.sampler) * args.world_size < len(val_loader.dataset))),
        [batch_time, losses, top1, top5],
        prefix='Test: ')

    # switch to evaluate mode
    model.eval()

    run_validate(val_loader)
    if args.distributed:
        #top1=top1.cuda(args.gpu, non_blocking=True)
        #top5=top5.cuda(args.gpu, non_blocking=True)
        top1.all_reduce()
        top5.all_reduce()

    if args.distributed and (len(val_loader.sampler) * args.world_size < len(val_loader.dataset)):
        aux_val_dataset = Subset(val_loader.dataset,
                                 range(len(val_loader.sampler) * args.world_size, len(val_loader.dataset)))
        aux_val_loader = torch.utils.data.DataLoader(
            aux_val_dataset, batch_size=args.batch_size, shuffle=False,
            num_workers=args.workers, pin_memory=True)
        run_validate(aux_val_loader, len(val_loader))

    progress.display_summary()

    return top1.avg

def save_warm_checkpoint(state, is_best, file_dir, filename='warm_checkpoint.pth.tar'):
    torch.save(state, os.path.join(file_dir, filename))
    if is_best:
        shutil.copyfile(os.path.join(file_dir,filename), os.path.join(file_dir, 'warm_model_best.pth.tar'))

def save_checkpoint(state, is_best, file_dir, filename='checkpoint.pth.tar'):
    torch.save(state, os.path.join(file_dir, filename))
    if is_best:
        shutil.copyfile(os.path.join(file_dir,filename), os.path.join(file_dir, 'model_best.pth.tar'))

class Summary(Enum):
    NONE = 0
    AVERAGE = 1
    SUM = 2
    COUNT = 3

class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self, name, fmt=':f', summary_type=Summary.AVERAGE):
        self.name = name
        self.fmt = fmt
        self.summary_type = summary_type
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def all_reduce(self):
        total = torch.FloatTensor([self.sum, self.count]).cuda(self.sum.device)
        dist.all_reduce(total, dist.ReduceOp.SUM, async_op=False)
        self.sum, self.count = total.tolist()
        self.avg = self.sum / self.count

    def __str__(self):
        fmtstr = '{name} {val' + self.fmt + '} ({avg' + self.fmt + '})'
        return fmtstr.format(**self.__dict__)
    
    def summary(self):
        fmtstr = ''
        if self.summary_type is Summary.NONE:
            fmtstr = ''
        elif self.summary_type is Summary.AVERAGE:
            fmtstr = '{name} {avg:.3f}'
        elif self.summary_type is Summary.SUM:
            fmtstr = '{name} {sum:.3f}'
        elif self.summary_type is Summary.COUNT:
            fmtstr = '{name} {count:.3f}'
        else:
            raise ValueError('invalid summary type %r' % self.summary_type)
        
        return fmtstr.format(**self.__dict__)


class ProgressMeter(object):
    def __init__(self, num_batches, meters, prefix=""):
        self.batch_fmtstr = self._get_batch_fmtstr(num_batches)
        self.meters = meters
        self.prefix = prefix

    def display(self, batch):
        entries = [self.prefix + self.batch_fmtstr.format(batch)]
        entries += [str(meter) for meter in self.meters]
        q_log.log('\t'.join(entries))
        
    def display_summary(self):
        entries = [" *"]
        entries += [meter.summary() for meter in self.meters]
        q_log.log(' '.join(entries))

    def _get_batch_fmtstr(self, num_batches):
        num_digits = len(str(num_batches // 1))
        fmt = '{:' + str(num_digits) + 'd}'
        return '[' + fmt + '/' + fmt.format(num_batches) + ']'

def accuracy(output, target, topk=(1,)):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].reshape(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res


if __name__ == '__main__':


    main()