#!/bin/bash

set -euo pipefail
mkdir -p logs

# Data splits and models
DATA_SPLITS=("test200" "test400" "test600" "test800" "test1000" "test1100" "test1200")
API_MODELS=("gemini-2.5-flash" "claude-sonnet-4-20250514" "gpt-5" "o3" "o4-mini" "gpt-4.1" "gpt-4.1-mini" "gpt-4o" "Llama-4-Maverick-17B-128E-Instruct-FP8")
OPEN_MODELS=("Qwen2.5-VL-7B-Instruct" "Qwen2.5-VL-72B-Instruct" "Llama-3.2-11B-Vision-Instruct" "Llama-3.2-90B-Vision-Instruct" "deepseek-vl2")
 
MODELS=("Llama-4-Maverick-17B-128E-Instruct-FP8")


HOLDOUT_RATIO=0.2
SEED=42
TOP_K=3
 
RUN_FEWSHOT=true
RUN_MCQ_CTX_AID=false

is_api_model() {
  local m="$1"
  for a in "${API_MODELS[@]}"; do
    if [[ "$m" == "$a" ]]; then
      return 0
    fi
  done
  return 1
}

# Build global image dirs and gold CSV arrays
IMAGE_DIRS=()
GOLD_CSVS=()
for SPLIT in "${DATA_SPLITS[@]}"; do
  d="benchmark/images/${SPLIT}"
  c="benchmark/annotations/${SPLIT}.csv"
  if [[ -d "$d" ]]; then
    IMAGE_DIRS+=("$d")
  fi
  if [[ -f "$c" ]]; then
    GOLD_CSVS+=("$c")
  fi
done

if [[ ${#IMAGE_DIRS[@]} -eq 0 || ${#GOLD_CSVS[@]} -eq 0 ]]; then
  echo "No valid image dirs or CSVs found" >&2
  exit 1
fi

for MODEL in "${MODELS[@]}"; do
  echo "Model: $MODEL"
  # Decide Azure usage automatically for API models
  USE_AZURE_FLAG=()
  if is_api_model "$MODEL"; then
    USE_AZURE_FLAG=(--use-azure)
  fi

  OUT_DIR="benchmark/results/"
  mkdir -p "$OUT_DIR"

  if [[ "$RUN_FEWSHOT" == true ]]; then
    echo "  - fewshot (global)"
    python src/contextual_aid.py \
      --image-dirs "${IMAGE_DIRS[@]}" \
      --gold-csv "${GOLD_CSVS[@]}" \
      --holdout-ratio "$HOLDOUT_RATIO" \
      --seed "$SEED" \
      --top-k "$TOP_K" \
      --mode fewshot \
      --model-type "$MODEL" \
      --out-dir "$OUT_DIR" \
      "${USE_AZURE_FLAG[@]}"
  fi

  if [[ "$RUN_MCQ_CTX_AID" == true ]]; then
    echo "  - mcq_contextual_aid (global)"
    python src/contextual_aid.py \
      --image-dirs "${IMAGE_DIRS[@]}" \
      --gold-csv "${GOLD_CSVS[@]}" \
      --holdout-ratio "$HOLDOUT_RATIO" \
      --seed "$SEED" \
      --top-k "$TOP_K" \
      --mode mcq_contextual_aid \
      --model-type "$MODEL" \
      --out-dir "$OUT_DIR" \
      "${USE_AZURE_FLAG[@]}"
  fi

  echo "### End $MODEL"
  echo
 done


