
import torch
import json

class DatasetMath(torch.utils.data.Dataset) :
  def __init__(self, path, mode="train", mask=None) :
    self.path = path
    with open(path, "r") as f :
      self.data_raw = json.load(f)
    if mask is not None :
      self.data_raw = [self.data_raw[idx] for idx in mask]
    if mode in ("train", ) :
      self.data_valid = list(filter(lambda x: x["pgm_rate"] > 0 or any(x["ans"]), self.data_raw))
    elif mode in ("test", ) :
      self.data_valid = self.data_raw
    self.N = len(self.data_valid)
    self.programs = [None if data["pgm"] is None else tuple(data["pgm"]) for data in self.data_valid]
    self.programs_raw = [None if data["pgm"] is None else tuple(MathPgmFunc.program_p2r(data["pgm"], pad = 50)) for data in self.data_valid]
    self.programs_rate = [data["pgm_rate"] for data in self.data_valid]
    self.weight = [len(data["idx"])**0.5 * data["pgm_rate"] for data in self.data_valid]

  def __getitem__(self, index) :
    que = self.data_valid[index]["toks"]
    num = self.data_valid[index]["nums"]
    ans = self.data_valid[index]["ans"]
    pgm_raw = self.programs_raw[index]
    weight = self.weight[index]
    return que, num ,ans, pgm_raw, weight

  def __len__(self) :
    return self.N

  @staticmethod
  def collate_fn(data) :
    return list(zip(*data))


class MathPgmFunc() :
  def __init__(self) :
    pass

  @staticmethod
  def action_r2p(a) :
    if a in (0, 1, 2) :
      return None
    elif a in (3, ) :
      return -1
    else :
      return a - 4

  @staticmethod
  def program_r2p(raw, pad = None) :
    program = []
    for a in raw :
      p0 = MathPgmFunc.action_r2p(a)
      if p0 is not None :
        program.append(p0)
    if pad is not None :
      program = program + [-1] * (pad - len(program))
    return tuple(program)

  @staticmethod
  def program_p2r(program, pad = None) :
    raw = [1]
    for a in program :
      if a in (-1, ) :
        raw.append(3)
      else :
        raw.append(a + 4)
    raw += [2]
    if pad is not None :
      raw = raw + [0] * (pad - len(raw))
    return raw

  @staticmethod
  def seq_to_bool(data, word_size) :
    return (torch.zeros(data.shape + (word_size,), device=data.device).scatter_(-1, data.unsqueeze(-1), 1).sum(1) >= 1 ).float()

  @staticmethod
  def program_regularizing(program, cfg) :
    program = list(filter(lambda x:x in cfg.modules_meta_info, program))
    if len(program) > 0 :
      i = 0
      branch_left = 1
      while True :
        branch_left += (cfg.modules_meta_info[program[i]]["shape"][0] - 1)
        i += 1
        if branch_left <= 0 :
          if i > 1 :
            program = program[:i]
          else :
            program = None
          break
        elif i + branch_left >= cfg.max_program_length or i >= len(program) :
          if i + branch_left > cfg.max_program_length :
            program = tuple(list(program[:i - 1]) + [-1] * (branch_left + 1 - cfg.modules_meta_info[program[i - 1]]["shape"][0]))
          else :
            program = tuple(list(program[:i]) + [-1] * branch_left)
          break
    else :
      program = None
    return program


class DatasetSample(torch.utils.data.Dataset) :
  def __init__(self, base = None) :
    self.base = base
    self.samples_set = set()
    self.samples = []

  def __getitem__(self, index) :
    return self.base[self.samples[index]]

  def __len__(self) :
    return len(self.samples)

  def add(self, n) :
    if type(n) in (int, ) :
      self.samples_set.add(n)
    elif type(n) in (list, tuple, set) :
      self.samples_set.update(set(n))
    self.samples = list(self.samples_set)


math23_public_test = [29, 63, 74, 146, 154, 199, 210, 215, 222, 284, 307, 310, 335, 358, 361, 363, 415, 418, 455, 486, 493, 495, 502, 505, 586, 615, 639, 714, 720, 724, 763, 798, 843, 844, 902, 934, 946, 953, 963, 996, 1026, 1049, 1056, 1115, 1156, 1222, 1251, 1283, 1291, 1305, 1313, 1324, 1358, 1360, 1375, 1380, 1386, 1409, 1481, 1488, 1495, 1536, 1557, 1568, 1609, 1650, 1662, 1674, 1696, 1708, 1714, 1733, 1763, 1768, 1777, 1782, 1830, 1837, 1928, 1955, 1987, 1997, 2003, 2006, 2012, 2088, 2105, 2150, 2153, 2181, 2187, 2261, 2275, 2282, 2306, 2338, 2348, 2398, 2419, 2460, 2470, 2476, 2516, 2518, 2529, 2553, 2556, 2572, 2637, 2638, 2651, 2659, 2692, 2695, 2727, 2739, 2748, 2790, 2803, 2808, 2841, 2847, 2938, 2969, 2989, 2994, 3004, 3016, 3026, 3092, 3109, 3127, 3170, 3180, 3182, 3230, 3236, 3266, 3301, 3350, 3356, 3371, 3389, 3401, 3484, 3527, 3552, 3553, 3613, 3614, 3622, 3672, 3681, 3697, 3703, 3728, 3787, 3801, 3852, 3873, 3899, 3903, 3925, 3940, 3946, 3952, 3981, 3986, 3988, 4022, 4054, 4079, 4094, 4113, 4118, 4140, 4166, 4201, 4205, 4206, 4219, 4234, 4249, 4304, 4316, 4351, 4360, 4376, 4395, 4454, 4515, 4523, 4526, 4535, 4546, 4586, 4593, 4596, 4604, 4727, 4745, 4763, 4777, 4795, 4891, 4928, 4953, 5003, 5031, 5041, 5046, 5057, 5075, 5124, 5128, 5131, 5155, 5174, 5188, 5197, 5225, 5245, 5270, 5293, 5338, 5352, 5356, 5399, 5402, 5410, 5418, 5425, 5430, 5431, 5434, 5435, 5444, 5459, 5493, 5535, 5601, 5622, 5629, 5634, 5672, 5735, 5781, 5805, 5811, 5815, 5824, 5836, 5863, 5887, 5889, 5892, 5958, 6005, 6027, 6047, 6072, 6086, 6092, 6154, 6182, 6190, 6211, 6218, 6242, 6258, 6267, 6276, 6300, 6302, 6398, 6425, 6441, 6454, 6461, 6473, 6480, 6486, 6531, 6534, 6545, 6570, 6603, 6616, 6654, 6655, 6687, 6694, 6707, 6715, 6737, 6741, 6750, 6785, 6793, 6833, 6859, 6948, 7033, 7042, 7067, 7088, 7090, 7092, 7093, 7127, 7146, 7155, 7191, 7196, 7199, 7225, 7231, 7262, 7286, 7305, 7314, 7315, 7331, 7341, 7348, 7360, 7387, 7401, 7443, 7466, 7538, 7556, 7659, 7682, 7691, 7693, 7715, 7722, 7736, 7760, 7798, 7812, 7814, 7834, 7851, 7873, 7880, 7888, 7970, 7983, 8032, 8055, 8067, 8073, 8090, 8096, 8110, 8121, 8130, 8135, 8145, 8166, 8172, 8173, 8195, 8196, 8276, 8334, 8377, 8397, 8400, 8412, 8455, 8456, 8466, 8478, 8528, 8552, 8653, 8677, 8730, 8741, 8748, 8752, 8777, 8783, 8786, 8789, 8791, 8820, 8860, 9052, 9055, 9070, 9071, 9080, 9086, 9145, 9156, 9160, 9258, 9280, 9285, 9289, 9296, 9390, 9399, 9405, 9457, 9458, 9468, 9483, 9549, 9583, 9591, 9604, 9618, 9620, 9713, 9718, 9750, 9757, 9763, 9787, 9816, 9821, 9853, 9890, 9897, 9899, 9987, 10003, 10009, 10022, 10044, 10072, 10104, 10174, 10180, 10235, 10330, 10342, 10364, 10368, 10371, 10438, 10464, 10523, 10536, 10567, 10587, 10606, 10628, 10675, 10688, 10689, 10758, 10789, 10803, 10892, 10897, 11033, 11042, 11046, 11047, 11118, 11151, 11229, 11234, 11250, 11267, 11301, 11316, 11346, 11377, 11417, 11420, 11430, 11432, 11501, 11502, 11510, 11520, 11523, 11542, 11584, 11611, 11618, 11625, 11627, 11651, 11654, 11660, 11724, 11735, 11736, 11742, 11749, 11804, 11834, 11847, 11859, 11861, 11919, 11979, 12024, 12073, 12081, 12084, 12091, 12127, 12138, 12197, 12203, 12236, 12241, 12251, 12257, 12356, 12363, 12390, 12400, 12402, 12420, 12424, 12466, 12524, 12536, 12549, 12553, 12633, 12656, 12698, 12745, 12756, 12778, 12782, 12791, 12812, 12849, 12977, 13009, 13040, 13062, 13212, 13230, 13345, 13354, 13390, 13391, 13410, 13433, 13438, 13477, 13503, 13515, 13536, 13539, 13592, 13613, 13652, 13694, 13743, 13751, 13755, 13764, 13796, 13813, 13846, 13851, 13867, 13917, 13929, 13931, 13960, 13978, 14010, 14070, 14077, 14166, 14204, 14207, 14231, 14258, 14295, 14304, 14319, 14399, 14407, 14435, 14463, 14592, 14605, 14665, 14707, 14727, 14738, 14745, 14791, 14804, 14814, 14860, 14889, 14903, 14915, 14922, 14933, 14939, 14951, 14954, 14977, 14980, 14999, 15036, 15066, 15079, 15097, 15129, 15160, 15171, 15172, 15191, 15194, 15209, 15215, 15232, 15249, 15254, 15279, 15310, 15316, 15323, 15326, 15356, 15374, 15423, 15447, 15475, 15478, 15480, 15483, 15487, 15497, 15503, 15533, 15554, 15569, 15581, 15591, 15597, 15632, 15656, 15688, 15744, 15772, 15776, 15807, 15840, 15852, 15882, 15959, 15972, 15996, 16000, 16005, 16028, 16037, 16057, 16081, 16102, 16118, 16146, 16170, 16204, 16290, 16358, 16363, 16375, 16402, 16505, 16519, 16563, 16567, 16573, 16584, 16592, 16608, 16629, 16634, 16660, 16671, 16673, 16688, 16821, 16840, 16850, 16901, 16908, 16917, 16950, 16959, 16969, 16985, 17010, 17025, 17060, 17082, 17107, 17110, 17111, 17118, 17122, 17135, 17190, 17217, 17240, 17261, 17297, 17319, 17334, 17339, 17361, 17415, 17416, 17423, 17433, 17454, 17487, 17550, 17557, 17597, 17602, 17635, 17685, 17702, 17703, 17744, 17757, 17781, 17820, 17821, 17847, 17865, 17914, 17921, 17943, 17994, 18015, 18019, 18027, 18045, 18053, 18089, 18115, 18133, 18135, 18148, 18153, 18168, 18203, 18232, 18244, 18295, 18299, 18310, 18320, 18334, 18338, 18363, 18371, 18400, 18422, 18441, 18448, 18452, 18474, 18489, 18505, 18509, 18546, 18622, 18630, 18702, 18711, 18729, 18739, 18740, 18756, 18810, 18828, 18829, 18831, 18852, 18903, 18906, 18915, 18938, 18989, 19084, 19115, 19185, 19206, 19231, 19268, 19279, 19299, 19305, 19332, 19333, 19336, 19361, 19409, 19423, 19428, 19431, 19525, 19544, 19547, 19552, 19576, 19582, 19615, 19618, 19628, 19644, 19652, 19655, 19668, 19673, 19702, 19739, 19827, 19845, 19892, 19911, 20020, 20039, 20045, 20067, 20112, 20141, 20144, 20160, 20163, 20174, 20183, 20231, 20244, 20279, 20318, 20329, 20344, 20390, 20445, 20449, 20452, 20457, 20462, 20473, 20475, 20483, 20503, 20531, 20544, 20570, 20584, 20593, 20698, 20705, 20736, 20766, 20815, 20817, 20829, 20832, 20929, 20932, 21028, 21040, 21055, 21115, 21151, 21155, 21197, 21215, 21222, 21226, 21229, 21257, 21266, 21278, 21282, 21287, 21330, 21334, 21341, 21342, 21353, 21420, 21425, 21426, 21447, 21448, 21468, 21469, 21476, 21482, 21520, 21552, 21558, 21562, 21570, 21598, 21614, 21677, 21698, 21700, 21717, 21722, 21728, 21741, 21755, 21760, 21763, 21764, 21777, 21800, 21804, 21813, 21834, 21835, 21841, 21846, 21856, 21870, 21910, 21918, 21973, 21993, 21994, 22073, 22090, 22101, 22139, 22145, 22154, 22158, 22178, 22256, 22259, 22277, 22282, 22299, 22300, 22322, 22324, 22329, 22361, 22368, 22375, 22420, 22430, 22450, 22464, 22518, 22526, 22654, 22660, 22670, 22675, 22703, 22716, 22722, 22745, 22759, 22767, 22780, 22785, 22786, 22807, 22825, 22832, 22840, 22846, 22852, 22915, 22929, 22947, 22970, 22975, 23009, 23033, 23053, 23054, 23064, 23070, 23074, 23147, 23150, 23155]
