# copied from https://github.com/kuangliu/pytorch-cifar/blob/master/models/resnet.py

'''ResNet in PyTorch.
For Pre-activation ResNet, see 'preact_resnet.py'.
Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
'''
import torch
import torch.nn as nn
import torch.nn.functional as F


class BasicBlock_BN(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, affine=True):
        super(BasicBlock_BN, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes, affine=affine)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes, affine=affine)
        # flags if skip connections require conv
        if stride != 1 or in_planes != self.expansion*planes:
            self.skip_require_conv = True
        else:
            self.skip_require_conv = False
        self.shortcut = nn.Sequential()
        self.conv10 = nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False)
        self.bn10 = nn.BatchNorm2d(self.expansion*planes, affine=affine)
        
    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        if self.skip_require_conv:
            out += self.bn10(self.conv10(x))
        else:
            out += self.shortcut(x)
        out = F.relu(out)
        return out


class BasicBlock_groupnorm(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, group_number, stride=1, affine=True):
        super(BasicBlock_groupnorm, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.gn1 = nn.GroupNorm(group_number, planes, affine=affine)
        self.gn2 = nn.GroupNorm(group_number, planes, affine=affine)
        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                        kernel_size=1, stride=stride, bias=False),
                )

    def forward(self, x):
        out = F.relu(self.gn1(self.conv1(x)))
        out = self.gn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out

class BasicBlock_groupnorm_after(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, group_number, stride=1, affine=True):
        super(BasicBlock_groupnorm_after, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.gn1 = nn.GroupNorm(group_number, planes, affine=affine)
        self.gn2 = nn.GroupNorm(group_number, planes, affine=affine)
        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                        kernel_size=1, stride=stride, bias=False),
                )

    def forward(self, x):
        out = self.gn1(F.relu(self.conv1(x)))
        out = self.conv2(out)
        out += self.shortcut(x)
        out = self.gn2(F.relu(out))
        return out


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
            )

    def forward(self, x):
        out = F.relu(self.conv1(x))
        out = self.conv2(out)
        out += self.shortcut(x)
        out = F.relu(out)
        return out



class ResNet(nn.Module):
    def __init__(self, norm_method, group_number, block, num_blocks, num_classes=10, affine=True, return_feature=False):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.group_number=group_number
        self.affine = affine
        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.norm_method = norm_method
        print('norm method:', self.norm_method, 'affine:', affine)
        if norm_method == 'batch_norm':
            self.bn1 = nn.BatchNorm2d(64, affine)
        elif 'group_norm' in norm_method:
            self.gn1 =  nn.GroupNorm(group_number, 64, affine)

        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        if num_classes == 200:
            feature_dim = 4 # To adapt the model for TinyImageNet
        elif num_classes==7:
            feature_dim = 49 # To adapt the model for PACS dataset
        else:
            feature_dim = 1 # To adapt the model for CIFAR datasets
        self.w = nn.Linear(feature_dim*512*block.expansion, num_classes)
        self.return_feature = return_feature
        print('return feature:', return_feature)
        
    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks - 1)
        layers = []
        for stride in strides:
            if 'group_norm' in self.norm_method:
                layers.append(block(self.in_planes, planes, self.group_number, stride, self.affine))
            elif 'batch_norm' in self.norm_method:
                layers.append(block(self.in_planes, planes, stride, self.affine))
            else:
                layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        if self.norm_method == 'batch_norm':
            out = F.relu(self.bn1(self.conv1(x)))
        elif self.norm_method == 'group_norm':
            out = F.relu(self.gn1(self.conv1(x)))
        elif self.norm_method == 'group_norm_after':
            out = self.gn1(F.relu(self.conv1(x)))
        else:
            out = F.relu(self.conv1(x))

        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.reshape(out.size(0), -1)
        if self.norm_method == 'fn':
            out = F.normalize(out, p=2, dim=1) * torch.sqrt(torch.tensor(out.shape[1]))
        feature = out
        out = self.w(out)
        if self.return_feature:
            return out, feature
        else:
            return out 

def resnet18(num_classes, norm_method, affine, group_number=1, **kwargs):
    if norm_method == 'no_norm' or norm_method == 'fn':
        return ResNet(norm_method, group_number=0, block=BasicBlock, num_blocks=[2, 2, 2, 2], num_classes=num_classes, **kwargs)
    elif norm_method == 'group_norm':
        return ResNet(norm_method, group_number=group_number, block=BasicBlock_groupnorm, num_blocks=[2, 2, 2, 2], num_classes=num_classes, affine=affine)
    elif norm_method == 'group_norm_after':
        return ResNet(norm_method, group_number=group_number, block=BasicBlock_groupnorm_after, num_blocks=[2, 2, 2, 2], num_classes=num_classes, affine=affine)
    elif norm_method == 'batch_norm':
        return ResNet(norm_method, group_number=0, block=BasicBlock_BN, num_blocks=[2, 2, 2, 2], num_classes=num_classes, affine=affine)
    else:
        raise NotImplementedError


if __name__=='__main__':
    pass



