import logging, random, numpy, torch, json

def seed_everything(seed):
    random.seed(seed)
    numpy.random.seed(seed)
    torch.manual_seed(seed)

def get_learning_rate(optimizer):
    for param_group in optimizer.param_groups:
        return param_group["lr"]

def chain_callbacks(*callbacks):
    class ChainCallback:
        def __init__(self, callbacks): self.callbacks = callbacks
        def step (self, local): 
            for callback in self.callbacks: callback.step (local)
        def epoch(self, local): 
            for callback in self.callbacks: callback.epoch(local)
    return ChainCallback(callbacks)

def get_file_logger(path):
    logger = logging.getLogger(path)
    logger.setLevel(logging.INFO)

    handler = logging.FileHandler(path)
    handler.setLevel(logging.INFO)

    formatter = logging.Formatter(json.dumps(json.loads("""
        {
            "data"    : "%(asctime)s", 
            "level"   : "%(levelname)s", 
            "process" : { 
                "id"   : "%(process)d", 
                "name" : "%(processName)s"
            }, 
            "thread"  : {
                "id"   : "%(thread)d", 
                "name" : "%(threadName)s"
            }, 
            "message" : "MESSAGE" 
        }
    """), indent=None, separators=(",",":")).replace("\"MESSAGE\"","%(message)s"))
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    return logger

def chain_options(*decs):
    def deco(f):
        for dec in reversed(decs): f = dec(f)
        return f
    return deco
