import numpy as np
import pickle
import torch
import pandas as pd
import sys
import os
sys.path.insert(1, os.path.join(sys.path[0], os.pardir, os.pardir))
from utils import NonlinearNonconvexProblem

LINEAR_CONSTRAINTS = False
CONVEX_OBJECTIVE = False

os.environ['KMP_DUPLICATE_LIB_OK']='True'
torch.set_default_dtype(torch.float64)

# parameters
num_var = 200
num_eq = 150
num_ex = 10000
np.random.seed(17)

# generate data
Q = np.diag(np.random.rand(num_var))
p = np.random.rand(num_var) * 0.0 if CONVEX_OBJECTIVE else np.random.rand(num_var)
Alist, clist, dlist, elist = [], [], [], []
for _ in range(num_eq):
    A = np.random.randn(num_var, num_var)
    A = 0.5*(A + A.T)*0.0 if LINEAR_CONSTRAINTS else 0.5*(A + A.T)
    Alist.append(A)
    clist.append(np.random.randn(num_var))
    dlist.append(np.random.randn())
# X holds per-sample RHS for each of the num_eq constraints
lb = -5
ub = 5
theta = np.random.uniform(lb,ub,size=(num_ex, num_eq))

# solve and save
if num_var <= 401:
    problem = NonlinearNonconvexProblem(Q, p, Alist, clist, dlist, theta)
    problem.calc_Y()
    print("Generated", len(problem._Y), "solutions")
    objective_type = "" if CONVEX_OBJECTIVE else "non"
    constraints_type = "" if LINEAR_CONSTRAINTS else "non"
    filename = f"random_{constraints_type}linear{objective_type}convex_dataset_dim{num_var}_eq{num_eq}_ex{num_ex}_int{lb}_{ub}.pkl"
    with open(f"data/nonlinear/{filename}", 'wb') as f:
        pickle.dump(problem, f)
    print("Saved to", filename)

    # === Extract X and Y ===
    X = problem._X
    Y = problem._Y

    # === Create a table of the first few dimensions ===

    df = pd.DataFrame(X, columns=[f"X[{i}]" for i in range(num_eq)])
    df = pd.concat([df, pd.DataFrame(Y, columns=[f"Y[{i}]" for i in range(num_var)])], axis=1)

    # Save the dataset in csv format
    X_df = pd.DataFrame(X, columns=[f"X[{i}]" for i in range(X.shape[1])])
    Y_df = pd.DataFrame(Y, columns=[f"Y[{i}]" for i in range(Y.shape[1])])
    X_df.to_csv(f"data/nonlinear/X_data_{filename.split('.')[0]}.csv", index=False)
    Y_df.to_csv(f"data/nonlinear/Y_data_{filename.split('.')[0]}.csv", index=False)
    print("Saved X and Y data to csv files")

    parameters = {
        "A": problem._Alist,
        "c": problem._c,
        "d": problem._d,
        "Q": problem._Q,
        "p": problem._p,
    }
    with open(f"data/nonlinear/parameters_{filename}", 'wb') as f:
        pickle.dump(parameters, f)
    print("Saved problem parameters to pickle file")

else:
    # Save just the generated data without solving the problem
    X = theta
    objective_type = "" if CONVEX_OBJECTIVE else "non"
    constraints_type = "" if LINEAR_CONSTRAINTS else "non"
    filename = f"random_{constraints_type}linear{objective_type}convex_dataset_dim{num_var}_eq{num_eq}_ex{num_ex}_int{lb}_{ub}"
    
    
    # Save X data to csv
    X_df = pd.DataFrame(X, columns=[f"X[{i}]" for i in range(X.shape[1])])
    X_df.to_csv(f"data/nonlinear/X_data_{filename}.csv", index=False)
    print("Saved X data to csv file")

    # Save fake Y data to csv (all zeros, with correct dimension)
    Y = np.zeros((num_ex, num_var))
    Y_df = pd.DataFrame(Y, columns=[f"Y[{i}]" for i in range(Y.shape[1])])
    Y_df.to_csv(f"data/nonlinear/Y_data_{filename}.csv", index=False)
    print("Saved Y data to csv file")

    Q = torch.tensor(Q, dtype=torch.float64)
    p = torch.tensor(p, dtype=torch.float64)
    Alist = [torch.tensor(A, dtype=torch.float64) for A in Alist]
    clist = [torch.tensor(ci, dtype=torch.float64) for ci in clist]
    dlist = [torch.tensor(di, dtype=torch.float64) for di in dlist]
    
    # Save parameters
    parameters = {
        "A": Alist,
        "c": clist,
        "d": dlist,
        "Q": Q,
        "p": p,
    }
    with open(f"data/nonlinear/parameters_{filename}.pkl", 'wb') as f:
        pickle.dump(parameters, f)
    print("Saved problem parameters to pickle file")
