import torch
import torch.nn as nn
from torch.nn import functional as F
import math
import numpy as np
from torch.distributions import Categorical
from algorithms.utils.util import check, init, Multi_Agent_Distribute
from algorithms.utils.transformer_act import discrete_autoregreesive_act
from algorithms.utils.transformer_act import discrete_parallel_act
from algorithms.utils.transformer_act import continuous_autoregreesive_act
from algorithms.utils.transformer_act import continuous_parallel_act
from algorithms.utils.transformer_act import continuous_autoregreesive_distb_gail
from algorithms.utils.transformer_act import continuous_parallel_act_distb_gail


def init_(m, gain=0.01, activate=False):
    if activate:
        gain = nn.init.calculate_gain('relu')
    return init(m, nn.init.orthogonal_, lambda x: nn.init.constant_(x, 0), gain=gain)


def init_disc(module):
    if isinstance(module, nn.Linear):
        nn.init.xavier_normal_(module.weight)
        nn.init.zeros_(module.bias)


class SelfAttention(nn.Module):

    def __init__(self, n_embd, n_head, n_agent, masked=False):
        super(SelfAttention, self).__init__()

        assert n_embd % n_head == 0
        self.masked = masked
        self.n_head = n_head
        # key, query, value projections for all heads
        self.key = init_(nn.Linear(n_embd, n_embd))
        self.query = init_(nn.Linear(n_embd, n_embd))
        self.value = init_(nn.Linear(n_embd, n_embd))
        # output projection
        self.proj = init_(nn.Linear(n_embd, n_embd))
        # if self.masked:
        # causal mask to ensure that attention is only applied to the left in the input sequence
        self.register_buffer("mask", torch.tril(torch.ones(n_agent + 1, n_agent + 1))
                             .view(1, 1, n_agent + 1, n_agent + 1))

        self.att_bp = None

    def forward(self, key, value, query):
        B, L, D = query.size()

        # calculate query, key, values for all heads in batch and move head forward to be the batch dim
        k = self.key(key).view(B, L, self.n_head, D // self.n_head).transpose(1, 2)  # (B, nh, L, hs)
        q = self.query(query).view(B, L, self.n_head, D // self.n_head).transpose(1, 2)  # (B, nh, L, hs)
        v = self.value(value).view(B, L, self.n_head, D // self.n_head).transpose(1, 2)  # (B, nh, L, hs)

        # causal attention: (B, nh, L, hs) x (B, nh, hs, L) -> (B, nh, L, L)
        att = (q @ k.transpose(-2, -1)) * (1.0 / math.sqrt(k.size(-1)))

        # self.att_bp = F.softmax(att, dim=-1)

        if self.masked:
            att = att.masked_fill(self.mask[:, :, :L, :L] == 0, float('-inf'))
        att = F.softmax(att, dim=-1)

        y = att @ v  # (B, nh, L, L) x (B, nh, L, hs) -> (B, nh, L, hs)
        y = y.transpose(1, 2).contiguous().view(B, L, D)  # re-assemble all head outputs side by side

        # output projection
        y = self.proj(y)
        return y


class EncodeBlock(nn.Module):
    """ an unassuming Transformer block """

    def __init__(self, n_embd, n_head, n_agent):
        super(EncodeBlock, self).__init__()

        self.ln1 = nn.LayerNorm(n_embd)
        self.ln2 = nn.LayerNorm(n_embd)
        # self.attn = SelfAttention(n_embd, n_head, n_agent, masked=True)
        self.attn = SelfAttention(n_embd, n_head, n_agent, masked=False)
        self.mlp = nn.Sequential(
            init_(nn.Linear(n_embd, 1 * n_embd), activate=True),
            nn.GELU(),
            init_(nn.Linear(1 * n_embd, n_embd))
        )

    def forward(self, x):
        x = self.ln1(x + self.attn(x, x, x))
        x = self.ln2(x + self.mlp(x))
        return x


class DecodeBlock(nn.Module):
    """ an unassuming Transformer block """

    def __init__(self, n_embd, n_head, n_agent):
        super(DecodeBlock, self).__init__()

        self.ln1 = nn.LayerNorm(n_embd)
        self.ln2 = nn.LayerNorm(n_embd)
        self.ln3 = nn.LayerNorm(n_embd)
        self.attn1 = SelfAttention(n_embd, n_head, n_agent, masked=True)
        self.attn2 = SelfAttention(n_embd, n_head, n_agent, masked=True)
        self.mlp = nn.Sequential(
            init_(nn.Linear(n_embd, 1 * n_embd), activate=True),
            nn.GELU(),
            init_(nn.Linear(1 * n_embd, n_embd))
        )

    def forward(self, x, rep_enc):
        x = self.ln1(x + self.attn1(x, x, x))
        x = self.ln2(rep_enc + self.attn2(key=x, value=x, query=rep_enc))
        x = self.ln3(x + self.mlp(x))
        return x


class Encoder(nn.Module):

    def __init__(self, state_dim, obs_dim, n_block, n_embd, n_head, n_agent, encode_state):
        super(Encoder, self).__init__()

        self.state_dim = state_dim
        self.obs_dim = obs_dim
        self.n_embd = n_embd
        self.n_agent = n_agent
        self.encode_state = encode_state
        # self.agent_id_emb = nn.Parameter(torch.zeros(1, n_agent, n_embd))
        self.state_encoder = nn.Sequential(
            nn.LayerNorm(state_dim), init_(nn.Linear(state_dim, n_embd), activate=True), nn.GELU())
        self.obs_encoder = nn.Sequential(
            nn.LayerNorm(obs_dim), init_(nn.Linear(obs_dim, n_embd), activate=True), nn.GELU())
        self.ln = nn.LayerNorm(n_embd)
        self.blocks = nn.Sequential(*[EncodeBlock(n_embd, n_head, n_agent) for _ in range(n_block)])
        # delete critic head to train off line
        # self.head = nn.Sequential(init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
        #                           init_(nn.Linear(n_embd, 1)))

    def forward(self, state, obs):
        # state: (batch, n_agent, state_dim)
        # obs: (batch, n_agent, obs_dim)
        if self.encode_state:
            state_embeddings = self.state_encoder(state)
            x = state_embeddings
        else:
            obs_embeddings = self.obs_encoder(obs)
            x = obs_embeddings

        rep = self.blocks(self.ln(x))
        # v_loc = self.head(rep)

        return rep


class Decoder(nn.Module):

    def __init__(self, obs_dim, action_dim, n_block, n_embd, n_head, n_agent,
                 action_type='Discrete', dec_actor=False, share_actor=False):
        super(Decoder, self).__init__()

        self.action_dim = action_dim
        self.n_embd = n_embd
        self.n_agent = n_agent
        self.dec_actor = dec_actor
        self.share_actor = share_actor
        self.action_type = action_type

        if action_type != 'Discrete':
            log_std = torch.ones(action_dim)
            # log_std = torch.zeros(action_dim)
            self.log_std = torch.nn.Parameter(log_std)
            # self.log_std = torch.nn.Parameter(torch.zeros(action_dim))

        if self.dec_actor:
            if self.share_actor:
                print("mac_dec!!!!!")
                self.mlp = nn.Sequential(nn.LayerNorm(obs_dim),
                                         init_(nn.Linear(obs_dim, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
                                         init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
                                         init_(nn.Linear(n_embd, action_dim)))
            else:
                self.mlp = nn.ModuleList()
                for n in range(n_agent):
                    actor = nn.Sequential(nn.LayerNorm(obs_dim),
                                          init_(nn.Linear(obs_dim, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
                                          init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
                                          init_(nn.Linear(n_embd, action_dim)))
                    self.mlp.append(actor)
        else:
            # self.agent_id_emb = nn.Parameter(torch.zeros(1, n_agent, n_embd))
            if action_type == 'Discrete':
                self.action_encoder = nn.Sequential(init_(nn.Linear(action_dim + 1, n_embd, bias=False), activate=True), nn.GELU())
            else:
                self.action_encoder = nn.Sequential(init_(nn.Linear(action_dim, n_embd), activate=True), nn.GELU())
            self.obs_encoder = nn.Sequential(
                nn.LayerNorm(obs_dim), init_(nn.Linear(obs_dim, n_embd), activate=True), nn.GELU())
            self.ln = nn.LayerNorm(n_embd)
            self.blocks = nn.Sequential(*[DecodeBlock(n_embd, n_head, n_agent) for _ in range(n_block)])
            self.head = nn.Sequential(init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
                                      init_(nn.Linear(n_embd, action_dim)))

    def zero_std(self, device):
        if self.action_type != 'Discrete':
            log_std = torch.zeros(self.action_dim).to(device)
            self.log_std.data = log_std

    # state, action, and return
    def forward(self, action, obs_rep, obs):
        # action: (batch, n_agent, action_dim), one-hot/logits?
        # obs_rep: (batch, n_agent, n_embd)
        if self.dec_actor:
            if self.share_actor:
                logit = self.mlp(obs)
            else:
                logit = []
                for n in range(len(self.mlp)):
                    logit_n = self.mlp[n](obs[:, n, :])
                    logit.append(logit_n)
                logit = torch.stack(logit, dim=1)
        else:
            action_embeddings = self.action_encoder(action)
            x = self.ln(action_embeddings)
            for block in self.blocks:
                x = block(x, obs_rep)
            logit = self.head(x)

        return logit


class DiscObsEncoder(nn.Module):

    def __init__(self, state_dim, obs_dim, n_block, n_embd, n_head, n_agent, encode_state, use_first_token=False):
        super(DiscObsEncoder, self).__init__()

        self.state_dim = state_dim
        self.obs_dim = obs_dim
        self.n_embd = n_embd
        self.n_agent = n_agent
        self.encode_state = encode_state
        self.use_first_token = use_first_token
        # self.agent_id_emb = nn.Parameter(torch.zeros(1, n_agent, n_embd))

        self.state_encoder = nn.Sequential(nn.LayerNorm(state_dim),
                                           init_(nn.Linear(state_dim, n_embd), activate=True), nn.GELU())
        self.obs_encoder = nn.Sequential(nn.LayerNorm(obs_dim),
                                         init_(nn.Linear(obs_dim, n_embd), activate=True), nn.GELU())
        # end token for obs if calculate last loss
        # obs_end_token: (1, 1, n_embd)
        self.obs_start_token = nn.Parameter(
            torch.zeros(n_embd, dtype=torch.float32).unsqueeze(0).unsqueeze(0),
            requires_grad=True,
        ) if self.use_first_token else None

        # if use_first_token, add another position for start of token
        self.ln = nn.LayerNorm(n_embd)
        self.blocks = nn.Sequential(*[EncodeBlock(
            n_embd, n_head, n_agent if not self.use_first_token else n_agent + 1
        ) for _ in range(n_block)])
        # delete critic head to train off line
        # self.head = nn.Sequential(init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
        #                           init_(nn.Linear(n_embd, 1)))

    def forward(self, obs):
        # state: (batch, n_agent, state_dim)
        # obs: (batch, n_agent, obs_dim)
        obs_embeddings = self.obs_encoder(obs)
        x = obs_embeddings

        # obs_end_token: (1, 1, n_embd) --> (batch, 1, n_embd)
        # x: (batch, n_agent, n_embd)
        if self.use_first_token:
            x = torch.cat([
                self.obs_start_token.repeat(x.shape[0], 1, 1), x
            ], dim=1)
        rep = self.blocks(self.ln(x))
        if self.use_first_token:
            rep = rep[:, 0: 1, ]

        return rep


class DiscActEncoder(nn.Module):

    def __init__(self, act_dim, n_block, n_embd, n_head, n_agent, action_type,  use_act_prob=False, use_first_token=False):
        super(DiscActEncoder, self).__init__()
        self.act_dim = act_dim
        self.n_embd = n_embd
        self.n_agent = n_agent
        self.use_act_prob = use_act_prob
        self.use_first_token = use_first_token
        # self.agent_id_emb = nn.Parameter(torch.zeros(1, n_agent, n_embd))
        if action_type == 'Discrete' and not self.use_act_prob:
            self.act_encoder = nn.Sequential(init_(nn.Linear(act_dim, n_embd, bias=False), activate=True), nn.GELU())
        else:
            self.act_encoder = nn.Sequential(init_(nn.Linear(act_dim, n_embd), activate=True), nn.GELU())
        # end token for obs if calculate last loss
        # obs_end_token: (1, 1, n_embd)
        self.act_start_token = nn.Parameter(
            torch.zeros(n_embd, dtype=torch.float32).unsqueeze(0).unsqueeze(0),
            requires_grad=True,
        ) if self.use_first_token else None

        # if use_first_token, add another position for start of token
        self.ln = nn.LayerNorm(n_embd)
        self.blocks = nn.Sequential(*[EncodeBlock(
            n_embd, n_head, n_agent if not self.use_first_token else n_agent + 1
        ) for _ in range(n_block)])
        # delete critic head to train off line
        # self.head = nn.Sequential(init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd),
        #                           init_(nn.Linear(n_embd, 1)))

    def forward(self, act):
        # act: (batch, n_agent, act_dim)
        action_embeddings = self.act_encoder(act)
        x = action_embeddings

        # obs_end_token: (1, 1, n_embd) --> (batch, 1, n_embd)
        # x: (batch, n_agent, n_embd)
        if self.use_first_token:
            x = torch.cat([
                self.act_start_token.repeat(x.shape[0], 1, 1), x
            ], dim=1)
        rep = self.blocks(self.ln(x))
        if self.use_first_token:
            rep = rep[:, 0: 1, ]

        return rep


class Critic(nn.Module):
    def __init__(self, state_dim, obs_dim, n_embd):
        super(Critic, self).__init__()
        self.state_dim = state_dim
        self.obs_dim = obs_dim
        self.n_embd = n_embd
        self.net = nn.Sequential(
            init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(),
            init_(nn.Linear(n_embd, n_embd), activate=True), nn.GELU(),
            init_(nn.Linear(n_embd, 1)),
        )

    # critic network should get state that processed by encoder
    def forward(self, rep):
        return self.net(rep)


class Discriminator(nn.Module):
    def __init__(self, state_dim, obs_dim, action_dim, n_block, n_embd, n_head, n_agent,
                 action_type, use_act_prob=False, agent_independent=False,
                 disc_mlp_obs_encoder=False, disc_mlp_act_encoder=False,
                 disc_mlp_use_first_token=False):
        super().__init__()
        self.state_dim = state_dim
        self.obs_dim = obs_dim
        self.action_dim = action_dim
        self.n_embd = n_embd
        self.n_agent = n_agent
        self.action_type = action_type
        self.use_act_prob = use_act_prob
        self.agent_independent = agent_independent
        self.disc_mlp_obs_encoder = disc_mlp_obs_encoder
        self.disc_mlp_act_encoder = disc_mlp_act_encoder
        self.disc_mlp_use_first_token = disc_mlp_use_first_token
        # use union state and union action as disc mlp input
        self.net_in_dim = n_agent * (n_embd + n_embd) if not self.agent_independent and not self.disc_mlp_use_first_token else n_embd + n_embd
        self.mlp = nn.Sequential(
            nn.Linear(self.net_in_dim, n_embd),
            nn.Tanh(),
            nn.Linear(n_embd, n_embd),
            nn.Tanh(),
            nn.Linear(n_embd, 1),
        )
        self.apply(init_disc)

        # obs encoder for mlp disc
        self.obs_encoder = nn.Sequential(nn.LayerNorm(obs_dim), init_(nn.Linear(obs_dim, n_embd), activate=True), nn.GELU()) \
            if not self.disc_mlp_obs_encoder else DiscObsEncoder(
            state_dim, obs_dim, n_block, n_embd, n_head, n_agent,
            encode_state=False, use_first_token=disc_mlp_use_first_token
        )

        # action encoder for mlp disc
        if self.disc_mlp_act_encoder:
            self.action_encoder = DiscActEncoder(
                action_dim, n_block, n_embd, n_head, n_agent, action_type,
                use_act_prob=use_act_prob, use_first_token=disc_mlp_use_first_token
            )
        elif action_type == 'Discrete' and not self.use_act_prob:
            self.action_encoder = nn.Sequential(init_(nn.Linear(action_dim, n_embd, bias=False), activate=True), nn.GELU(), nn.LayerNorm(n_embd))
        else:
            self.action_encoder = nn.Sequential(init_(nn.Linear(action_dim, n_embd), activate=True), nn.GELU(), nn.LayerNorm(n_embd))

    def forward(self, states, obs, actions):
        # mlp disc not use encoder to process obs
        obs = self.obs_encoder(obs)
        # change action to onehot(if not pre-onehot before forward) before feed into action encoder
        if self.action_type == 'Discrete' and actions.shape[-1] == 1 and not self.use_act_prob:
            actions = F.one_hot(actions.squeeze(-1).long(), num_classes=self.action_dim).float()
        # act encoder struct has ln already
        actions = self.action_encoder(actions)
        # cat all agent obs and action together if use center disc
        if not self.agent_independent and not self.disc_mlp_use_first_token:
            obs = obs.reshape(obs.shape[0], -1).unsqueeze(1)
            actions = actions.reshape(actions.shape[0], -1).unsqueeze(1)

        return self.mlp(torch.cat([obs, actions], dim=-1)).repeat(1, self.n_agent, 1) \
            if not self.agent_independent else self.mlp(torch.cat([obs, actions], dim=-1))

    def get_rewards(self, states, obs, actions):
        return -F.logsigmoid(self.forward(states, obs, actions))

    def get_rewards_from_logits(self, logits):
        """
        logits: torch.Size([batch, num_agents, 1])
        rewards: (batch * num_agents, )
        """
        disc_value = -F.logsigmoid(logits)

        return disc_value.detach().cpu().numpy()  # .reshape(-1)


class MultiAgentTransformerGailMLP(nn.Module):

    def __init__(self, state_dim, obs_dim, action_dim, n_agent,
                 n_block, n_embd, n_head, disc_inner_dim, encode_state=False, device=torch.device("cpu"),
                 action_type='Discrete', dec_actor=False, share_actor=False, use_gail=False,
                 disc_use_act_prob=False, disc_agent_independent=False,
                 disc_mlp_obs_encoder=False, disc_mlp_act_encoder=False,
                 disc_mlp_use_first_token=False):
        super(MultiAgentTransformerGailMLP, self).__init__()

        self.n_agent = n_agent
        self.action_dim = action_dim
        self.tpdv = dict(dtype=torch.float32, device=device)
        self.action_type = action_type
        self.device = device

        # if use gail to train model(default to False)
        self.use_gail = use_gail

        # state unused
        state_dim = 37

        self.encoder = Encoder(state_dim, obs_dim, n_block, n_embd, n_head, n_agent, encode_state)
        self.decoder = Decoder(obs_dim, action_dim, n_block, n_embd, n_head, n_agent, self.action_type,
                               dec_actor=dec_actor, share_actor=share_actor)
        # critic network is optional according to train algorithm
        self.critic = Critic(state_dim, obs_dim, n_embd) if self.use_gail else None
        # discriminator network is optional according to train algorithm
        self.discriminator = Discriminator(
            state_dim, obs_dim, action_dim, n_block, n_embd, n_head, n_agent,
            action_type, disc_use_act_prob, disc_agent_independent,
            disc_mlp_obs_encoder, disc_mlp_act_encoder,
            disc_mlp_use_first_token,
        ) if self.use_gail else None
        # copy MAT to current device
        self.to(device)

    def zero_std(self):
        if self.action_type != 'Discrete':
            self.decoder.zero_std(self.device)

    def forward(self, state, obs, action, available_actions=None):
        # state: (batch, n_agent, state_dim)
        # obs: (batch, n_agent, obs_dim)
        # action: (batch, n_agent, 1)
        # available_actions: (batch, n_agent, act_dim)

        # state unused
        ori_shape = np.shape(state)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        action = check(action).to(**self.tpdv)

        if available_actions is not None:
            available_actions = check(available_actions).to(**self.tpdv)

        batch_size = np.shape(state)[0]
        obs_rep = self.encoder(state, obs)
        if self.action_type == 'Discrete':
            action = action.long()
            action_log, entropy = discrete_parallel_act(self.decoder, obs_rep, obs, action, batch_size,
                                                        self.n_agent, self.action_dim, self.tpdv, available_actions)
        else:
            action_log, entropy = continuous_parallel_act(self.decoder, obs_rep, obs, action, batch_size,
                                                          self.n_agent, self.action_dim, self.tpdv)

        return action_log, entropy

    def get_actions(self, state, obs, available_actions=None, deterministic=False):
        # state unused
        ori_shape = np.shape(obs)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        if available_actions is not None:
            available_actions = check(available_actions).to(**self.tpdv)

        batch_size = np.shape(obs)[0]

        obs_rep = self.encoder(state, obs)
        if self.action_type == "Discrete":
            output_action, output_action_log, output_action_prob = discrete_autoregreesive_act(
                self.decoder, obs_rep, obs, batch_size, self.n_agent, self.action_dim, self.tpdv, available_actions, deterministic)
        else:
            output_action, output_action_log, output_action_prob = continuous_autoregreesive_act(
                self.decoder, obs_rep, obs, batch_size, self.n_agent, self.action_dim, self.tpdv, deterministic)

        return output_action, output_action_log, output_action_prob

    def get_critic_values(self, state, obs):
        """
        state torch.Size([ep_len, agent_num, share_obs_dim])
        obs torch.Size([ep_len, agent_num, obs_dim])
        obs_rep torch.Size([ep_len, agent_num, n_embd])
        action torch.Size([ep_len, agent_num, act_dim])
        """
        # state unused
        ori_shape = np.shape(state)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        # critic network should get state that processed by encoder
        # obs_rep = self.encoder(state, obs).reshape(obs.shape[0], -1)
        obs_rep = self.encoder(state, obs)
        v_tot = self.critic(obs_rep)

        return v_tot

    def get_discriminator_logit(self, state, obs, action):
        """
        state torch.Size([ep_len, agent_num, share_obs_dim])
        obs torch.Size([ep_len, agent_num, obs_dim])
        obs_rep torch.Size([ep_len, agent_num, n_embd])
        action torch.Size([ep_len, agent_num, act_dim])
        """
        # state unused
        ori_shape = np.shape(state)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        action = check(action).to(**self.tpdv)
        dis_logit = self.discriminator(state, obs, action)

        return dis_logit

    def get_discriminator_reward(self, state, obs, action):
        # state unused
        ori_shape = np.shape(state)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        action = check(action).to(**self.tpdv)
        dis_reward = self.discriminator.get_rewards(state, obs, action)

        return dis_reward

    def get_discriminator_rewards_from_logits(self, logits):
        return self.discriminator.get_rewards_from_logits(logits)

    ######################### function unused
    def get_actor_distb(self, state, obs, available_actions=None):
        # state unused
        ori_shape = np.shape(obs)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        if available_actions is not None:
            available_actions = check(available_actions).to(**self.tpdv)

        batch_size = np.shape(obs)[0]
        obs_rep = self.encoder(state, obs)

        # only consider continue and use gal condition
        multi_agent_distri = continuous_autoregreesive_distb_gail(
            self.decoder, obs_rep, obs, batch_size,
            self.n_agent, self.action_dim, self.tpdv, False,
        )
        multi_agent_distri = Multi_Agent_Distribute(multi_agent_distri, self.action_type)

        return multi_agent_distri

    def get_parallel_actor_distb(self, state, obs, action, available_actions=None):
        # state unused
        ori_shape = np.shape(obs)
        state = np.zeros((*ori_shape[:-1], 37), dtype=np.float32)

        state = check(state).to(**self.tpdv)
        obs = check(obs).to(**self.tpdv)
        if available_actions is not None:
            available_actions = check(available_actions).to(**self.tpdv)

        batch_size = np.shape(obs)[0]
        obs_rep = self.encoder(state, obs)

        # only consider continue and use gal condition
        multi_agent_distri = continuous_parallel_act_distb_gail(
            self.decoder, obs_rep, obs, action, batch_size,
            self.n_agent, self.action_dim, self.tpdv
        )
        # multi_agent_distri = Multi_Agent_Distribute(multi_agent_distri, self.action_type)

        return multi_agent_distri
