import argparse


def get_config():
    """
    The configuration parser for hyper-parameters of all environment.
    Please reach each `scripts/train/<env>_runner.py` file to find private hyper-parameters
    only used in <env>.

    Prepare parameters:
        --algorithm_name <algorithm_name>
            specifiy the algorithm, including `["mat", "mat_dec"]`
        --experiment_name <str>
            an identifier to distinguish different experiment.
        --seed <int>
            set seed for numpy and torch 
        --cuda
            by default True, will use GPU to train; or else will use CPU; 
        --cuda_deterministic
            by default, make sure random seed effective. if set, bypass such function.
        --n_training_threads <int>
            number of training threads working in parallel. by default 1
        --n_rollout_threads <int>
            number of parallel envs for training rollout. by default 32
        --n_eval_rollout_threads <int>
            number of parallel envs for evaluating rollout. by default 1
        --n_render_rollout_threads <int>
            number of parallel envs for rendering, could only be set as 1 for some environments.
        --num_env_steps <int>
            number of env steps to train (default: 10e6)
        --user_name <str>
            [for wandb usage], to specify user's name for simply collecting training data.
        --use_wandb
            [for wandb usage], by default True, will log date to wandb server. or else will use tensorboard to log data.
    
    Env parameters:
        --env_name <str>
            specify the name of environment
        --use_obs_instead_of_state
            [only for some env] by default False, will use global state; or else will use concatenated local obs.
    
    Replay Buffer parameters:
        --episode_length <int>
            the max length of episode in the buffer. 
    
    Network parameters:
        --share_policy
            by default True, all agents will share the same network; set to make training agents use different policies. 
        --use_centralized_V
            by default True, use centralized training mode; or else will decentralized training mode.
        --stacked_frames <int>
            Number of input frames which should be stack together.
        --hidden_size <int>
            Dimension of hidden layers for actor/critic networks
        --layer_N <int>
            Number of layers for actor/critic networks
        --use_ReLU
            by default True, will use ReLU. or else will use Tanh.
        --use_popart
            by default True, use PopArt to normalize rewards. 
        --use_valuenorm
            by default True, use running mean and std to normalize rewards. 
        --use_feature_normalization
            by default True, apply layernorm to normalize inputs. 
        --use_orthogonal
            by default True, use Orthogonal initialization for weights and 0 initialization for biases. or else, will use xavier uniform inilialization.
        --gain
            by default 0.01, use the gain # of last action layer
        --use_naive_recurrent_policy
            by default False, use the whole trajectory to calculate hidden states.
        --use_recurrent_policy
            by default, use Recurrent Policy. If set, do not use.
        --recurrent_N <int>
            The number of recurrent layers ( default 1).
        --data_chunk_length <int>
            Time length of chunks used to train a recurrent_policy, default 10.
    
    Optimizer parameters:
        --lr <float>
            learning rate parameter,  (default: 5e-4, fixed).
        --critic_lr <float>
            learning rate of critic  (default: 5e-4, fixed)
        --opti_eps <float>
            RMSprop optimizer epsilon (default: 1e-5)
        --weight_decay <float>
            coefficience of weight decay (default: 0)
    
    PPO parameters:
        --ppo_epoch <int>
            number of ppo epochs (default: 15)
        --use_clipped_value_loss 
            by default, clip loss value. If set, do not clip loss value.
        --clip_param <float>
            ppo clip parameter (default: 0.2)
        --num_mini_batch <int>
            number of batches for ppo (default: 1)
        --entropy_coef <float>
            entropy term coefficient (default: 0.01)
        --use_max_grad_norm 
            by default, use max norm of gradients. If set, do not use.
        --max_grad_norm <float>
            max norm of gradients (default: 0.5)
        --use_gae
            by default, use generalized advantage estimation. If set, do not use gae.
        --gamma <float>
            discount factor for rewards (default: 0.99)
        --gae_lambda <float>
            gae lambda parameter (default: 0.95)
        --use_proper_time_limits
            by default, the return value does consider limits of time. If set, compute returns with considering time limits factor.
        --use_huber_loss
            by default, use huber loss. If set, do not use huber loss.
        --use_value_active_masks
            by default True, whether to mask useless data in value loss.  
        --huber_delta <float>
            coefficient of huber loss.  
    
    PPG parameters:
        --aux_epoch <int>
            number of auxiliary epochs. (default: 4)
        --clone_coef <float>
            clone term coefficient (default: 0.01)
    
    Run parameters：
        --use_linear_lr_decay
            by default, do not apply linear decay to learning rate. If set, use a linear schedule on the learning rate
    
    Save & Log parameters:
        --save_interval <int>
            time duration between contiunous twice models saving.
        --log_interval <int>
            time duration between contiunous twice log printing.
    
    Eval parameters:
        --use_eval
            by default, do not start evaluation. If set`, start evaluation alongside with training.
        --eval_interval <int>
            time duration between contiunous twice evaluation progress.
        --eval_episodes <int>
            number of episodes of a single evaluation.
    
    Render parameters:
        --save_gifs
            by default, do not save render video. If set, save video.
        --use_render
            by default, do not render the env during training. If set, start render. Note: something, the environment has internal render process which is not controlled by this hyperparam.
        --render_episodes <int>
            the number of episodes to render a given env
        --ifi <float>
            the play interval of each rendered image in saved video.
    
    Pretrained parameters:
        --model_dir <str>
            by default None. set the path to pretrained model.
    """
    parser = argparse.ArgumentParser(
        description='onpolicy', formatter_class=argparse.RawDescriptionHelpFormatter)

    # prepare parameters
    parser.add_argument("--algorithm_name", type=str,
                        default='mat', choices=["mat", "mat_dec", "mat_encoder", "mat_decoder", "mat_gru"])

    parser.add_argument("--experiment_name", type=str, default="check", help="an identifier to distinguish different experiment.")
    parser.add_argument("--seed", type=int, default=1, help="Random seed for numpy/torch")
    parser.add_argument("--cuda", action='store_false', default=True, help="by default True, will use GPU to train; or else will use CPU;")
    parser.add_argument("--cuda_deterministic",
                        action='store_false', default=True, help="by default, make sure random seed effective. if set, bypass such function.")
    parser.add_argument("--n_training_threads", type=int,
                        default=1, help="Number of torch threads for training")
    parser.add_argument("--n_rollout_threads", type=int, default=32,
                        help="Number of parallel envs for training rollouts")
    parser.add_argument("--n_eval_rollout_threads", type=int, default=1,
                        help="Number of parallel envs for evaluating rollouts")
    parser.add_argument("--n_render_rollout_threads", type=int, default=1,
                        help="Number of parallel envs for rendering rollouts")
    parser.add_argument("--num_env_steps", type=int, default=10e6,
                        help='Number of environment steps to train (default: 10e6)')
    parser.add_argument("--user_name", type=str, default='xxx',help="[for wandb usage], to specify user's name for simply collecting training data.")
    parser.add_argument("--use_wandb", action='store_false', default=False, help="[for wandb usage], by default True, will log date to wandb server. or else will use tensorboard to log data.")

    # env parameters
    parser.add_argument("--env_name", type=str, default='StarCraft2', help="specify the name of environment")
    parser.add_argument("--use_obs_instead_of_state", action='store_true',
                        default=False, help="Whether to use global state or concatenated obs")

    # replay buffer parameters
    parser.add_argument("--episode_length", type=int,
                        default=200, help="Max length for any episode")

    # network parameters
    parser.add_argument("--share_policy", action='store_false',
                        default=True, help='Whether agent share the same policy')
    parser.add_argument("--use_centralized_V", action='store_false',
                        default=True, help="Whether to use centralized V function")
    parser.add_argument("--stacked_frames", type=int, default=1,
                        help="Dimension of hidden layers for actor/critic networks")
    parser.add_argument("--use_stacked_frames", action='store_true',
                        default=False, help="Whether to use stacked_frames")
    parser.add_argument("--hidden_size", type=int, default=64,
                        help="Dimension of hidden layers for actor/critic networks") 
    parser.add_argument("--layer_N", type=int, default=2,
                        help="Number of layers for actor/critic networks")
    parser.add_argument("--use_ReLU", action='store_false',
                        default=True, help="Whether to use ReLU")
    parser.add_argument("--use_popart", action='store_true', default=False, help="by default False, use PopArt to normalize rewards.")
    parser.add_argument("--use_valuenorm", action='store_false', default=True, help="by default True, use running mean and std to normalize rewards.")
    parser.add_argument("--use_feature_normalization", action='store_false', default=True, help="Whether to apply layernorm to the inputs")
    parser.add_argument("--use_orthogonal", action='store_false', default=True,
                        help="Whether to use Orthogonal initialization for weights and 0 initialization for biases")
    parser.add_argument("--gain", type=float, default=0.01,
                        help="The gain # of last action layer")

    # recurrent parameters
    parser.add_argument("--use_naive_recurrent_policy", action='store_true',
                        default=False, help='Whether to use a naive recurrent policy')
    parser.add_argument("--use_recurrent_policy", action='store_true',
                        default=False, help='use a recurrent policy')
    parser.add_argument("--recurrent_N", type=int, default=1, help="The number of recurrent layers.")
    parser.add_argument("--data_chunk_length", type=int, default=10,
                        help="Time length of chunks used to train a recurrent_policy")

    # optimizer parameters
    parser.add_argument("--lr", type=float, default=5e-4,
                        help='learning rate (default: 5e-4)')
    parser.add_argument("--critic_lr", type=float, default=5e-4,
                        help='critic learning rate (default: 5e-4)')
    parser.add_argument("--disc_lr", type=float, default=5e-5,
                        help='learning rate (default: 5e-4)')
    parser.add_argument("--opti_eps", type=float, default=1e-5,
                        help='RMSprop optimizer epsilon (default: 1e-5)')
    parser.add_argument("--weight_decay", type=float, default=0)

    # ppo parameters
    parser.add_argument("--ppo_epoch", type=int, default=15,
                        help='number of ppo epochs (default: 15)')
    parser.add_argument("--use_clipped_value_loss",
                        action='store_false', default=True, help="by default, clip loss value. If set, do not clip loss value.")
    parser.add_argument("--clip_param", type=float, default=0.2,
                        help='ppo clip parameter (default: 0.2)')
    parser.add_argument("--num_mini_batch", type=int, default=1,
                        help='number of batches for ppo (default: 1)')
    parser.add_argument("--entropy_coef", type=float, default=0.01,
                        help='entropy term coefficient (default: 0.01)')
    parser.add_argument("--value_loss_coef", type=float,
                        default=1, help='value loss coefficient (default: 0.5)')
    parser.add_argument("--use_max_grad_norm",
                        action='store_false', default=True, help="by default, use max norm of gradients. If set, do not use.")
    parser.add_argument("--max_grad_norm", type=float, default=10.0,
                        help='max norm of gradients (default: 0.5)')
    parser.add_argument("--use_gae", action='store_false',
                        default=True, help='use generalized advantage estimation')
    parser.add_argument("--gamma", type=float, default=0.96,
                        help='discount factor for rewards (default: 0.99)')
    parser.add_argument("--gae_lambda", type=float, default=0.95,
                        help='gae lambda parameter (default: 0.95)')
    parser.add_argument("--use_proper_time_limits", action='store_true',
                        default=False, help='compute returns taking into account time limits')
    parser.add_argument("--use_huber_loss", action='store_false', default=True, help="by default, use huber loss. If set, do not use huber loss.")
    parser.add_argument("--use_value_active_masks",
                        action='store_false', default=True, help="by default True, whether to mask useless data in value loss.")
    parser.add_argument("--use_policy_active_masks",
                        action='store_false', default=True, help="by default True, whether to mask useless data in policy loss.")
    parser.add_argument("--huber_delta", type=float, default=10.0, help=" coefficience of huber loss.")

    # run parameters
    parser.add_argument("--use_linear_lr_decay", action='store_true',
                        default=False, help='use a linear schedule on the learning rate')
    # save parameters
    parser.add_argument("--save_interval", type=int, default=100, help="time duration between contiunous twice models saving.")

    # log parameters
    parser.add_argument("--log_interval", type=int, default=5, help="time duration between contiunous twice log printing.")

    # eval parameters
    parser.add_argument("--use_eval", action='store_true', default=False, help="by default, do not start evaluation. If set`, start evaluation alongside with training.")
    parser.add_argument("--eval_interval", type=int, default=25, help="time duration between contiunous twice evaluation progress.")
    parser.add_argument("--eval_episodes", type=int, default=32, help="number of episodes of a single evaluation.")

    # render parameters
    parser.add_argument("--save_gifs", action='store_true', default=False, help="by default, do not save render video. If set, save video.")
    parser.add_argument("--use_render", action='store_true', default=False, help="by default, do not render the env during training. If set, start render. Note: something, the environment has internal render process which is not controlled by this hyperparam.")
    parser.add_argument("--render_episodes", type=int, default=5, help="the number of episodes to render a given env")
    parser.add_argument("--ifi", type=float, default=0.1, help="the play interval of each rendered image in saved video.")

    # pretrained parameters
    parser.add_argument("--model_dir", type=str, default=None, help="by default None. set the path to pretrained model.")
    parser.add_argument("--optim_dir", type=str, default=None, help="by default None. set the path to pretrained optim.")

    # add for transformer
    parser.add_argument("--encode_state", action='store_true', default=False)
    parser.add_argument("--n_block", type=int, default=1)
    parser.add_argument("--n_embd", type=int, default=64)
    parser.add_argument("--n_head", type=int, default=1)
    parser.add_argument("--dec_actor", action='store_true', default=False)
    parser.add_argument("--share_actor", action='store_true', default=False)

    # add for online multi-task
    parser.add_argument("--train_maps", type=str, nargs='+', default=None)
    parser.add_argument("--eval_maps", type=str, nargs='+', default=None)

    ############## add for offline training
    parser.add_argument("--num_epochs", type=int, default=100, help="Number episode to train in train dataset.")
    parser.add_argument("--num_steps_per_epochs", type=int, default=1000, help="Number of steps to train in an episode.")
    parser.add_argument('--lr_decay_step_size', default=1000, type=int, help='Lr decay frequency.')
    parser.add_argument('--lr_decay_gamma', default=0.99, type=float, help='Lr decay rate.')
    parser.add_argument("--batch_size", type=int, default=128, help="Size of a training batch.")
    parser.add_argument("--train_data_path", type=str, default='./data/train_set.pkl', help="Path to load train set data.")
    parser.add_argument("--quick_tqdm", action='store_true', default=False, help="If quickly update tqdm info while training.")

    ############## add for gail training
    parser.add_argument("--use_gail", action='store_true', default=False, help="If use gail to train instead of behavior cloning.")
    parser.add_argument("--normalize_advantage", action='store_true', default=False, help="If normalize gae advantage.")
    parser.add_argument("--epoch_disc", type=int, default=10, help="Number of epoch to train discriminator in an update.")
    parser.add_argument("--gail_epsilon", type=float, default=0.01, help="Epsilon to calculate conjugate gradient step length.")
    parser.add_argument("--fix_encoder", action='store_true', default=False, help="If load pretrain encoder parameters and fix encoder.")
    parser.add_argument("--encoder_dir", type=str, default=None, help="by default None. set the path to pretrained encoder.")
    parser.add_argument("--use_disc_early_stop", action='store_true', default=False, help="If early stop disc update while training.")
    parser.add_argument("--use_disc_lr_decay", action='store_true', default=False, help="If early stop disc update while training.")
    parser.add_argument("--disc_lr_decay_step", type=int, default=50, help="Step size for disc lr decay.")
    parser.add_argument("--disc_lr_decay_gamma", type=float, default=0.98, help="Gamma for disc lr decay.")
    parser.add_argument("--disc_stop_acc", type=float, default=0.9, help="Acc value to stop training disc.")
    parser.add_argument("--disc_stop_round", type=int, default=10, help="Stopping training round for disc.")
    parser.add_argument("--disc_restart_type", type=str, default='fix_epoch', help="Disc restart rule{fix_epoch, low_acc}.")
    parser.add_argument("--disc_restart_acc", type=float, default=0.7, help="Acc value to restart training disc.")
    parser.add_argument("--disc_batch_size", type=int, default=128, help="Disc fix batch size if disc_fix_batch is true.")
    parser.add_argument("--disc_warm_up", action='store_true', default=False, help="If use warm up to train disc.")
    parser.add_argument("--disc_warm_up_round", type=int, default=10, help="Warm up stage for first n rounds.")
    parser.add_argument("--disc_warm_up_epoch", type=int, default=100, help="Disc training epoch at warmup stage.")
    ############## add for wail training
    parser.add_argument("--use_wail", action='store_true', default=False, help="If use wail loss to train disc.")
    parser.add_argument("--use_disc_grad_penalty", action='store_true', default=False, help="If use use grad penalty when training disc.")
    parser.add_argument("--add_expert_noise", action='store_true', default=False, help="If add noise to expert action to prevent disc learning.")
    parser.add_argument("--expert_noise_rate", type=float, default=1.0, help="Expert action noise rate of ori expert action std.")
    parser.add_argument("--disc_use_act_prob", action='store_true', default=False, help="If disc learn logprob for discrete action.")
    parser.add_argument("--disc_agent_independent", action='store_true', default=False, help="If use independent disc for different agents.")
    ############## add for encoder-decoder disc
    parser.add_argument("--disc_use_decoder", action='store_true', default=False, help="If add decoder structure for disc.")
    parser.add_argument("--disc_share_value", action='store_true', default=False, help="If use same value for all agent(sum up all agent disc).")
    parser.add_argument("--disc_mask_action", action='store_false', default=True, help="If mask action in disc decoder.")
    parser.add_argument("--disc_inner_dim", type=int, default=32, help="Hidden dim for disc encoder and decoder.")
    parser.add_argument("--disc_cal_last_loss", action='store_true', default=False, help="If only use last token to calculate disc loss.")
    parser.add_argument("--disc_drop_cross_atten", action='store_true', default=False, help="If not use cross attention between obs and actions.")
    ############## add for gru history mat
    parser.add_argument("--mat_use_history", action='store_true', default=False, help="If add multi step obs to mat.")
    parser.add_argument("--cat_his_with_now", action='store_true', default=False, help="If cat his obs and now obs.")
    parser.add_argument('--history_obs_len', default=5, type=int, help='Len of history state obs sequence.')
    parser.add_argument('--gru_num_layers', default=2, type=int, help='Numbers of gru layers.')
    parser.add_argument('--gru_hidden_size', default=64, type=int, help='Hidden size for gru structure.')
    ############## add for mlp disc
    parser.add_argument("--disc_use_mlp", action='store_true', default=False, help="If use pure mlp structure for disc.")
    parser.add_argument("--disc_mlp_obs_encoder", action='store_true', default=False, help="If use obs encoder for mlp.")
    parser.add_argument("--disc_mlp_act_encoder", action='store_true', default=False, help="If use act encoder for mlp.")
    parser.add_argument("--disc_mlp_use_first_token", action='store_true', default=False, help="If use first token as center disc.")
    ############## add for gmlp disc
    parser.add_argument("--disc_use_gmlp", action='store_true', default=False, help="If use gmlp structure for disc.")
    parser.add_argument('--disc_gmlp_dim_ff', default=2, type=int, help='Expand multiple for gmlp inner dim.')
    parser.add_argument("--disc_gmlp_use_causal", action='store_true', default=False, help="If use causal for gmlp disc.")
    parser.add_argument("--disc_gmlp_add_embd", action='store_true', default=False, help="If add or cat obs and actions embd.")
    parser.add_argument("--disc_gmlp_obs_encoder", action='store_true', default=False, help="If use gmlp as obs encoder.")
    ############## add for pretrain classify model
    parser.add_argument("--pretrain_classifier", action='store_true', default=False)
    parser.add_argument("--classifier_only_action", action='store_true', default=False)
    parser.add_argument("--classifier_use_gru", action='store_true', default=False)
    parser.add_argument('--classifier_gru_his_len', default=10, type=int)
    parser.add_argument('--classifier_gru_num_layer', default=2, type=int)
    parser.add_argument("--classifier_n_embd", type=int, default=64)
    parser.add_argument("--classifier_use_act_enc", action='store_true', default=False)
    parser.add_argument("--classifier_act_enc_mask", action='store_true', default=False)
    parser.add_argument("--classifier_use_data_tag", action='store_true', default=False)
    parser.add_argument("--classifier_data_tag_num", type=int, default=3)
    ############## add for classify model
    parser.add_argument("--use_classifier_reward", action='store_true', default=False)
    parser.add_argument("--classifier_model_path", type=str, default='./model.pt')
    parser.add_argument("--classifier_reward_rate", type=float, default=1.0)
    ############## add for shuffle among agents
    parser.add_argument("--shuffle_buffer_agents_data", action='store_true', default=False)
    parser.add_argument("--use_agents_pos_embd", action='store_true', default=False)
    ############## add for evaluation model
    parser.add_argument("--evaluate_model", action='store_true', default=False)

    return parser
