import pandas as pd
import numpy as np
import torch
import torch.nn as nn
import matplotlib
import matplotlib.pyplot as plt
from torchsummary import summary

# load a single file as a numpy array


def load_file(filepath):
    dataframe = pd.read_csv(filepath, header=None, delim_whitespace=True)
    return dataframe.values

# load a list of files into a 3D array of [samples, timesteps, features]


def load_group(filenames, prefix=''):
    loaded = list()
    for name in filenames:
        data = load_file(prefix + name)
        loaded.append(data)
    # stack group so that features are the 3rd dimension
    loaded = np.dstack(loaded)
    return loaded

# load a dataset group, such as train or test


def load_dataset_group(group, prefix=''):
    filepath = prefix + group + '/Inertial Signals/'
    # load all 9 files as a single array
    filenames = list()
    # total acceleration
    filenames += ['total_acc_x_' + group + '.txt', 'total_acc_y_' +
                  group + '.txt', 'total_acc_z_' + group + '.txt']
    # body acceleration
    filenames += ['body_acc_x_' + group + '.txt', 'body_acc_y_' +
                  group + '.txt', 'body_acc_z_' + group + '.txt']
    # body gyroscope
    filenames += ['body_gyro_x_' + group + '.txt', 'body_gyro_y_' +
                  group + '.txt', 'body_gyro_z_' + group + '.txt']
    # load input data
    X = load_group(filenames, filepath)
    # load class output
    y = load_file(prefix + group + '/y_' + group + '.txt')
    return X, y


def to_categorical(y, num_classes=None, dtype='float32'):
    """Converts a class vector (integers) to binary class matrix.
    E.g. for use with categorical_crossentropy.
    # Arguments
        y: class vector to be converted into a matrix
            (integers from 0 to num_classes).
        num_classes: total number of classes.
        dtype: The data type expected by the input, as a string
            (`float32`, `float64`, `int32`...)
    # Returns
        A binary matrix representation of the input. The classes axis
        is placed last.
    # Example
    ```python
    # Consider an array of 5 labels out of a set of 3 classes {0, 1, 2}:
    > labels
    array([0, 2, 1, 2, 0])
    # `to_categorical` converts this into a matrix with as many
    # columns as there are classes. The number of rows
    # stays the same.
    > to_categorical(labels)
    array([[ 1.,  0.,  0.],
           [ 0.,  0.,  1.],
           [ 0.,  1.,  0.],
           [ 0.,  0.,  1.],
           [ 1.,  0.,  0.]], dtype=float32)
    ```
    """

    y = np.array(y, dtype='int')
    input_shape = y.shape
    if input_shape and input_shape[-1] == 1 and len(input_shape) > 1:
        input_shape = tuple(input_shape[:-1])
    y = y.ravel()
    if not num_classes:
        num_classes = np.max(y) + 1
    n = y.shape[0]
    categorical = np.zeros((n, num_classes), dtype=dtype)
    categorical[np.arange(n), y] = 1
    output_shape = input_shape + (num_classes,)
    categorical = np.reshape(categorical, output_shape)
    return categorical

# load the dataset, returns train and test X and y elements


def load_dataset(prefix=''):
    # load all train
    trainX, trainy = load_dataset_group('train', prefix)
    print(trainX.shape, trainy.shape)
    # load all test
    testX, testy = load_dataset_group('test', prefix)
    print(testX.shape, testy.shape)
    # zero-offset class values
    trainy = trainy - 1
    testy = testy - 1
    # one hot encode y
    trainy = to_categorical(trainy)
    testy = to_categorical(testy)
    print(trainX.shape, trainy.shape, testX.shape, testy.shape)
    return trainX, trainy, testX, testy


# define the model using pytorch
class ConvNet1D(nn.Module):
    def __init__(self):
        super().__init__()
        self.layer1 = nn.Sequential(
            nn.Conv1d(9, 64, kernel_size=3),
            nn.ReLU(),
            nn.Dropout(0.2))
        self.layer2 = nn.Flatten()
        self.layer3 = nn.Sequential(
            nn.Linear(126*64, 100),
            nn.ReLU())
        self.layer4 = nn.Sequential(
            nn.Linear(100, 6))

    def forward(self, x):
        out = self.layer1(x)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        return out


if __name__ == '__main__':

    device = 'cuda' if torch.cuda.is_available() else 'cpu'

    # load data set and split into training and testing inputs (X) and outputs (y)
    trainX, trainy, testX, testy = load_dataset(
        '/workspace/data/HAR/UCI HAR Dataset/')

    n_timesteps, n_features, n_outputs = trainX.shape[1], trainX.shape[2], trainy.shape[1]

    total_step = len(trainX)

    # transformation of data into torch tensors
    trainXT = torch.from_numpy(trainX)
    # input is (N, Cin, Lin) = Ntimesteps, Nfeatures, 128
    trainXT = trainXT.transpose(1, 2).float()
    trainyT = torch.from_numpy(trainy).float()
    testXT = torch.from_numpy(testX)
    testXT = testXT.transpose(1, 2).float()
    testyT = torch.from_numpy(testy).float()

    model = ConvNet1D()
    model.to(device)
    # print(summary(model,(9,128)))

    # Loss and optimizer
    criterion = nn.CrossEntropyLoss()
    optimizer = torch.optim.Adam(model.parameters(), lr=0.001)

    # Train the model
    num_epochs = 20
    batch_size = 32

    loss_list = []
    acc_list = []
    acc_list_epoch = []
    for epoch in range(num_epochs):
        correct_sum = 0
        for i in range(int(np.floor(total_step/batch_size))):  # split data into batches
            trainXT_seg = trainXT[i*batch_size:(i+1)*batch_size]
            trainyT_seg = trainyT[i*batch_size:(i+1)*batch_size]
            trainXT_seg = trainXT_seg.to(device)
            trainyT_seg = trainyT_seg.to(device)
            # Run the forward pass
            outputs = model(trainXT_seg)
            loss = criterion(outputs, torch.max(trainyT_seg, 1)[1])
            loss_list.append(loss.item())

            # Backprop and perform Adam optimisation
            optimizer.zero_grad()
            loss.backward()
            optimizer.step()
            # Track the accuracy
            total = trainyT_seg.size(0)
            _, predicted = torch.max(outputs, 1)
            _, actual = torch.max(trainyT_seg, 1)
            correct = (predicted == actual).sum().item()
            correct_sum = correct_sum + (correct/total)
            acc_list.append(correct / total)
        print(
            f"Epoch: {epoch}, Accuracy:{correct_sum/int(np.floor(total_step/batch_size))}")
        acc_list_epoch.append(correct_sum/int(np.floor(total_step/batch_size)))

    # Test the model
    model.eval()
    with torch.no_grad():
        test_outputs = model(testXT.to(device))
        _, predictedt = torch.max(test_outputs, 1)
        _, actual = torch.max(testyT, 1)
        total_t = testyT.size(0)
        correct_t = (predictedt == actual.to(device)).sum().item()
        print(f'Test accuracy: {(correct_t/total_t)*100}')
