'''
KP (end-to-end)
'''
import sys
import pickle
import types
import warnings
import time
from typing import Any
import numpy as np
from prompts import GetPrompts


class KP():
    """Evaluator for knapsack problem."""

    def __init__(self,
                 problem_size=100,
                 running_time=30,
                 dirname=None,  # or str e.g., 'dataset_mcts'
                 mode='train',
                 debug=False,
                 **kwargs):

        self.time = running_time
        self.dirname = dirname
        self.debug = debug
        self.problem_size = problem_size
        if self.problem_size == 50:
            self.capacity = 12.5
        else:  # 100, 200
            self.capacity = 25.
        self.prompts = GetPrompts(problem_size)
        if self.dirname is None:  # NOTE: unused
            # Load from files
            with open(f'all_data_{mode}.pkl', 'rb') as f:
                all_data = pickle.load(f)
            self._datasets = all_data[self.problem_size]
            self.n_instance = len(self._datasets)
        else:
            if self.dirname == 'dataset_mcts':
                data = np.load(f'{self.dirname}/{mode}{problem_size}_dataset.npy')
                self._datasets = []
                self.n_instance = data.shape[0]
                for i in range(self.n_instance):
                    weights = data[i,:,0]
                    values = data[i,:,1]
                    self._datasets.append((weights, values, self.capacity))

    def get_seed_algorithms(self):
        algorithms = {
            'constructive': "...",
        }
        return algorithms

    # def evaluate_program(self, program_str: str, callable_func: callable) -> Any | None:
    #     return self.evaluate(callable_func)

    def calObjectives(self, alg, reduction=None):
        '''
        alg: a Python function coded by LLM
        '''
        start_time = time.time()
        total_value = 0

        for i, (weights, values, capacity) in enumerate(self._datasets):

            if time.time()-start_time > self.time:
                # print(i)
                return None, None

            if reduction is None:
                items = alg.pack_items(weights, values, capacity)
            else:
                input_A = (weights, values, capacity)
                input_B = reduction.convert_input_A_to_B(*input_A)
                solution_B = alg.solve_B(input_B)
                items = np.array(reduction.convert_solution_B_to_A(solution_B))
                # print(items)
                if items.size == 0 or np.unique(items).size != items.size or np.sum(weights[items]) > capacity:  # invalid packing
                    return None, None
                else:
                    if self.debug and i == 0:
                        with open(f'example_bag.pkl', 'wb') as f:
                            pickle.dump(items, f)

            total_value += np.sum(values[items])

        ave_value = total_value / self.n_instance
        # print("average value: ",ave_value)
        return ave_value, time.time()-start_time


    def evaluate(self, code_string, reduction=None):
        '''
        Return: fitness score (higher is better) for the LLM-generated heuristic (as code)
        '''
        #try:
        #Suppress warnings
        with warnings.catch_warnings():
            warnings.simplefilter("ignore")

            if reduction is not None:
            # Create a new module object for the reduction
                reduction_module = types.ModuleType("reduction_module")
                
                # Execute the code string in the new module's namespace
                exec(reduction, reduction_module.__dict__)

                # Add the module to sys.modules so it can be imported
                sys.modules[reduction_module.__name__] = reduction_module
            else:
                reduction_module = None

        # Create a new module object for the heuristic
            heuristic_module = types.ModuleType("heuristic_module")
            
            # Execute the code string in the new module's namespace
            exec(code_string, heuristic_module.__dict__)

            # Add the module to sys.modules so it can be imported
            sys.modules[heuristic_module.__name__] = heuristic_module

            fitness, runtime = self.calObjectives(heuristic_module, reduction_module)

            return fitness, runtime

