"""
Multi-EPL

File: src/network/network_digits.py
Contains the code for networks for Digits-Five experiments
"""

import torch
import torch.nn as nn
import torch.nn.functional as F


class GeneratorDigits(nn.Module):
    def __init__(self):
        super(GeneratorDigits, self).__init__()
        self.conv1 = nn.Conv2d(3, 64, kernel_size=5, stride=1, padding=2)
        self.bn1 = nn.BatchNorm2d(64)
        self.conv2 = nn.Conv2d(64, 64, kernel_size=5, stride=1, padding=2)
        self.bn2 = nn.BatchNorm2d(64)
        self.conv3 = nn.Conv2d(64, 128, kernel_size=5, stride=1, padding=2)
        self.bn3 = nn.BatchNorm2d(128)
        self.fc1 = nn.Linear(8192, 3072)
        self.bn1_fc = nn.BatchNorm1d(3072)
        self.fc2 = nn.Linear(3072, 2048)
        self.bn2_fc = nn.BatchNorm1d(2048)

    def forward(self, x):
        x = F.max_pool2d(F.relu(self.bn1(self.conv1(x))), stride=2, kernel_size=3, padding=1)
        x = F.max_pool2d(F.relu(self.bn2(self.conv2(x))), stride=2, kernel_size=3, padding=1)
        x = F.relu(self.bn3(self.conv3(x)))
        x = x.view(x.size(0), 8192)
        x = F.relu(self.bn1_fc(self.fc1(x)))
        x = F.dropout(x, training=self.training)
        x = F.relu(self.bn2_fc(self.fc2(x)))
        return x


class ExtractorClassifierDigits(nn.Module):
    def __init__(self, input_size=2048, output_size=2):
        super(ExtractorClassifierDigits, self).__init__()

        self.fc3 = nn.Linear(input_size, output_size)

    def forward(self, x):
        x = self.fc3(x)
        return x


class GradientReversalLayer(torch.autograd.Function):
    """
    Implement the gradient reversal layer for the convenience of domain adaptation neural network.
    The forward part is the identity function while the backward part is the negative function.
    """
    @staticmethod
    def forward(ctx, inputs):
        return inputs

    @staticmethod
    def backward(ctx, grad_output):
        grad_input = grad_output.clone()
        grad_input = -grad_input
        return grad_input


class LabelClassifierDigits(nn.Module):
    def __init__(self, input_size=2048):
        super(LabelClassifierDigits, self).__init__()

        self.fc3 = nn.Linear(input_size, 10)

    def forward(self, x):
        x = self.fc3(x)
        return x
