import torch
import numpy as np
import random
from sklearn.metrics import classification_report, f1_score, confusion_matrix, accuracy_score
import matplotlib.pyplot as plt
import seaborn as sns
import pandas as pd


cnames = {
    'aliceblue':            '#F0F8FF',
    'antiquewhite':         '#FAEBD7',
    'aqua':                 '#00FFFF',
    'aquamarine':           '#7FFFD4',
    'azure':                '#F0FFFF',
    'beige':                '#F5F5DC',
    'bisque':               '#FFE4C4',
    'black':                '#000000',
    'blanchedalmond':       '#FFEBCD',
    'blue':                 '#0000FF',
    'blueviolet':           '#8A2BE2',
    'brown':                '#A52A2A',
    'burlywood':            '#DEB887',
    'cadetblue':            '#5F9EA0',
    'chartreuse':           '#7FFF00',
    'chocolate':            '#D2691E',
    'coral':                '#FF7F50',
    'cornflowerblue':       '#6495ED',
    'cornsilk':             '#FFF8DC',
    'crimson':              '#DC143C',
    'cyan':                 '#00FFFF',
    'darkblue':             '#00008B',
    'darkcyan':             '#008B8B',
    'darkgoldenrod':        '#B8860B',
    'darkgray':             '#A9A9A9',
    'darkgreen':            '#006400',
    'darkkhaki':            '#BDB76B',
    'darkmagenta':          '#8B008B',
    'darkolivegreen':       '#556B2F',
    'darkorange':           '#FF8C00',
    'darkorchid':           '#9932CC',
    'darkred':              '#8B0000',
    'darksalmon':           '#E9967A',
    'darkseagreen':         '#8FBC8F',
    'darkslateblue':        '#483D8B',
    'darkslategray':        '#2F4F4F',
    'darkturquoise':        '#00CED1',
    'darkviolet':           '#9400D3',
    'deeppink':             '#FF1493',
    'deepskyblue':          '#00BFFF',
    'dimgray':              '#696969',
    'dodgerblue':           '#1E90FF',
    'firebrick':            '#B22222',
    'floralwhite':          '#FFFAF0',
    'forestgreen':          '#228B22',
    'fuchsia':              '#FF00FF',
    'gainsboro':            '#DCDCDC',
    'ghostwhite':           '#F8F8FF',
    'gold':                 '#FFD700',
    'goldenrod':            '#DAA520',
    'gray':                 '#808080',
    'green':                '#008000',
    'greenyellow':          '#ADFF2F',
    'honeydew':             '#F0FFF0',
    'hotpink':              '#FF69B4',
    'indianred':            '#CD5C5C',
    'indigo':               '#4B0082',
    'ivory':                '#FFFFF0',
    'khaki':                '#F0E68C',
    'lavender':             '#E6E6FA',
    'lavenderblush':        '#FFF0F5',
    'lawngreen':            '#7CFC00',
    'lemonchiffon':         '#FFFACD',
    'lightblue':            '#ADD8E6',
    'lightcoral':           '#F08080',
    'lightcyan':            '#E0FFFF',
    'lightgoldenrodyellow': '#FAFAD2',
    'lightgreen':           '#90EE90',
    'lightgray':            '#D3D3D3',
    'lightpink':            '#FFB6C1',
    'lightsalmon':          '#FFA07A',
    'lightseagreen':        '#20B2AA',
    'lightskyblue':         '#87CEFA',
    'lightslategray':       '#778899',
    'lightsteelblue':       '#B0C4DE',
    'lightyellow':          '#FFFFE0',
    'lime':                 '#00FF00',
    'limegreen':            '#32CD32',
    'linen':                '#FAF0E6',
    'magenta':              '#FF00FF',
    'maroon':               '#800000',
    'mediumaquamarine':     '#66CDAA',
    'mediumblue':           '#0000CD',
    'mediumorchid':         '#BA55D3',
    'mediumpurple':         '#9370DB',
    'mediumseagreen':       '#3CB371',
    'mediumslateblue':      '#7B68EE',
    'mediumspringgreen':    '#00FA9A',
    'mediumturquoise':      '#48D1CC',
    'mediumvioletred':      '#C71585',
    'midnightblue':         '#191970',
    'mintcream':            '#F5FFFA',
    'mistyrose':            '#FFE4E1',
    'moccasin':             '#FFE4B5',
    'navajowhite':          '#FFDEAD',
    'navy':                 '#000080',
    'oldlace':              '#FDF5E6',
    'olive':                '#808000',
    'olivedrab':            '#6B8E23',
    'orange':               '#FFA500',
    'orangered':            '#FF4500',
    'orchid':               '#DA70D6',
    'palegoldenrod':        '#EEE8AA',
    'palegreen':            '#98FB98',
    'paleturquoise':        '#AFEEEE',
    'palevioletred':        '#DB7093',
    'papayawhip':           '#FFEFD5',
    'peachpuff':            '#FFDAB9',
    'peru':                 '#CD853F',
    'pink':                 '#FFC0CB',
    'plum':                 '#DDA0DD',
    'powderblue':           '#B0E0E6',
    'purple':               '#800080',
    'red':                  '#FF0000',
    'rosybrown':            '#BC8F8F',
    'royalblue':            '#4169E1',
    'saddlebrown':          '#8B4513',
    'salmon':               '#FA8072',
    'sandybrown':           '#FAA460',
    'seagreen':             '#2E8B57',
    'seashell':             '#FFF5EE',
    'sienna':               '#A0522D',
    'silver':               '#C0C0C0',
    'skyblue':              '#87CEEB',
    'slateblue':            '#6A5ACD',
    'slategray':            '#708090',
    'snow':                 '#FFFAFA',
    'springgreen':          '#00FF7F',
    'steelblue':            '#4682B4',
    'tan':                  '#D2B48C',
    'teal':                 '#008080',
    'thistle':              '#D8BFD8',
    'tomato':               '#FF6347',
    'turquoise':            '#40E0D0',
    'violet':               '#EE82EE',
    'wheat':                '#F5DEB3',
    'white':                '#FFFFFF',
    'whitesmoke':           '#F5F5F5',
    'yellow':               '#FFFF00',
    'yellowgreen':          '#9ACD32'}

def set_requires_grad(model, requires_grad=True):
    """
    :param model: Instance of Part of Net
    :param requires_grad: Whether Need Gradient
    :return:
    """
    for param in model.parameters():
        param.requires_grad = requires_grad


def fix_randomness(SEED):
    """
    :param SEED:  Random SEED
    :return:
    """
    random.seed(SEED)
    np.random.seed(SEED)
    torch.manual_seed(SEED)
    torch.cuda.manual_seed(SEED)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False


class Evaluator(object):
    def __init__(self, true, pred):
        super(Evaluator, self).__init__()
        self.true = true
        self.pred = pred

    def metric_acc(self):
        return accuracy_score(self.true, self.pred)

    def metric_mf1(self):
        return f1_score(self.true, self.pred, average="macro")

    def metric_mf1_balance(self):
        return f1_score(self.true, self.pred, average="weighted")

    def metric_confusion_matrix(self):
        return confusion_matrix(self.true, self.pred)

    def classification_report(self):
        return classification_report(self.true, self.pred, target_names=['Sleep stage W',
                                                                         'Sleep stage 1',
                                                                         'Sleep stage 2',
                                                                         'Sleep stage 3/4',
                                                                         'Sleep stage R'])


def compute_aaa(acc_list):
    return np.mean(acc_list)


def compute_forget(acc_list):
    return abs(acc_list[0] - acc_list[-1]) / acc_list[0]


def compute_aaf1(mf1_list):
    out = []
    for i in range(len(mf1_list)):
        out.append(np.mean(mf1_list[:i+1]))
    return out


def analysis(parser):
    """
    :param parser: parser["old_task_performance"] = {"ACC": [], "MF1": [], "AAA": [], "FR": []}
                   parser["new_task_performance"] = {i: {"ACC": [], "MF1": []} for i in new_task_idx}
    :return:
    """
    new_task_acc_initial = []
    new_task_acc_before = []
    new_task_acc_after = []

    new_task_mf1_initial = []
    new_task_mf1_before = []
    new_task_mf1_after = []
    for i in parser["new_task_performance"].keys():
        new_task_acc_initial.append(parser["new_task_performance"][i]["ACC"][0])
        new_task_acc_before.append(parser["new_task_performance"][i]["ACC"][1])
        new_task_acc_after.append(parser["new_task_performance"][i]["ACC"][2])

        new_task_mf1_initial.append(parser["new_task_performance"][i]["MF1"][0])
        new_task_mf1_before.append(parser["new_task_performance"][i]["MF1"][1])
        new_task_mf1_after.append(parser["new_task_performance"][i]["MF1"][2])

    print(parser["old_task_performance"])
    print(parser["new_task_performance"])

    print("============Old Task Performance=============")
    print("Old Task ACC Curve", parser["old_task_performance"]["ACC"])
    print("Old Task MF1 Curve", parser["old_task_performance"]["MF1"])
    print("Old Task AAA Curve", parser["old_task_performance"]["AAA"])
    print("Old Task AAF1 Curve", compute_aaf1(parser["old_task_performance"]["MF1"]))
    print("Old Task FR  Curve", parser["old_task_performance"]["FR"])

    print("============New Task Performance=============")
    print("New Task Initial Incremental Learning ACC", np.mean(new_task_acc_initial))
    print("New Task Before Incremental Learning ACC", np.mean(new_task_acc_before))
    print("New Task After Incremental Learning ACC", np.mean(new_task_acc_after))

    print("New Task Initial Incremental Learning MF1", np.mean(new_task_mf1_initial))
    print("New Task Before Incremental Learning MF1", np.mean(new_task_mf1_before))
    print("New Task After Incremental Learning MF1", np.mean(new_task_mf1_after))


def old_task_acc_mf1_plot(acc, mf1, aaa):
    """
    :param acc: Accuracy
    :param mf1: Macro-F1
    :return: Plot of Curve
    """
    rc = {'font.sans-serif': ['Times New Roman']}
    sns.set(font_scale=3, style=None, rc=rc)
    plt.figure(figsize=(24, 12))
    plt.plot(range(len(acc)), acc, c="m", label="ACC", marker="o", markersize=8, linewidth=2)
    plt.plot(range(len(mf1)), mf1, c="rosybrown", label="MF1", marker="P", markersize=8, linewidth=2)
    plt.plot(range(len(aaa)), aaa, c="olivedrab", label="Average Area Accuracy", marker="*", markersize=8, linewidth=2)
    plt.xlabel("Incremental Individual")
    plt.ylabel("Metric")
    plt.title("Old Task Metric Curve")
    plt.legend()
    plt.grid(alpha=0.4)
    plt.show()


def old_task_aaa_fr_plot(aaa, fr):
    """
    :param acc: Accuracy
    :param mf1: Macro-F1
    :return: Plot of Curve
    """
    rc = {'font.sans-serif': ['Times New Roman']}
    sns.set(font_scale=3, style=None, rc=rc)
    plt.figure(figsize=(24, 12))
    plt.plot(range(len(aaa)), aaa, c="olivedrab", label="Average Area Accuracy", marker="o", markersize=8, linewidth=2)
    plt.plot(range(len(fr)), fr, c="royalblue", label="Forgetting Rate", marker="P", markersize=8, linewidth=2)
    plt.xlabel("Incremental Individual")
    plt.ylabel("Metric")
    plt.yticks([0.7, 0.71, 0.72, 0.73, 0.74, 0.75, 0.76, 0.77, 0.78, 0.79, 0.8])
    plt.title("Old Task Metric Curve")
    plt.legend()
    plt.grid(alpha=0.4)
    plt.show()


def old_task_rand_aaa_plot(aaa1, aaa2, aaa3, aaa4):
    """
    :param acc: Accuracy
    :param mf1: Macro-F1
    :return: Plot of Curve
    """
    rc = {'font.sans-serif': ['Times New Roman']}
    sns.set(font_scale=3, style=None, rc=rc)
    plt.figure(figsize=(24, 12))
    plt.plot(range(len(aaa1)), aaa1, c="m", label="4", marker="o", markersize=8, linewidth=2)
    plt.plot(range(len(aaa2)), aaa2, c="rosybrown", label="43", marker="P", markersize=8, linewidth=2)
    plt.plot(range(len(aaa3)), aaa3, c="olivedrab", label="432", marker="*", markersize=8, linewidth=2)
    plt.plot(range(len(aaa4)), aaa4, c="r", label="4321", marker="^", markersize=8,
             linewidth=2)
    plt.xlabel("Incremental Individual")
    plt.ylabel("Metric")
    plt.title("Old Task AAA Curve of Different Rands")
    plt.legend(loc='upper right')
    plt.grid(alpha=0.4)
    plt.show()




def new_task_plot(parser):
    new_task_acc_before = []
    new_task_acc_after = []
    new_task_mf1_before = []
    new_task_mf1_after = []
    for i in parser.keys():
        new_task_acc_before.append(parser[i]["ACC"][0])
        new_task_acc_after.append(parser[i]["ACC"][1])

        new_task_mf1_before.append(parser[i]["MF1"][0])
        new_task_mf1_after.append(parser[i]["MF1"][1])
    print(new_task_acc_before, new_task_acc_after)
    rc = {'font.sans-serif': ['Times New Roman']}
    sns.set(font_scale=3, style=None, rc=rc)
    plt.figure(figsize=(24, 12))
    x_index = range(len(new_task_acc_before))
    plt.bar(x_index, new_task_acc_before, label="ACC Before", width=0.45)
    plt.bar([i+0.45 for i in x_index], new_task_acc_after, label="ACC After", width=0.45)
    # plt.bar(range(len(new_task_acc_before)), new_task_acc_before, bottom=new_task_acc_after, label="ACC Before")
    plt.xlabel("Incremental Individual")
    plt.ylabel("Metric")
    plt.yticks([0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9], [0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9])
    plt.title("New Task ACC Bar")
    plt.legend()
    # plt.grid(alpha=0.4)
    plt.show()

    # acc = []
    # index = []
    # for i in range(len(new_task_acc_before)):
    #     acc.append(new_task_acc_before[i])
    #     index.append("ACC Before")
    #     acc.append(new_task_acc_after[i])
    #     index.append("ACC After")
    #
    # y_ = [(i+1)//2 for i in range(len(new_task_acc_before)*2)]
    # df = pd.DataFrame({"ACC": acc, 'y': y_, "time": index})
    #
    # rc = {'font.sans-serif': ['Times New Roman']}
    # sns.set(font_scale=3, style=None, rc=rc)
    # plt.figure(figsize=(24, 12))
    # # df.replace({"Sick or not": {0: "Normal", 1: "Patient"}}, inplace=True)
    # # df.replace({"Gender": {"男": "Male", "女": "Female"}}, inplace=True)
    # sns.barplot(x='y', y='ACC', hue='time',
    #
    #                  palette="pastel",
    #                  data=df,)
    # # plt.yticks(range(0, 20, 2), range(0, 20, 2), fontsize=15)
    # plt.xticks(range(0, 49, 1), range(0, 49, 1))
    # plt.ylabel("Metric", fontsize=15)
    # plt.xlabel("Incremental Individual", fontsize=15)
    # plt.show()


def compare_different_method(method1, method2):
    rc = {'font.sans-serif': ['Times New Roman']}
    sns.set(font_scale=2.5, style=None, rc=rc)
    plt.figure(figsize=(48, 12))
    ax1 = plt.subplot(1, 3, 1)
    ax1.plot(range(len(method1[0])), method1[0], c="m", label="Finetune", marker="o", markersize=8, linewidth=2)
    ax1.plot(range(len(method1[0])), method2[0], c="rosybrown", label="Ours", marker="P", markersize=8, linewidth=2)
    ax1.set_xlabel("Incremental Individual")
    ax1.set_ylabel("Metrics")
    ax1.set_title("Accuracy")
    ax1.set_yticks([round(i/100, 2) for i in range(50, 86, 5)])
    ax1.set_ylim(0.52, 0.84)
    ax1.legend(loc="lower right")
    ax1.grid(alpha=0.4)

    ax2 = plt.subplot(1, 3, 2)
    ax2.plot(range(len(method1[0])), method1[1], c="m", label="Finetune", marker="o", markersize=8, linewidth=2)
    ax2.plot(range(len(method1[0])), method2[1], c="rosybrown", label="Ours", marker="P", markersize=8, linewidth=2)
    ax2.set_xlabel("Incremental Individual")
    # ax2.set_ylabel("Metrics")
    ax2.set_title("Macro-F1")
    ax2.set_yticks([round(i/100, 2) for i in range(50, 86, 5)])
    ax2.set_ylim(0.52, 0.84)
    ax2.legend(loc="lower right")
    ax2.grid(alpha=0.4)

    ax3 = plt.subplot(1, 3, 3)
    ax3.plot(range(len(method1[0])), method1[2], c="m", label="Finetune", marker="o", markersize=8, linewidth=2)
    ax3.plot(range(len(method1[0])), method2[2], c="rosybrown", label="Ours", marker="P", markersize=8, linewidth=2)
    ax3.set_xlabel("Incremental Individual")
    # ax3.set_ylabel("Metrics")
    ax3.set_title("Average Area Accuracy")
    ax3.legend(loc="lower right")
    ax3.set_yticks([round(i/100, 2) for i in range(50, 86, 5)])
    ax3.set_ylim(0.52, 0.84)
    ax3.grid(alpha=0.4)

    plt.show()


def old_task_lr_aaa_plot(lr):
    """
    :param acc: Accuracy
    :param mf1: Macro-F1
    :return: Plot of Curve
    """
    rc = {'font.sans-serif': ['Times New Roman']}
    sns.set(font_scale=3, style=None, rc=rc)
    plt.figure(figsize=(24, 12))
    plt.plot(range(len(lr[0])), lr[0], c="m", label="CPC_Buffer", marker="o", markersize=8, linewidth=2)
    plt.plot(range(len(lr[0])), lr[1], c="rosybrown", label="CPC_Buffer_Merge", marker="P", markersize=8, linewidth=2)
    plt.plot(range(len(lr[0])), lr[2], c="olivedrab", label="KL_Buffer", marker="*", markersize=8, linewidth=2)
    plt.plot(range(len(lr[0])), lr[3], c="cyan", label="KL_Buffer_Merge", marker="^", markersize=8, linewidth=2)
    plt.plot(range(len(lr[0])), lr[4], c="r", label="MMD_Buffer", marker="s", markersize=8, linewidth=2)
    plt.plot(range(len(lr[0])), lr[5], c="cadetblue", label="MMD_Buffer_Merge", marker="d", markersize=8, linewidth=2)
    plt.xlabel("Incremental Individual")
    plt.ylabel("Metric")
    plt.title("Old Task AAA Curve of Different Methods")
    plt.legend(loc='lower right', fontsize=20)
    plt.grid(alpha=0.4)
    plt.show()


def guassian_kernel(source, target, kernel_mul=2.0, kernel_num=5, fix_sigma=None):
    '''
    将源域数据和目标域数据转化为核矩阵，即上文中的K
    Params:
    source: 源域数据（n * len(x))
    target: 目标域数据（m * len(y))
    kernel_mul:
    kernel_num: 取不同高斯核的数量
    fix_sigma: 不同高斯核的sigma值
	Return:
	sum(kernel_val): 多个核矩阵之和
    '''

    n_samples = int(source.size()[0])+int(target.size()[0])  # 求矩阵的行数，一般source和target的尺度是一样的，这样便于计算
    total = torch.cat([source, target], dim=0)  # 将source,target按列方向合并  32, 16, 512
    #  将total复制（n+m）份
    total0 = total.unsqueeze(0).expand(int(total.size(0)), int(total.size(0)), int(total.size(1)), int(total.size(2)))
    #  将total的每一行都复制成（n+m）行，即每个数据都扩展成（n+m）份
    total1 = total.unsqueeze(1).expand(int(total.size(0)), int(total.size(0)), int(total.size(1)), int(total.size(2)))
    #  求任意两个数据之间的和，得到的矩阵中坐标（i,j）代表total中第i行数据和第j行数据之间的l2 distance(i==j时为0）
    L2_distance = ((total0-total1)**2).sum(2)
    #  调整高斯核函数的sigma值
    if fix_sigma:
        bandwidth = fix_sigma
    else:
        bandwidth = torch.sum(L2_distance.data) / (n_samples**2-n_samples)
    #  以fix_sigma为中值，以kernel_mul为倍数取kernel_num个bandwidth值（比如fix_sigma为1时，得到[0.25,0.5,1,2,4]
    bandwidth /= kernel_mul ** (kernel_num // 2)
    bandwidth_list = [bandwidth * (kernel_mul**i) for i in range(kernel_num)]
    #  高斯核函数的数学表达式
    kernel_val = [torch.exp(-L2_distance / bandwidth_temp) for bandwidth_temp in bandwidth_list]
    #  得到最终的核矩阵
    return sum(kernel_val)  # /len(kernel_val)


def mmd_rbf(source, target, kernel_mul=2.0, kernel_num=5, fix_sigma=None):
    """
    计算源域数据和目标域数据的MMD距离
    Params:
    source: 源域数据（n * len(x))
    target: 目标域数据（m * len(y))
    kernel_mul:
    kernel_num: 取不同高斯核的数量
    fix_sigma: 不同高斯核的sigma值
	Return:
    loss: MMD loss
    """
    batch_size = int(source.size()[0])#一般默认为源域和目标域的batchsize相同
    kernels = guassian_kernel(source, target,
        kernel_mul=kernel_mul, kernel_num=kernel_num, fix_sigma=fix_sigma)
    #根据式（3）将核矩阵分成4部分
    XX = kernels[:batch_size, :batch_size]
    YY = kernels[batch_size:, batch_size:]
    XY = kernels[:batch_size, batch_size:]
    YX = kernels[batch_size:, :batch_size]
    loss = torch.mean(XX + YY - XY -YX)
    return loss#因为一般都是n==m，所以L矩阵一般不加入计算


if __name__ == '__main__':
    isruc_4321_CEA_DCB_MF1 = [0.6986579952263389, 0.7069151430107674, 0.7134966020186452, 0.7194241752781274, 0.7076409703950419, 0.7086152713316418, 0.7239819679899633, 0.7212335371036159, 0.709620688355982, 0.7144844615548416, 0.7216093551854276, 0.7209012590570716, 0.732212171679786, 0.7259383767179688, 0.7222684652449731, 0.7191620157264581, 0.7035685353819581, 0.689611954071482, 0.7241154432096155, 0.7171675547763748, 0.7187245134426027, 0.7208301928014069, 0.7197617299673008, 0.6965926284358144, 0.7268573154763454, 0.7297569103976093, 0.7281082335632234, 0.7190762801905056, 0.7195368374586241, 0.7256770346870509, 0.7272530816162647, 0.7183747307230574, 0.7224817548390103, 0.7462030015273905, 0.7184859017876357, 0.7461608746753146, 0.7187943988012357, 0.7253918058692259, 0.7482844864325535, 0.723875501527582, 0.7414042153324789, 0.7260395576354137, 0.7346101841000842, 0.7175125003489993, 0.724987681294565, 0.7282841448231756, 0.7153578068729904, 0.7199555305733473, 0.7193653424600732]
    isruc_4321_CEA_DCB_AAF1 = [0.6986579952263389, 0.7027865691185531, 0.7063565800852505, 0.7096234788834697, 0.7092269771857842, 0.7091250262100938, 0.7112474464643608, 0.7124957077942677, 0.7121762611900138, 0.7124070812264967, 0.7132436515863995, 0.7138817855422888, 0.7152918152451732, 0.7160522839218013, 0.7164666960100128, 0.7166351534922907, 0.7158665288975652, 0.7144079414072273, 0.7149188625547215, 0.715031297165804, 0.7152071646075564, 0.7154627567981859, 0.7156496686751039, 0.7148556253318002, 0.715335692937582, 0.715890355147583, 0.7163428691629771, 0.7164404909853889, 0.7165472615534314, 0.7168515873245521, 0.7171871193984783, 0.7172242322523714, 0.7173835511186332, 0.7182311820130084, 0.7182384597208549, 0.7190140823584787, 0.7190081449650397, 0.7191761360414657, 0.7199225040002115, 0.7200213289383959, 0.7205428627528858, 0.720673736440565, 0.7209978398745073, 0.720918627612564, 0.7210090510277196, 0.7211672052406644, 0.7210436010200755, 0.7210209328857685, 0.7209871453260605]

    isruc_4321_DCB_MF1 = [0.6986579952263389, 0.6933628941223986, 0.7095841019232431, 0.7157009540746173, 0.7116274115071509, 0.7156567152090814, 0.7222918698830395, 0.7177994811727901, 0.7092227541636559, 0.71928079872624, 0.7025863339177828, 0.7134279729820749, 0.7164186556720014, 0.7183350323191321, 0.7207252087116167, 0.7151341760153191, 0.7334490285051999, 0.5910403182108048, 0.6657452891916359, 0.7209989238536013, 0.7202763867227249, 0.7169876486360629, 0.7068411473896893, 0.7170205099337086, 0.7090982332110791, 0.7260840159742108, 0.7175291578294726, 0.7194188566306604, 0.7311831874281138, 0.6791817785157294, 0.7145272158007181, 0.7168083903309508, 0.7204835773459376, 0.7176194001026205, 0.7420979513252243, 0.715843393739321, 0.7305230568702712, 0.7176327912168599, 0.7315369316636353, 0.7344480708472775, 0.7271290489715049, 0.7595904887433754, 0.6724022358526721, 0.7442861859377192, 0.7182327207481227, 0.7140921632812546, 0.7072498560277244, 0.7112563464734131, 0.7191925266794608, 0.7127499928632173]
    isruc_4321_DCB_AAF1 = [0.6986579952263389, 0.6960104446743687, 0.7005349970906601, 0.7043264863366494, 0.7057866713707497, 0.7074316786771383, 0.7095545631351242, 0.7105851778898324, 0.7104337974758128, 0.7113184976008555, 0.710524664538758, 0.7107666069090345, 0.7112013798908011, 0.7117109264928247, 0.7123118786407442, 0.7124882722266552, 0.7137212578900991, 0.7069056501301383, 0.7047393153439012, 0.7055522957693863, 0.7062534429576405, 0.7067413613975688, 0.7067456999189653, 0.7071738170029129, 0.7072507936512396, 0.7079751483559693, 0.7083290005586916, 0.7087250668469762, 0.70949948479805, 0.7084888945886393, 0.7086836791438677, 0.708937576368464, 0.7092874551859633, 0.7095325123893943, 0.7104629535018466, 0.7106124101751097, 0.7111505357614655, 0.7113211214313443, 0.7118394755398646, 0.7124046904225498, 0.7127638211188658, 0.7138787417765922, 0.7129141718713847, 0.7136271721910742, 0.713729517714564, 0.71373740131384, 0.713599368435412, 0.7135505554778705, 0.7136656977472908, 0.7136473836496093]

    isruc_4321_CEA_MF1 = [0.6986579952263389, 0.7069151430107674, 0.7098898040064082, 0.7228424571840849, 0.7297461180841818, 0.7116637306392473, 0.7402905667143778, 0.7276054830084542, 0.7133972001315458, 0.7263580031780406, 0.738214938258779, 0.6604182096151572, 0.6792286788359067, 0.689862597989346, 0.6529851912080844, 0.7205665597540338, 0.7033957084497009, 0.5631670344766407, 0.7030252257815078, 0.7257760066552579, 0.7205736005383951, 0.7263072343345216, 0.6967684191756274, 0.6819788605573208, 0.726552785086385, 0.7329322886496282, 0.7233036190185617, 0.7097304833048976, 0.7348499901693206, 0.7245166658058861, 0.7424732772742315, 0.6592638339220513, 0.6998244217320899, 0.7235266080155538, 0.7338970618002134, 0.6238245311139299, 0.7526101991851971, 0.7284471612943358, 0.7224274460462932, 0.7232108602586912, 0.6990752402267792, 0.754366725590037, 0.6731334408711029, 0.751356596028244, 0.7223218932192199, 0.7256576725474098, 0.7138374350550553, 0.7088757633908448, 0.7156182803960112, 0.7166243413870319]
    isruc_4321_CEA_AAF1 = [0.6986579952263389, 0.7027865691185531, 0.7051543140811715, 0.7095763498568998, 0.7136103035023561, 0.713285874691838, 0.717143687837915, 0.7184514122342326, 0.7178898331117118, 0.7187366501183446, 0.7205074035856569, 0.7154999707547819, 0.712709871376407, 0.7110779232773312, 0.7072050744727147, 0.7080401673027971, 0.7077669638408502, 0.6997336344317274, 0.6999068760817159, 0.7012003326103929, 0.7021228691783931, 0.7032221585036716, 0.7029415611415828, 0.7020681152839053, 0.7030475020760045, 0.7041969169442208, 0.7049045725766038, 0.7050769265311858, 0.706103583898018, 0.7067173532949469, 0.7078707701975046, 0.7063518034388965, 0.7061540039932357, 0.7066649629350685, 0.7074430229026442, 0.7051202870196243, 0.7064037981592344, 0.7069838866627897, 0.7073798753649309, 0.7077756499872748, 0.7075634448711652, 0.708677808697805, 0.7078511954925328, 0.7088399545956171, 0.7091395532316972, 0.7094986427820389, 0.709590957511252, 0.7095760576337433, 0.7096993683023609, 0.7098378677640543]

    isruc_4321_MF1 = [0.6986579952263389, 0.6933628941223986, 0.7092135210695, 0.7187753908368731, 0.7240983880315608, 0.7123120484702985, 0.7193146484443714, 0.723270370596769, 0.7093810105316719, 0.7204002931140356, 0.7182704597692314, 0.7045691875543151, 0.6815745739060681, 0.7088759836879457, 0.6783474936631209, 0.7189807495125413, 0.7049548315408372, 0.548941120970192, 0.677960607436348, 0.7270776994457444, 0.7228606149414125, 0.7230501559330429, 0.7110609729238846, 0.695996039645134, 0.7181297906148467, 0.7073482334872889, 0.7183806550652075, 0.716855634857859, 0.7293254871741557, 0.7248948601144075, 0.7242047580362933, 0.686391191147704, 0.7014468003324772, 0.7201017306229166, 0.7104035857984827, 0.6716570212161849, 0.7532344047048273, 0.7270993352065596, 0.7234964079979237, 0.7220186547604789, 0.7050975532218782, 0.7605201052345056, 0.685458135591697, 0.7442270037402929, 0.7176051795507942, 0.7164410164911479, 0.7137905842773915, 0.7010764732771724, 0.7129480309535643, 0.7096838721462732]
    isruc_4321_AAF1 = [0.6986579952263389, 0.6960104446743687, 0.7004114701394125, 0.7050024503137776, 0.7088216378573342, 0.709403372959495, 0.7108192694573345, 0.7123756570997637, 0.7120429185921979, 0.7128786560443817, 0.7133688200193681, 0.7126355173139469, 0.7102462139748794, 0.7101483403829556, 0.7080282839349666, 0.708712813033565, 0.7084917552986987, 0.6996278311693371, 0.698487450972864, 0.699916963396508, 0.7010095182319797, 0.7020113654002098, 0.7024048265968913, 0.702137793807235, 0.7027774736795394, 0.7029532721336835, 0.7035246566867029, 0.7040007630499584, 0.7048740293990686, 0.7055413904229133, 0.7061434345394739, 0.7055261769334812, 0.7054025594607235, 0.7058348880243175, 0.7059654222464364, 0.7050124111067072, 0.7063157082309807, 0.7068626457829696, 0.70728915250643, 0.7076573900627812, 0.7075949550178812, 0.7088550776420866, 0.7083109627106822, 0.709127236370446, 0.7093156351077871, 0.7094705347030775, 0.7095624506514671, 0.7093856594561693, 0.7094583609152997, 0.7094628711399191]


    face_432_MF1 = [0.1874819175619343, 0.1659125599149137, 0.25007612859572803, 0.3507575815006989, 0.3812549073707545, 0.1777815814846289, 0.24701092302986982, 0.3077801583652757, 0.38042722904714293, 0.40916159016741677, 0.4070590955781385, 0.42250015980537603, 0.3169122482883763, 0.4014930695439836, 0.30249655898063477, 0.40607698830455985, 0.3358824691192126, 0.3626382618973265, 0.40640672655669396, 0.30489016038551703, 0.17452209213529235, 0.2850712295123456, 0.3452059960277258, 0.3782050036600251, 0.4103612955128933, 0.16525652053762613, 0.2924659362015215, 0.3102106540605896, 0.43100595844871986, 0.2025542094661606, 0.30329970428561515, 0.3893591209187799, 0.41464771309088455, 0.3092797599109554, 0.2874235853338703, 0.3925751029285095, 0.35689348295972373, 0.41688818016751245, 0.4199222023496827, 0.238174235681555, 0.3118128715429788, 0.21832175252873073, 0.3441266003738914, 0.41037754120635506, 0.282976812094594, 0.2641907792251935, 0.3300242469608896, 0.3869260105049441, 0.24303139577500377, 0.3231345429118657, 0.3691033825648271, 0.43541187399192943, 0.4302181086087985, 0.20141461633673152, 0.36365175289136764, 0.25292877284618775, 0.2629698325781283, 0.3008882069335528, 0.27578784826211744, 0.2945532900799306, 0.3985186133423716, 0.4157673549023869]
    face_432_AAF1 = [0.1874819175619343, 0.176697238738424, 0.2011568686908587, 0.23855704689331875, 0.2670966189888059, 0.2522107794047764, 0.25146794277978973, 0.2585069697279755, 0.272053665207883, 0.2857644577038364, 0.2967912429651366, 0.3072669860351565, 0.3080089292854042, 0.3146863678753027, 0.3138737139489915, 0.31963641859621456, 0.32059206862697914, 0.32292796825310954, 0.32732158711119297, 0.32620001577490915, 0.31897725750635597, 0.31743607441571914, 0.3186434623118934, 0.3211251932013988, 0.32469463729385867, 0.31856240203400354, 0.31759586626243014, 0.317332108683793, 0.32125189660672154, 0.3172953070353695, 0.31684383597892585, 0.3191099386332962, 0.32200502270776854, 0.32163075027256816, 0.3206534027028911, 0.3226512277091583, 0.3235766940672816, 0.3260322594909719, 0.3284396939232466, 0.3261830574672043, 0.325832565127589, 0.3232727838752352, 0.32375775635194814, 0.32572638782591196, 0.3247763972541049, 0.32345931860130245, 0.3235989979281022, 0.3249183106901197, 0.3232471491612398, 0.3232448970362523, 0.3241440830270087, 0.32628384823787254, 0.32824487201845604, 0.3258961635799056, 0.32658262884011396, 0.32526738141165107, 0.3241744419584314, 0.32377295514765764, 0.32295964825129253, 0.32248620894843655, 0.3237326418073535, 0.32521707266372496]

    bci_4321_MF1 = [0.46335815763159616, 0.47341951753990175, 0.45765103389474315, 0.4731562261196499, 0.47977671167969677, 0.4734586396034117, 0.4818724978307587, 0.4763270210079966, 0.45795657334510026, 0.48467834430956974, 0.4868811692731768, 0.48623028170726706, 0.48747918981574434, 0.47977962147714903, 0.48032471545845956, 0.47949035488947445, 0.48000234794111596, 0.48032725191537096, 0.4779884198833207, 0.4866463013207279, 0.4809843498786148, 0.4930855328127771, 0.4833148372987554, 0.49415896402935733, 0.4858490381360373, 0.46355065345962454, 0.4875785481102761, 0.47567571041148243, 0.4903187468392433, 0.4818361120203783, 0.48822466299894385, 0.48274696925432226, 0.49077821766448587, 0.48014119600592475, 0.48618149831341007, 0.4905642088487833, 0.489955445166493, 0.48815537583, 0.48066867572817207, 0.48381451555026145, 0.48404931890637004, 0.488995951755747, 0.46517365065327465, 0.4848854207132525, 0.48985929046981824, 0.483520093913355, 0.48870317562655474, 0.489865543655279, 0.48971391810081766, 0.4818396287104281, 0.48997191360919845, 0.49307870438783186]
    bci_4321_AAF1 = [0.46335815763159616, 0.46838883758574895, 0.46480956968874704, 0.4668962337964727, 0.4694723293731175, 0.47013671441149985, 0.47181325489996545, 0.4723774756634693, 0.4707751531836506, 0.4721654722962425, 0.4735032629305093, 0.47456384782857247, 0.4755573356737395, 0.47585892751684017, 0.47615664671294816, 0.47636500347398103, 0.4765789649132243, 0.4767872030800102, 0.47685042501702657, 0.47734021883221167, 0.4775137488820404, 0.47822155724252846, 0.4784430042014948, 0.479097835860989, 0.47936788395199087, 0.4787595289330537, 0.47908615927295084, 0.4789643575278984, 0.47935588819380687, 0.47943856232135923, 0.4797219849238619, 0.4798165156841889, 0.4801486884714707, 0.48014846810483697, 0.4803208403965105, 0.4806053784090737, 0.480858082916031, 0.4810501169400828, 0.48104033639618765, 0.4811096908750393, 0.48118138911970587, 0.4813674501348497, 0.48099085014690607, 0.48107936311432303, 0.48127447261111184, 0.4813232904655084, 0.4814803092987222, 0.48165500168115055, 0.4818194693631846, 0.4818198725501295, 0.4819797164924642, 0.4821931585673751]


    print(compute_aaf1(bci_4321_MF1))



