#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# Copyright (c) BaseDetection, Inc. and its affiliates. All Rights Reserved

import logging
import os
import os.path as osp
from copy import deepcopy

import numpy as np

import torch

from cvpods.utils import Timer

from cvpods.data.base_dataset import BaseDataset
from cvpods.data.detection_utils import annotations_to_instances, filter_empty_instances
from cvpods.data.registry import DATASETS
from .paths_route import _PREDEFINED_SPLITS_IMAGENET

"""
This file contains functions to parse ImageNet-format annotations into dicts in "cvpods format".
"""

logger = logging.getLogger(__name__)


@DATASETS.register()
class ImageNetDataset(BaseDataset):
    def __init__(self, cfg, dataset_name, transforms=[], is_train=True):
        super(ImageNetDataset, self).__init__(cfg, dataset_name, transforms, is_train)

        image_root, label_file = _PREDEFINED_SPLITS_IMAGENET["imagenet"][self.name]
        self.label_file = osp.join(self.data_root, image_root, label_file) \
            if "://" not in image_root else osp.join(image_root, label_file)
        self.image_root = osp.join(self.data_root, image_root) \
            if "://" not in image_root else image_root

        self.meta = self._get_metadata()
        self.dataset_dicts = self._load_annotations()
        self._set_group_flag()

        self.eval_with_gt = cfg.TEST.get("WITH_GT", False)

    def __getitem__(self, index):
        """Load data, apply transforms, converto to Instances.
        """
        dataset_dict = deepcopy(self.dataset_dicts[index])

        # read image
        image = self._read_data(dataset_dict["file_name"])

        annotations = dataset_dict.get("annotations", None)

        # apply transfrom
        images, annotations = self._apply_transforms(
            image, annotations)

        def process(dd, img, annos):
            if isinstance(annos, list):
                annos = [a for a in annos if a is not None]

            if len(img.shape) == 3:
                image_shape = img.shape[:2]  # h, w
                dd["image"] = torch.as_tensor(
                    np.ascontiguousarray(img.transpose(2, 0, 1)))
            elif len(img.shape) == 4:
                image_shape = img.shape[1:3]
                # NHWC -> NCHW
                dd["image"] = torch.as_tensor(
                    np.ascontiguousarray(img.transpose(0, 3, 1, 2)))

            if isinstance(annos, list) and len(annos) > 0:
                instances = annotations_to_instances(annos, image_shape)
                dd["instances"] = filter_empty_instances(instances)

            return dd

        if isinstance(images, dict):
            ret = {}
            # multiple input pipelines
            for desc, item in images.items():
                img, anno = item
                ret[desc] = process(deepcopy(dataset_dict), img, anno)
            return ret
        else:
            return process(dataset_dict, images, annotations)

    def __len__(self):
        return len(self.dataset_dicts)

    def _get_metadata(self):
        assert len(IMAGENET_CATEGORIES.keys()) == 1000
        cat_ids = [v[0] for v in IMAGENET_CATEGORIES.values()]
        assert min(cat_ids) == 1 and max(cat_ids) == len(cat_ids), \
            "Category ids are not in [1, #categories], as expected"
        # Ensure that the category list is sroted by id
        imagenet_categories = sorted(IMAGENET_CATEGORIES.items(), key=lambda x: x[1][0])
        thing_classes = [v[1][1] for v in imagenet_categories]
        meta = {
            "thing_classes": thing_classes,
            "evaluator_type": _PREDEFINED_SPLITS_IMAGENET["evaluator_type"]["imagenet"],
        }
        return meta

    def _load_annotations(self):
        timer = Timer()

        """Constructs the imdb."""
        # Compile the split data path
        logger.info('{} data path: {}'.format(self.name, self.label_file))
        # Images are stored per class in subdirs (format: n<number>)
        class_ids = [k for k, v in IMAGENET_CATEGORIES.items()]
        class_id_cont_id = {k: v[0] for k, v in IMAGENET_CATEGORIES.items()}
        # class_ids = sorted([
        #     f for f in os.listdir(split_path) if re.match(r'^n[0-9]+$', f)
        # ])
        # # Map ImageNet class ids to contiguous ids
        # class_id_cont_id = {v: i for i, v in enumerate(class_ids)}
        # Construct the image db
        imdb = []
        if "://" not in self.image_root:
            for class_id in class_ids:
                cont_id = class_id_cont_id[class_id]
                im_dir = os.path.join(self.label_file, class_id)
                for im_name in os.listdir(im_dir):
                    imdb.append({
                        'im_path': os.path.join(im_dir, im_name),
                        'class': cont_id,
                    })

        logging.info("Loading {} takes {:.2f} seconds.".format(self.label_file, timer.seconds()))

        dataset_dicts = []
        for i, item in enumerate(imdb):
            dataset_dicts.append({
                "image_id": i,
                "category_id": item["class"],
                "file_name": item["im_path"],
            })

        return dataset_dicts


IMAGENET_CATEGORIES = {
    'n01440764': (449, 'tench'),
    'n01443537': (450, 'goldfish'),
    'n01484850': (442, 'great_white_shark'),
    'n01491361': (443, 'tiger_shark'),
    'n01494475': (444, 'hammerhead'),
    'n01496331': (445, 'electric_ray'),
    'n01498041': (446, 'stingray'),
    'n01514668': (383, 'cock'),
    'n01514859': (384, 'hen'),
    'n01518878': (385, 'ostrich'),
    'n01530575': (386, 'brambling'),
    'n01531178': (387, 'goldfinch'),
    'n01532829': (388, 'house_finch'),
    'n01534433': (389, 'junco'),
    'n01537544': (390, 'indigo_bunting'),
    'n01558993': (391, 'robin'),
    'n01560419': (392, 'bulbul'),
    'n01580077': (393, 'jay'),
    'n01582220': (394, 'magpie'),
    'n01592084': (395, 'chickadee'),
    'n01601694': (396, 'water_ouzel'),
    'n01608432': (397, 'kite'),
    'n01614925': (398, 'bald_eagle'),
    'n01616318': (399, 'vulture'),
    'n01622779': (400, 'great_grey_owl'),
    'n01629819': (494, 'European_fire_salamander'),
    'n01630670': (495, 'common_newt'),
    'n01631663': (496, 'eft'),
    'n01632458': (497, 'spotted_salamander'),
    'n01632777': (498, 'axolotl'),
    'n01641577': (499, 'bullfrog'),
    'n01644373': (500, 'tree_frog'),
    'n01644900': (501, 'tailed_frog'),
    'n01664065': (458, 'loggerhead'),
    'n01665541': (459, 'leatherback_turtle'),
    'n01667114': (460, 'mud_turtle'),
    'n01667778': (461, 'terrapin'),
    'n01669191': (462, 'box_turtle'),
    'n01675722': (463, 'banded_gecko'),
    'n01677366': (464, 'common_iguana'),
    'n01682714': (465, 'American_chameleon'),
    'n01685808': (466, 'whiptail'),
    'n01687978': (467, 'agama'),
    'n01688243': (468, 'frilled_lizard'),
    'n01689811': (469, 'alligator_lizard'),
    'n01692333': (470, 'Gila_monster'),
    'n01693334': (471, 'green_lizard'),
    'n01694178': (472, 'African_chameleon'),
    'n01695060': (473, 'Komodo_dragon'),
    'n01697457': (475, 'African_crocodile'),
    'n01698640': (476, 'American_alligator'),
    'n01704323': (474, 'triceratops'),
    'n01728572': (477, 'thunder_snake'),
    'n01728920': (478, 'ringneck_snake'),
    'n01729322': (479, 'hognose_snake'),
    'n01729977': (480, 'green_snake'),
    'n01734418': (481, 'king_snake'),
    'n01735189': (482, 'garter_snake'),
    'n01737021': (483, 'water_snake'),
    'n01739381': (484, 'vine_snake'),
    'n01740131': (485, 'night_snake'),
    'n01742172': (486, 'boa_constrictor'),
    'n01744401': (487, 'rock_python'),
    'n01748264': (488, 'Indian_cobra'),
    'n01749939': (489, 'green_mamba'),
    'n01751748': (490, 'sea_snake'),
    'n01753488': (491, 'horned_viper'),
    'n01755581': (492, 'diamondback'),
    'n01756291': (493, 'sidewinder'),
    'n01768244': (601, 'trilobite'),
    'n01770081': (602, 'harvestman'),
    'n01770393': (603, 'scorpion'),
    'n01773157': (604, 'black_and_gold_garden_spider'),
    'n01773549': (605, 'barn_spider'),
    'n01773797': (606, 'garden_spider'),
    'n01774384': (607, 'black_widow'),
    'n01774750': (608, 'tarantula'),
    'n01775062': (609, 'wolf_spider'),
    'n01776313': (610, 'tick'),
    'n01784675': (611, 'centipede'),
    'n01795545': (401, 'black_grouse'),
    'n01796340': (402, 'ptarmigan'),
    'n01797886': (403, 'ruffed_grouse'),
    'n01798484': (404, 'prairie_chicken'),
    'n01806143': (405, 'peacock'),
    'n01806567': (406, 'quail'),
    'n01807496': (407, 'partridge'),
    'n01817953': (408, 'African_grey'),
    'n01818515': (409, 'macaw'),
    'n01819313': (410, 'sulphur-crested_cockatoo'),
    'n01820546': (411, 'lorikeet'),
    'n01824575': (412, 'coucal'),
    'n01828970': (413, 'bee_eater'),
    'n01829413': (414, 'hornbill'),
    'n01833805': (415, 'hummingbird'),
    'n01843065': (416, 'jacamar'),
    'n01843383': (417, 'toucan'),
    'n01847000': (418, 'drake'),
    'n01855032': (419, 'red-breasted_merganser'),
    'n01855672': (420, 'goose'),
    'n01860187': (421, 'black_swan'),
    'n01871265': (214, 'tusker'),
    'n01872401': (215, 'echidna'),
    'n01873310': (217, 'platypus'),
    'n01877812': (216, 'wallaby'),
    'n01882714': (213, 'koala'),
    'n01883070': (218, 'wombat'),
    'n01910747': (647, 'jellyfish'),
    'n01914609': (648, 'sea_anemone'),
    'n01917289': (649, 'brain_coral'),
    'n01924916': (650, 'flatworm'),
    'n01930112': (651, 'nematode'),
    'n01943899': (652, 'conch'),
    'n01944390': (653, 'snail'),
    'n01945685': (654, 'slug'),
    'n01950731': (655, 'sea_slug'),
    'n01955084': (656, 'chiton'),
    'n01968897': (226, 'chambered_nautilus'),
    'n01978287': (613, 'Dungeness_crab'),
    'n01978455': (614, 'rock_crab'),
    'n01980166': (615, 'fiddler_crab'),
    'n01981276': (616, 'king_crab'),
    'n01983481': (617, 'American_lobster'),
    'n01984695': (618, 'spiny_lobster'),
    'n01985128': (619, 'crayfish'),
    'n01986214': (620, 'hermit_crab'),
    'n01990800': (612, 'isopod'),
    'n02002556': (422, 'white_stork'),
    'n02002724': (423, 'black_stork'),
    'n02006656': (424, 'spoonbill'),
    'n02007558': (425, 'flamingo'),
    'n02009229': (427, 'little_blue_heron'),
    'n02009912': (426, 'American_egret'),
    'n02011460': (428, 'bittern'),
    'n02012849': (429, 'crane'),
    'n02013706': (430, 'limpkin'),
    'n02017213': (438, 'European_gallinule'),
    'n02018207': (431, 'American_coot'),
    'n02018795': (432, 'bustard'),
    'n02025239': (433, 'ruddy_turnstone'),
    'n02027492': (434, 'red-backed_sandpiper'),
    'n02028035': (435, 'redshank'),
    'n02033041': (436, 'dowitcher'),
    'n02037110': (437, 'oystercatcher'),
    'n02051845': (439, 'pelican'),
    'n02056570': (440, 'king_penguin'),
    'n02058221': (441, 'albatross'),
    'n02066245': (6, 'grey_whale'),
    'n02071294': (22, 'killer_whale'),
    'n02074367': (193, 'dugong'),
    'n02077923': (14, 'sea_lion'),
    'n02085620': (173, 'Chihuahua'),
    'n02085782': (99, 'Japanese_spaniel'),
    'n02085936': (87, 'Maltese_dog'),
    'n02086079': (69, 'Pekinese'),
    'n02086240': (116, 'Shih-Tzu'),
    'n02086646': (198, 'Blenheim_spaniel'),
    'n02086910': (43, 'papillon'),
    'n02087046': (89, 'toy_terrier'),
    'n02087394': (200, 'Rhodesian_ridgeback'),
    'n02088094': (98, 'Afghan_hound'),
    'n02088238': (161, 'basset'),
    'n02088364': (132, 'beagle'),
    'n02088466': (32, 'bloodhound'),
    'n02088632': (180, 'bluetick'),
    'n02089078': (42, 'black-and-tan_coonhound'),
    'n02089867': (18, 'Walker_hound'),
    'n02089973': (207, 'English_foxhound'),
    'n02090379': (181, 'redbone'),
    'n02090622': (105, 'borzoi'),
    'n02090721': (77, 'Irish_wolfhound'),
    'n02091032': (189, 'Italian_greyhound'),
    'n02091134': (20, 'whippet'),
    'n02091244': (204, 'Ibizan_hound'),
    'n02091467': (63, 'Norwegian_elkhound'),
    'n02091635': (31, 'otterhound'),
    'n02091831': (66, 'Saluki'),
    'n02092002': (21, 'Scottish_deerhound'),
    'n02092339': (25, 'Weimaraner'),
    'n02093256': (45, 'Staffordshire_bullterrier'),
    'n02093428': (170, 'American_Staffordshire_terrier'),
    'n02093647': (119, 'Bedlington_terrier'),
    'n02093754': (210, 'Border_terrier'),
    'n02093859': (107, 'Kerry_blue_terrier'),
    'n02093991': (126, 'Irish_terrier'),
    'n02094114': (88, 'Norfolk_terrier'),
    'n02094258': (145, 'Norwich_terrier'),
    'n02094433': (59, 'Yorkshire_terrier'),
    'n02095314': (160, 'wire-haired_fox_terrier'),
    'n02095570': (152, 'Lakeland_terrier'),
    'n02095889': (72, 'Sealyham_terrier'),
    'n02096051': (33, 'Airedale'),
    'n02096177': (91, 'cairn'),
    'n02096294': (4, 'Australian_terrier'),
    'n02096437': (27, 'Dandie_Dinmont'),
    'n02096585': (113, 'Boston_bull'),
    'n02097047': (123, 'miniature_schnauzer'),
    'n02097130': (36, 'giant_schnauzer'),
    'n02097209': (156, 'standard_schnauzer'),
    'n02097298': (109, 'Scotch_terrier'),
    'n02097474': (158, 'Tibetan_terrier'),
    'n02097658': (131, 'silky_terrier'),
    'n02098105': (26, 'soft-coated_wheaten_terrier'),
    'n02098286': (71, 'West_Highland_white_terrier'),
    'n02098413': (56, 'Lhasa'),
    'n02099267': (146, 'flat-coated_retriever'),
    'n02099429': (144, 'curly-coated_retriever'),
    'n02099601': (125, 'golden_retriever'),
    'n02099712': (176, 'Labrador_retriever'),
    'n02099849': (139, 'Chesapeake_Bay_retriever'),
    'n02100236': (134, 'German_short-haired_pointer'),
    'n02100583': (90, 'vizsla'),
    'n02100735': (2, 'English_setter'),
    'n02100877': (192, 'Irish_setter'),
    'n02101006': (154, 'Gordon_setter'),
    'n02101388': (150, 'Brittany_spaniel'),
    'n02101556': (94, 'clumber'),
    'n02102040': (5, 'English_springer'),
    'n02102177': (19, 'Welsh_springer_spaniel'),
    'n02102318': (191, 'cocker_spaniel'),
    'n02102480': (196, 'Sussex_spaniel'),
    'n02102973': (117, 'Irish_water_spaniel'),
    'n02104029': (141, 'kuvasz'),
    'n02104365': (68, 'schipperke'),
    'n02105056': (93, 'groenendael'),
    'n02105162': (51, 'malinois'),
    'n02105251': (208, 'briard'),
    'n02105412': (184, 'kelpie'),
    'n02105505': (97, 'komondor'),
    'n02105641': (29, 'Old_English_sheepdog'),
    'n02105855': (171, 'Shetland_sheepdog'),
    'n02106030': (124, 'collie'),
    'n02106166': (128, 'Border_collie'),
    'n02106382': (47, 'Bouvier_des_Flandres'),
    'n02106550': (64, 'Rottweiler'),
    'n02106662': (211, 'German_shepherd'),
    'n02107142': (112, 'Doberman'),
    'n02107312': (187, 'miniature_pinscher'),
    'n02107574': (114, 'Greater_Swiss_Mountain_dog'),
    'n02107683': (86, 'Bernese_mountain_dog'),
    'n02107908': (115, 'Appenzeller'),
    'n02108000': (79, 'EntleBucher'),
    'n02108089': (130, 'boxer'),
    'n02108422': (140, 'bull_mastiff'),
    'n02108551': (110, 'Tibetan_mastiff'),
    'n02108915': (82, 'French_bulldog'),
    'n02109047': (17, 'Great_Dane'),
    'n02109525': (177, 'Saint_Bernard'),
    'n02109961': (149, 'Eskimo_dog'),
    'n02110063': (15, 'malamute'),
    'n02110185': (3, 'Siberian_husky'),
    'n02110341': (41, 'dalmatian'),
    'n02110627': (127, 'affenpinscher'),
    'n02110806': (84, 'basenji'),
    'n02110958': (143, 'pug'),
    'n02111129': (133, 'Leonberg'),
    'n02111277': (60, 'Newfoundland'),
    'n02111500': (172, 'Great_Pyrenees'),
    'n02111889': (179, 'Samoyed'),
    'n02112018': (118, 'Pomeranian'),
    'n02112137': (168, 'chow'),
    'n02112350': (148, 'keeshond'),
    'n02112706': (70, 'Brabancon_griffon'),
    'n02113023': (197, 'Pembroke'),
    'n02113186': (50, 'Cardigan'),
    'n02113624': (106, 'toy_poodle'),
    'n02113712': (49, 'miniature_poodle'),
    'n02113799': (151, 'standard_poodle'),
    'n02113978': (46, 'Mexican_hairless'),
    'n02114367': (205, 'timber_wolf'),
    'n02114548': (102, 'white_wolf'),
    'n02114712': (28, 'red_wolf'),
    'n02114855': (58, 'coyote'),
    'n02115641': (155, 'dingo'),
    'n02115913': (136, 'dhole'),
    'n02116738': (202, 'African_hunting_dog'),
    'n02117135': (34, 'hyena'),
    'n02119022': (62, 'red_fox'),
    'n02119789': (1, 'kit_fox'),
    'n02120079': (159, 'Arctic_fox'),
    'n02120505': (67, 'grey_fox'),
    'n02123045': (174, 'tabby'),
    'n02123159': (55, 'tiger_cat'),
    'n02123394': (10, 'Persian_cat'),
    'n02123597': (95, 'Siamese_cat'),
    'n02124075': (8, 'Egyptian_cat'),
    'n02125311': (11, 'cougar'),
    'n02127052': (201, 'lynx'),
    'n02128385': (85, 'leopard'),
    'n02128757': (153, 'snow_leopard'),
    'n02128925': (30, 'jaguar'),
    'n02129165': (190, 'lion'),
    'n02129604': (76, 'tiger'),
    'n02130308': (206, 'cheetah'),
    'n02132136': (61, 'brown_bear'),
    'n02133161': (163, 'American_black_bear'),
    'n02134084': (103, 'ice_bear'),
    'n02134418': (209, 'sloth_bear'),
    'n02137549': (74, 'mongoose'),
    'n02138441': (35, 'meerkat'),
    'n02165105': (621, 'tiger_beetle'),
    'n02165456': (622, 'ladybug'),
    'n02167151': (623, 'ground_beetle'),
    'n02168699': (624, 'long-horned_beetle'),
    'n02169497': (625, 'leaf_beetle'),
    'n02172182': (626, 'dung_beetle'),
    'n02174001': (627, 'rhinoceros_beetle'),
    'n02177972': (628, 'weevil'),
    'n02190166': (629, 'fly'),
    'n02206856': (630, 'bee'),
    'n02219486': (224, 'ant'),
    'n02226429': (631, 'grasshopper'),
    'n02229544': (632, 'cricket'),
    'n02231487': (633, 'walking_stick'),
    'n02233338': (634, 'cockroach'),
    'n02236044': (635, 'mantis'),
    'n02256656': (636, 'cicada'),
    'n02259212': (637, 'leafhopper'),
    'n02264363': (638, 'lacewing'),
    'n02268443': (639, 'dragonfly'),
    'n02268853': (640, 'damselfly'),
    'n02276258': (641, 'admiral'),
    'n02277742': (642, 'ringlet'),
    'n02279972': (643, 'monarch'),
    'n02280649': (644, 'cabbage_butterfly'),
    'n02281406': (645, 'sulphur_butterfly'),
    'n02281787': (646, 'lycaenid'),
    'n02317335': (225, 'starfish'),
    'n02319095': (657, 'sea_urchin'),
    'n02321529': (658, 'sea_cucumber'),
    'n02325366': (188, 'wood_rabbit'),
    'n02326432': (129, 'hare'),
    'n02328150': (164, 'Angora'),
    'n02342885': (157, 'hamster'),
    'n02346627': (13, 'porcupine'),
    'n02356798': (53, 'fox_squirrel'),
    'n02361337': (183, 'marmot'),
    'n02363005': (195, 'beaver'),
    'n02364673': (101, 'guinea_pig'),
    'n02389026': (39, 'sorrel'),
    'n02391049': (80, 'zebra'),
    'n02395406': (147, 'hog'),
    'n02396427': (78, 'wild_boar'),
    'n02397096': (120, 'warthog'),
    'n02398521': (167, 'hippopotamus'),
    'n02403003': (108, 'ox'),
    'n02408429': (162, 'water_buffalo'),
    'n02410509': (165, 'bison'),
    'n02412080': (81, 'ram'),
    'n02415577': (52, 'bighorn'),
    'n02417914': (9, 'ibex'),
    'n02422106': (65, 'hartebeest'),
    'n02422699': (57, 'impala'),
    'n02423022': (12, 'gazelle'),
    'n02437312': (121, 'Arabian_camel'),
    'n02437616': (186, 'llama'),
    'n02441942': (48, 'weasel'),
    'n02442845': (23, 'mink'),
    'n02443114': (182, 'polecat'),
    'n02443484': (40, 'black-footed_ferret'),
    'n02444819': (212, 'otter'),
    'n02445715': (44, 'skunk'),
    'n02447366': (16, 'badger'),
    'n02454379': (178, 'armadillo'),
    'n02457408': (38, 'three-toed_sloth'),
    'n02480495': (83, 'orangutan'),
    'n02480855': (104, 'gorilla'),
    'n02481823': (96, 'chimpanzee'),
    'n02483362': (185, 'gibbon'),
    'n02483708': (122, 'siamang'),
    'n02484975': (73, 'guenon'),
    'n02486261': (135, 'patas'),
    'n02486410': (137, 'baboon'),
    'n02487347': (138, 'macaque'),
    'n02488291': (203, 'langur'),
    'n02488702': (54, 'colobus'),
    'n02489166': (100, 'proboscis_monkey'),
    'n02490219': (175, 'marmoset'),
    'n02492035': (142, 'capuchin'),
    'n02492660': (166, 'howler_monkey'),
    'n02493509': (37, 'titi'),
    'n02493793': (111, 'spider_monkey'),
    'n02494079': (92, 'squirrel_monkey'),
    'n02497673': (199, 'Madagascar_cat'),
    'n02500267': (75, 'indri'),
    'n02504013': (194, 'Indian_elephant'),
    'n02504458': (24, 'African_elephant'),
    'n02509815': (7, 'lesser_panda'),
    'n02510455': (169, 'giant_panda'),
    'n02514041': (447, 'barracouta'),
    'n02526121': (451, 'eel'),
    'n02536864': (448, 'coho'),
    'n02606052': (452, 'rock_beauty'),
    'n02607072': (453, 'anemone_fish'),
    'n02640242': (456, 'sturgeon'),
    'n02641379': (457, 'gar'),
    'n02643566': (454, 'lionfish'),
    'n02655020': (455, 'puffer'),
    'n02666196': (547, 'abacus'),
    'n02667093': (853, 'abaya'),
    'n02669723': (896, 'academic_gown'),
    'n02672831': (223, 'accordion'),
    'n02676566': (345, 'acoustic_guitar'),
    'n02687172': (246, 'aircraft_carrier'),
    'n02690373': (230, 'airliner'),
    'n02692877': (232, 'airship'),
    'n02699494': (677, 'altar'),
    'n02701002': (265, 'ambulance'),
    'n02704792': (264, 'amphibian'),
    'n02708093': (522, 'analog_clock'),
    'n02727426': (688, 'apiary'),
    'n02730930': (845, 'apron'),
    'n02747177': (752, 'ashcan'),
    'n02749479': (540, 'assault_rifle'),
    'n02769748': (847, 'backpack'),
    'n02776631': (704, 'bakery'),
    'n02777292': (767, 'balance_beam'),
    'n02782093': (233, 'balloon'),
    'n02783161': (907, 'ballpoint'),
    'n02786058': (967, 'Band_Aid'),
    'n02787622': (341, 'banjo'),
    'n02788148': (718, 'bannister'),
    'n02790996': (916, 'barbell'),
    'n02791124': (307, 'barber_chair'),
    'n02791270': (705, 'barbershop'),
    'n02793495': (683, 'barn'),
    'n02794156': (518, 'barometer'),
    'n02795169': (905, 'barrel'),
    'n02797295': (258, 'barrow'),
    'n02799071': (807, 'baseball'),
    'n02802426': (908, 'basketball'),
    'n02804414': (296, 'bassinet'),
    'n02804610': (353, 'bassoon'),
    'n02807133': (785, 'bathing_cap'),
    'n02808304': (909, 'bath_towel'),
    'n02808440': (884, 'bathtub'),
    'n02814533': (266, 'beach_wagon'),
    'n02814860': (733, 'beacon'),
    'n02815834': (991, 'beaker'),
    'n02817516': (849, 'bearskin'),
    'n02823428': (777, 'beer_bottle'),
    'n02823750': (811, 'beer_glass'),
    'n02825657': (933, 'bell_cote'),
    'n02834397': (941, 'bib'),
    'n02835271': (254, 'bicycle-built-for-two'),
    'n02837789': (985, 'bikini'),
    'n02840245': (835, 'binder'),
    'n02841315': (533, 'binoculars'),
    'n02843684': (839, 'birdhouse'),
    'n02859443': (689, 'boathouse'),
    'n02860847': (252, 'bobsled'),
    'n02865351': (940, 'bolo_tie'),
    'n02869837': (805, 'bonnet'),
    'n02870880': (300, 'bookcase'),
    'n02871525': (706, 'bookshop'),
    'n02877765': (779, 'bottlecap'),
    'n02879718': (538, 'bow'),
    'n02883205': (817, 'bow_tie'),
    'n02892201': (716, 'brass'),
    'n02892767': (872, 'brassiere'),
    'n02894605': (719, 'breakwater'),
    'n02895154': (949, 'breastplate'),
    'n02906734': (851, 'broom'),
    'n02909870': (820, 'bucket'),
    'n02910353': (580, 'buckle'),
    'n02916936': (833, 'bulletproof_vest'),
    'n02917067': (887, 'bullet_train'),
    'n02927161': (707, 'butcher_shop'),
    'n02930766': (267, 'cab'),
    'n02939185': (673, 'caldron'),
    'n02948072': (591, 'candle'),
    'n02950826': (539, 'cannon'),
    'n02951358': (239, 'canoe'),
    'n02951585': (377, 'can_opener'),
    'n02963159': (836, 'cardigan'),
    'n02965783': (576, 'car_mirror'),
    'n02966193': (568, 'carousel'),
    'n02966687': (891, "carpenter's_kit"),
    'n02971356': (749, 'carton'),
    'n02974003': (563, 'car_wheel'),
    'n02977058': (548, 'cash_machine'),
    'n02978881': (890, 'cassette'),
    'n02979186': (929, 'cassette_player'),
    'n02980441': (701, 'castle'),
    'n02981792': (241, 'catamaran'),
    'n02988304': (987, 'CD_player'),
    'n02992211': (342, 'cello'),
    'n02992529': (914, 'cellular_telephone'),
    'n02999410': (827, 'chain'),
    'n03000134': (721, 'chainlink_fence'),
    'n03000247': (902, 'chain_mail'),
    'n03000684': (382, 'chain_saw'),
    'n03014705': (762, 'chest'),
    'n03016953': (303, 'chiffonier'),
    'n03017168': (335, 'chime'),
    'n03018349': (301, 'china_cabinet'),
    'n03026506': (801, 'Christmas_stocking'),
    'n03028079': (690, 'church'),
    'n03032252': (695, 'cinema'),
    'n03041632': (370, 'cleaver'),
    'n03042490': (713, 'cliff_dwelling'),
    'n03045698': (797, 'cloak'),
    'n03047690': (979, 'clog'),
    'n03062245': (761, 'cocktail_shaker'),
    'n03063599': (996, 'coffee_mug'),
    'n03063689': (674, 'coffeepot'),
    'n03065424': (698, 'coil'),
    'n03075370': (583, 'combination_lock'),
    'n03085013': (543, 'computer_keyboard'),
    'n03089624': (708, 'confectionery'),
    'n03095699': (243, 'container_ship'),
    'n03100240': (268, 'convertible'),
    'n03109150': (376, 'corkscrew'),
    'n03110669': (347, 'cornet'),
    'n03124043': (910, 'cowboy_boot'),
    'n03124170': (881, 'cowboy_hat'),
    'n03125729': (297, 'cradle'),
    'n03126707': (545, 'crane'),
    'n03127747': (778, 'crash_helmet'),
    'n03127925': (898, 'crate'),
    'n03131574': (298, 'crib'),
    'n03133878': (670, 'Crock_Pot'),
    'n03134739': (756, 'croquet_ball'),
    'n03141823': (856, 'crutch'),
    'n03146219': (865, 'cuirass'),
    'n03160309': (720, 'dam'),
    'n03179701': (313, 'desk'),
    'n03180011': (550, 'desktop_computer'),
    'n03187595': (959, 'dial_telephone'),
    'n03188531': (966, 'diaper'),
    'n03196217': (523, 'digital_clock'),
    'n03197337': (529, 'digital_watch'),
    'n03201208': (315, 'dining_table'),
    'n03207743': (821, 'dishrag'),
    'n03207941': (667, 'dishwasher'),
    'n03208938': (579, 'disk_brake'),
    'n03216828': (715, 'dock'),
    'n03218198': (253, 'dogsled'),
    'n03220513': (897, 'dome'),
    'n03223299': (972, 'doormat'),
    'n03240683': (834, 'drilling_platform'),
    'n03249569': (336, 'drum'),
    'n03250847': (799, 'drumstick'),
    'n03255030': (1000, 'dumbbell'),
    'n03259280': (662, 'Dutch_oven'),
    'n03271574': (512, 'electric_fan'),
    'n03272010': (346, 'electric_guitar'),
    'n03272562': (262, 'electric_locomotive'),
    'n03290653': (316, 'entertainment_center'),
    'n03291819': (879, 'envelope'),
    'n03297495': (660, 'espresso_maker'),
    'n03314780': (808, 'face_powder'),
    'n03325584': (796, 'feather_boa'),
    'n03337140': (305, 'file'),
    'n03344393': (235, 'fireboat'),
    'n03345487': (279, 'fire_engine'),
    'n03347037': (919, 'fire_screen'),
    'n03355925': (995, 'flagpole'),
    'n03372029': (356, 'flute'),
    'n03376595': (309, 'folding_chair'),
    'n03379051': (784, 'football_helmet'),
    'n03384352': (261, 'forklift'),
    'n03388043': (712, 'fountain'),
    'n03388183': (934, 'fountain_pen'),
    'n03388549': (299, 'four-poster'),
    'n03393912': (256, 'freight_car'),
    'n03394916': (348, 'French_horn'),
    'n03400231': (671, 'frying_pan'),
    'n03404251': (757, 'fur_coat'),
    'n03417042': (280, 'garbage_truck'),
    'n03424325': (971, 'gasmask'),
    'n03425413': (567, 'gas_pump'),
    'n03443371': (955, 'goblet'),
    'n03444034': (275, 'go-kart'),
    'n03445777': (792, 'golf_ball'),
    'n03445924': (276, 'golfcart'),
    'n03447447': (236, 'gondola'),
    'n03447721': (337, 'gong'),
    'n03450230': (911, 'gown'),
    'n03452741': (227, 'grand_piano'),
    'n03457902': (684, 'greenhouse'),
    'n03459775': (725, 'grille'),
    'n03461385': (703, 'grocery_store'),
    'n03467068': (517, 'guillotine'),
    'n03476684': (581, 'hair_slide'),
    'n03476991': (895, 'hair_spray'),
    'n03478589': (249, 'half_track'),
    'n03481172': (375, 'hammer'),
    'n03482405': (840, 'hamper'),
    'n03483316': (505, 'hand_blower'),
    'n03485407': (551, 'hand-held_computer'),
    'n03485794': (750, 'handkerchief'),
    'n03492542': (573, 'hard_disc'),
    'n03494278': (350, 'harmonica'),
    'n03495258': (344, 'harp'),
    'n03496892': (554, 'harvester'),
    'n03498962': (369, 'hatchet'),
    'n03527444': (787, 'holster'),
    'n03529860': (696, 'home_theater'),
    'n03530642': (730, 'honeycomb'),
    'n03532672': (562, 'hook'),
    'n03534580': (802, 'hoopskirt'),
    'n03535780': (924, 'horizontal_bar'),
    'n03538406': (293, 'horse_cart'),
    'n03544143': (525, 'hourglass'),
    'n03584254': (980, 'iPod'),
    'n03584829': (659, 'iron'),
    'n03590841': (592, "jack-o'-lantern"),
    'n03594734': (748, 'jean'),
    'n03594945': (269, 'jeep'),
    'n03595614': (961, 'jersey'),
    'n03598930': (963, 'jigsaw_puzzle'),
    'n03599486': (294, 'jinrikisha'),
    'n03602883': (560, 'joystick'),
    'n03617480': (770, 'kimono'),
    'n03623198': (773, 'knee_pad'),
    'n03627232': (582, 'knot'),
    'n03630383': (918, 'lab_coat'),
    'n03633091': (892, 'ladle'),
    'n03637318': (814, 'lampshade'),
    'n03642806': (228, 'laptop'),
    'n03649909': (374, 'lawn_mower'),
    'n03657121': (988, 'lens_cap'),
    'n03658185': (371, 'letter_opener'),
    'n03661043': (687, 'library'),
    'n03662601': (238, 'lifeboat'),
    'n03666591': (546, 'lighter'),
    'n03670208': (270, 'limousine'),
    'n03673027': (244, 'liner'),
    'n03676483': (867, 'lipstick'),
    'n03680355': (973, 'Loafer'),
    'n03690938': (894, 'lotion'),
    'n03691459': (508, 'loudspeaker'),
    'n03692522': (536, 'loupe'),
    'n03697007': (697, 'lumbermill'),
    'n03706229': (532, 'magnetic_compass'),
    'n03709823': (818, 'mailbag'),
    'n03710193': (917, 'mailbox'),
    'n03710637': (782, 'maillot'),
    'n03710721': (977, 'maillot'),
    'n03717622': (763, 'manhole_cover'),
    'n03720891': (338, 'maraca'),
    'n03721384': (339, 'marimba'),
    'n03724870': (781, 'mask'),
    'n03729826': (984, 'matchstick'),
    'n03733131': (598, 'maypole'),
    'n03733281': (922, 'maze'),
    'n03733805': (946, 'measuring_cup'),
    'n03742115': (302, 'medicine_chest'),
    'n03743016': (717, 'megalith'),
    'n03759954': (509, 'microphone'),
    'n03761084': (661, 'microwave'),
    'n03763968': (866, 'military_uniform'),
    'n03764736': (875, 'milk_can'),
    'n03769881': (920, 'minibus'),
    'n03770439': (880, 'miniskirt'),
    'n03770679': (271, 'minivan'),
    'n03773504': (251, 'missile'),
    'n03775071': (871, 'mitten'),
    'n03775546': (829, 'mixing_bowl'),
    'n03776460': (290, 'mobile_home'),
    'n03777568': (272, 'Model_T'),
    'n03777754': (764, 'modem'),
    'n03781244': (686, 'monastery'),
    'n03782006': (869, 'monitor'),
    'n03785016': (277, 'moped'),
    'n03786901': (824, 'mortar'),
    'n03787032': (854, 'mortarboard'),
    'n03788195': (691, 'mosque'),
    'n03788365': (852, 'mosquito_net'),
    'n03791053': (260, 'motor_scooter'),
    'n03792782': (255, 'mountain_bike'),
    'n03792972': (728, 'mountain_tent'),
    'n03793489': (511, 'mouse'),
    'n03794056': (599, 'mousetrap'),
    'n03796401': (284, 'moving_van'),
    'n03803284': (588, 'muzzle'),
    'n03804744': (585, 'nail'),
    'n03814639': (595, 'neck_brace'),
    'n03814906': (755, 'necklace'),
    'n03825788': (915, 'nipple'),
    'n03832673': (552, 'notebook'),
    'n03837869': (699, 'obelisk'),
    'n03838899': (354, 'oboe'),
    'n03840681': (351, 'ocarina'),
    'n03841143': (520, 'odometer'),
    'n03843555': (513, 'oil_filter'),
    'n03854065': (333, 'organ'),
    'n03857828': (870, 'oscilloscope'),
    'n03866082': (937, 'overskirt'),
    'n03868242': (295, 'oxcart'),
    'n03868863': (506, 'oxygen_mask'),
    'n03871628': (921, 'packet'),
    'n03873416': (826, 'paddle'),
    'n03874293': (564, 'paddlewheel'),
    'n03874599': (584, 'padlock'),
    'n03876231': (504, 'paintbrush'),
    'n03877472': (759, 'pajama'),
    'n03877845': (685, 'palace'),
    'n03884397': (352, 'panpipe'),
    'n03887697': (877, 'paper_towel'),
    'n03888257': (942, 'parachute'),
    'n03888605': (994, 'parallel_bars'),
    'n03891251': (306, 'park_bench'),
    'n03891332': (527, 'parking_meter'),
    'n03895866': (257, 'passenger_car'),
    'n03899768': (679, 'patio'),
    'n03902125': (843, 'pay-phone'),
    'n03903868': (732, 'pedestal'),
    'n03908618': (842, 'pencil_box'),
    'n03908714': (850, 'pencil_sharpener'),
    'n03916031': (883, 'perfume'),
    'n03920288': (783, 'Petri_dish'),
    'n03924679': (789, 'photocopier'),
    'n03929660': (575, 'pick'),
    'n03929855': (926, 'pickelhaube'),
    'n03930313': (722, 'picket_fence'),
    'n03930630': (281, 'pickup'),
    'n03933933': (596, 'pier'),
    'n03935335': (931, 'piggy_bank'),
    'n03937543': (901, 'pill_bottle'),
    'n03938244': (888, 'pillow'),
    'n03942813': (841, 'ping-pong_ball'),
    'n03944341': (565, 'pinwheel'),
    'n03947888': (245, 'pirate'),
    'n03950228': (983, 'pitcher'),
    'n03954731': (372, 'plane'),
    'n03956157': (693, 'planetarium'),
    'n03958227': (964, 'plastic_bag'),
    'n03961711': (731, 'plate_rack'),
    'n03967562': (381, 'plow'),
    'n03970156': (378, 'plunger'),
    'n03976467': (857, 'Polaroid_camera'),
    'n03976657': (923, 'pole'),
    'n03977966': (285, 'police_van'),
    'n03980874': (855, 'poncho'),
    'n03982430': (314, 'pool_table'),
    'n03983396': (788, 'pop_bottle'),
    'n03991062': (838, 'pot'),
    'n03992509': (566, "potter's_wheel"),
    'n03995372': (373, 'power_drill'),
    'n03998194': (769, 'prayer_rug'),
    'n04004767': (556, 'printer'),
    'n04005630': (702, 'prison'),
    'n04008634': (542, 'projectile'),
    'n04009552': (534, 'projector'),
    'n04019541': (572, 'puck'),
    'n04023962': (846, 'punching_bag'),
    'n04026417': (939, 'purse'),
    'n04033901': (862, 'quill'),
    'n04033995': (976, 'quilt'),
    'n04037443': (273, 'racer'),
    'n04039381': (860, 'racket'),
    'n04040759': (571, 'radiator'),
    'n04041544': (863, 'radio'),
    'n04044716': (537, 'radio_telescope'),
    'n04049303': (927, 'rain_barrel'),
    'n04065272': (286, 'recreational_vehicle'),
    'n04067472': (570, 'reel'),
    'n04069434': (965, 'reflex_camera'),
    'n04070727': (668, 'refrigerator'),
    'n04074963': (578, 'remote_control'),
    'n04081281': (694, 'restaurant'),
    'n04086273': (219, 'revolver'),
    'n04090263': (541, 'rifle'),
    'n04099969': (310, 'rocking_chair'),
    'n04111531': (663, 'rotisserie'),
    'n04116512': (997, 'rubber_eraser'),
    'n04118538': (876, 'rugby_ball'),
    'n04118776': (519, 'rule'),
    'n04120489': (760, 'running_shoe'),
    'n04125021': (753, 'safe'),
    'n04127249': (586, 'safety_pin'),
    'n04131690': (952, 'saltshaker'),
    'n04133789': (751, 'sandal'),
    'n04136333': (938, 'sarong'),
    'n04141076': (355, 'sax'),
    'n04141327': (809, 'scabbard'),
    'n04141975': (521, 'scale'),
    'n04146614': (962, 'school_bus'),
    'n04147183': (221, 'schooner'),
    'n04149813': (729, 'scoreboard'),
    'n04152593': (510, 'screen'),
    'n04153751': (587, 'screw'),
    'n04154565': (379, 'screwdriver'),
    'n04162706': (589, 'seat_belt'),
    'n04179913': (559, 'sewing_machine'),
    'n04192698': (800, 'shield'),
    'n04200800': (709, 'shoe_shop'),
    'n04201297': (832, 'shoji'),
    'n04204238': (950, 'shopping_basket'),
    'n04204347': (259, 'shopping_cart'),
    'n04208210': (380, 'shovel'),
    'n04209133': (868, 'shower_cap'),
    'n04209239': (747, 'shower_curtain'),
    'n04228054': (590, 'ski'),
    'n04229816': (776, 'ski_mask'),
    'n04235860': (943, 'sleeping_bag'),
    'n04238763': (549, 'slide_rule'),
    'n04239074': (726, 'sliding_door'),
    'n04243546': (557, 'slot'),
    'n04251144': (507, 'snorkel'),
    'n04252077': (288, 'snowmobile'),
    'n04252225': (278, 'snowplow'),
    'n04254120': (960, 'soap_dispenser'),
    'n04254680': (222, 'soccer_ball'),
    'n04254777': (986, 'sock'),
    'n04258138': (577, 'solar_dish'),
    'n04259630': (925, 'sombrero'),
    'n04263257': (822, 'soup_bowl'),
    'n04264628': (858, 'space_bar'),
    'n04265275': (515, 'space_heater'),
    'n04266014': (234, 'space_shuttle'),
    'n04270147': (676, 'spatula'),
    'n04273569': (237, 'speedboat'),
    'n04275548': (600, 'spider_web'),
    'n04277352': (775, 'spindle'),
    'n04285008': (274, 'sports_car'),
    'n04286575': (593, 'spotlight'),
    'n04296562': (804, 'stage'),
    'n04310018': (263, 'steam_locomotive'),
    'n04311004': (680, 'steel_arch_bridge'),
    'n04311174': (340, 'steel_drum'),
    'n04317175': (530, 'stethoscope'),
    'n04325704': (998, 'stole'),
    'n04326547': (724, 'stone_wall'),
    'n04328186': (528, 'stopwatch'),
    'n04330267': (516, 'stove'),
    'n04332243': (514, 'strainer'),
    'n04335435': (287, 'streetcar'),
    'n04336792': (957, 'stretcher'),
    'n04344873': (311, 'studio_couch'),
    'n04346328': (692, 'stupa'),
    'n04347754': (247, 'submarine'),
    'n04350905': (794, 'suit'),
    'n04355338': (526, 'sundial'),
    'n04355933': (574, 'sunglass'),
    'n04356056': (535, 'sunglasses'),
    'n04357314': (810, 'sunscreen'),
    'n04366367': (681, 'suspension_bridge'),
    'n04367480': (828, 'swab'),
    'n04370456': (837, 'sweatshirt'),
    'n04371430': (945, 'swimming_trunks'),
    'n04371774': (569, 'swing'),
    'n04372370': (561, 'switch'),
    'n04376876': (531, 'syringe'),
    'n04380533': (304, 'table_lamp'),
    'n04389033': (250, 'tank'),
    'n04392985': (978, 'tape_player'),
    'n04398044': (675, 'teapot'),
    'n04399382': (786, 'teddy'),
    'n04404412': (944, 'television'),
    'n04409515': (970, 'tennis_ball'),
    'n04417672': (989, 'thatch'),
    'n04418357': (903, 'theater_curtain'),
    'n04423845': (758, 'thimble'),
    'n04428191': (555, 'thresher'),
    'n04429376': (308, 'throne'),
    'n04435653': (780, 'tile_roof'),
    'n04442312': (664, 'toaster'),
    'n04443257': (710, 'tobacco_shop'),
    'n04447861': (312, 'toilet_seat'),
    'n04456115': (594, 'torch'),
    'n04458633': (700, 'totem_pole'),
    'n04461696': (282, 'tow_truck'),
    'n04462240': (711, 'toyshop'),
    'n04465501': (289, 'tractor'),
    'n04467665': (283, 'trailer_truck'),
    'n04476259': (766, 'tray'),
    'n04479046': (825, 'trench_coat'),
    'n04482393': (291, 'tricycle'),
    'n04483307': (242, 'trimaran'),
    'n04485082': (597, 'tripod'),
    'n04486054': (678, 'triumphal_arch'),
    'n04487081': (882, 'trolleybus'),
    'n04487394': (349, 'trombone'),
    'n04493381': (765, 'tub'),
    'n04501370': (727, 'turnstile'),
    'n04505470': (544, 'typewriter_keyboard'),
    'n04507155': (220, 'umbrella'),
    'n04509417': (292, 'unicycle'),
    'n04515003': (334, 'upright'),
    'n04517823': (666, 'vacuum'),
    'n04522168': (874, 'vase'),
    'n04523525': (990, 'vault'),
    'n04525038': (969, 'velvet'),
    'n04525305': (558, 'vending_machine'),
    'n04532106': (790, 'vestment'),
    'n04532670': (682, 'viaduct'),
    'n04536866': (343, 'violin'),
    'n04540053': (936, 'volleyball'),
    'n04542943': (665, 'waffle_iron'),
    'n04548280': (524, 'wall_clock'),
    'n04548362': (928, 'wallet'),
    'n04550184': (317, 'wardrobe'),
    'n04552348': (231, 'warplane'),
    'n04553703': (906, 'washbasin'),
    'n04554684': (669, 'washer'),
    'n04557648': (958, 'water_bottle'),
    'n04560804': (819, 'water_jug'),
    'n04562935': (795, 'water_tower'),
    'n04579145': (772, 'whiskey_jug'),
    'n04579432': (502, 'whistle'),
    'n04584207': (899, 'wig'),
    'n04589890': (912, 'window_screen'),
    'n04590129': (904, 'window_shade'),
    'n04591157': (935, 'Windsor_tie'),
    'n04591713': (831, 'wine_bottle'),
    'n04592741': (503, 'wing'),
    'n04596742': (672, 'wok'),
    'n04597913': (951, 'wooden_spoon'),
    'n04599235': (815, 'wool'),
    'n04604644': (723, 'worm_fence'),
    'n04606251': (248, 'wreck'),
    'n04612504': (240, 'yawl'),
    'n04613696': (714, 'yurt'),
    'n06359193': (553, 'web_site'),
    'n06596364': (930, 'comic_book'),
    'n06785654': (791, 'crossword_puzzle'),
    'n06794110': (932, 'street_sign'),
    'n06874185': (861, 'traffic_light'),
    'n07248320': (774, 'book_jacket'),
    'n07565083': (803, 'menu'),
    'n07579787': (754, 'plate'),
    'n07583066': (813, 'guacamole'),
    'n07584110': (844, 'consomme'),
    'n07590611': (771, 'hot_pot'),
    'n07613480': (793, 'trifle'),
    'n07614500': (974, 'ice_cream'),
    'n07615774': (968, 'ice_lolly'),
    'n07684084': (873, 'French_loaf'),
    'n07693725': (768, 'bagel'),
    'n07695742': (975, 'pretzel'),
    'n07697313': (993, 'cheeseburger'),
    'n07697537': (885, 'hotdog'),
    'n07711569': (734, 'mashed_potato'),
    'n07714571': (736, 'head_cabbage'),
    'n07714990': (737, 'broccoli'),
    'n07715103': (738, 'cauliflower'),
    'n07716358': (739, 'zucchini'),
    'n07716906': (740, 'spaghetti_squash'),
    'n07717410': (741, 'acorn_squash'),
    'n07717556': (742, 'butternut_squash'),
    'n07718472': (743, 'cucumber'),
    'n07718747': (744, 'artichoke'),
    'n07720875': (735, 'bell_pepper'),
    'n07730033': (745, 'cardoon'),
    'n07734744': (746, 'mushroom'),
    'n07742313': (318, 'Granny_Smith'),
    'n07745940': (229, 'strawberry'),
    'n07747607': (319, 'orange'),
    'n07749582': (320, 'lemon'),
    'n07753113': (321, 'fig'),
    'n07753275': (322, 'pineapple'),
    'n07753592': (323, 'banana'),
    'n07754684': (324, 'jackfruit'),
    'n07760859': (325, 'custard_apple'),
    'n07768694': (326, 'pomegranate'),
    'n07802026': (816, 'hay'),
    'n07831146': (999, 'carbonara'),
    'n07836838': (953, 'chocolate_sauce'),
    'n07860988': (864, 'dough'),
    'n07871810': (806, 'meat_loaf'),
    'n07873807': (948, 'pizza'),
    'n07875152': (830, 'potpie'),
    'n07880968': (900, 'burrito'),
    'n07892512': (798, 'red_wine'),
    'n07920052': (947, 'espresso'),
    'n07930864': (859, 'cup'),
    'n07932039': (823, 'eggnog'),
    'n09193705': (361, 'alp'),
    'n09229709': (992, 'bubble'),
    'n09246464': (359, 'cliff'),
    'n09256479': (365, 'coral_reef'),
    'n09288635': (368, 'geyser'),
    'n09332890': (366, 'lakeside'),
    'n09399592': (363, 'promontory'),
    'n09421951': (364, 'sandbar'),
    'n09428293': (367, 'seashore'),
    'n09468604': (360, 'valley'),
    'n09472597': (362, 'volcano'),
    'n09835506': (954, 'ballplayer'),
    'n10148035': (848, 'groom'),
    'n10565667': (982, 'scuba_diver'),
    'n11879895': (330, 'rapeseed'),
    'n11939491': (357, 'daisy'),
    'n12057211': (358, "yellow_lady's_slipper"),
    'n12144580': (331, 'corn'),
    'n12267677': (327, 'acorn'),
    'n12620546': (328, 'hip'),
    'n12768682': (332, 'buckeye'),
    'n12985857': (886, 'coral_fungus'),
    'n12998815': (913, 'agaric'),
    'n13037406': (956, 'gyromitra'),
    'n13040303': (893, 'stinkhorn'),
    'n13044778': (878, 'earthstar'),
    'n13052670': (812, 'hen-of-the-woods'),
    'n13054560': (981, 'bolete'),
    'n13133613': (329, 'ear'),
    'n15075141': (889, 'toilet_tissue')}
