import os
import mmcv
import numpy as np

from .base_dataset import BaseDataset
from .builder import DATASETS


def has_file_allowed_extension(filename, extensions):
    """Checks if a file is an allowed extension.

    Args:
        filename (string): path to a file

    Returns:
        bool: True if the filename ends with a known image extension
    """
    filename_lower = filename.lower()
    return any(filename_lower.endswith(ext) for ext in extensions)


def find_folders(root):
    """Find classes by folders under a root.

    Args:
        root (string): root directory of folders

    Returns:
        folder_to_idx (dict): the map from folder name to class idx
    """
    folders = [
        d for d in os.listdir(root) if os.path.isdir(os.path.join(root, d))
    ]
    folders.sort()
    folder_to_idx = {folders[i]: i for i in range(len(folders))}
    return folder_to_idx


def get_samples(root, folder_to_idx, extensions):
    """Make dataset by walking all images under a root.

    Args:
        root (string): root directory of folders
        folder_to_idx (dict): the map from class name to class idx
        extensions (tuple): allowed extensions

    Returns:
        samples (list): a list of tuple where each element is (image, label)
    """
    samples = []
    root = os.path.expanduser(root)
    for folder_name in sorted(os.listdir(root)):
        _dir = os.path.join(root, folder_name)
        if not os.path.isdir(_dir):
            continue

        for _, _, fns in sorted(os.walk(_dir)):
            for fn in sorted(fns):
                if has_file_allowed_extension(fn, extensions):
                    path = os.path.join(folder_name, fn)
                    item = (path, folder_to_idx[folder_name])
                    samples.append(item)
    return samples


@DATASETS.register_module()
class iNat18(BaseDataset):
    """`iNat18 <https://github.com/visipedia/inat_comp/tree/master/2018>`_ Dataset.

    This implementation is modified from
    """

    IMG_EXTENSIONS = ('.jpg', '.jpeg', '.png')
    CLASSES = [
        'Hermodice carunculata',
        'Phragmatopoma californica',
        'Eudistylia vancouveri',
        'Galeolaria hystrix',
        'Serpula columbiana',
        'Spirobranchus cariniferus',
        'Spirobranchus giganteus',
        'Acanthophrynus coronatus',
        'Metaltella simoni',
        'Araneus bicentenarius',
        'Araneus diadematus',
        'Araneus gemma',
        'Araniella displicata',
        'Argiope appensa',
        'Argiope argentata',
        'Argiope aurantia',
        'Argiope keyserlingi',
        'Argiope lobata',
        'Argiope trifasciata',
        'Colaranea viriditas',
        'Cyclosa turbinata',
        'Cyrtophora citricola',
        'Eriophora pustulosa',
        'Eriophora ravilla',
        'Gasteracantha cancriformis',
        'Gea heptagon',
        'Mangora acalypha',
        'Mangora gibberosa',
        'Mecynogea lemniscata',
        'Micrathena mitrata',
        'Neoscona arabesca',
        'Neoscona crucifera',
        'Neoscona domiciliorum',
        'Neoscona oaxacensis',
        'Neoscona orizabensis',
        'Nephila clavata',
        'Nephila clavipes',
        'Nephila pilipes',
        'Nuctenea umbratica',
        'Poecilopachys australasia',
        'Zygiella x-notata',
        'Nyssus coloripes',
        'Cupiennius salei',
        'Bothriocyrtum californicum',
        'Dysdera crocata',
        'Kukulcania hibernalis',
        'Herpyllus ecclesiasticus',
        'Neotama mexicana',
        'Porrhothele antipodiana',
        'Lampona murina',
        'Neriene radiata',
        'Anoteropsis litoralis',
        'Calisoga longitarsis',
        'Oecobius navus',
        'Oxyopes gracilipes',
        'Oxyopes salticus',
        'Oxyopes scalaris',
        'Peucetia viridans',
        'Dolomedes albineus',
        'Dolomedes minor',
        'Dolomedes scriptus',
        'Pisaura mirabilis',
        'Anasaitis canosa',
        'Colonus hesperus',
        'Colonus sylvanus',
        'Eris militaris',
        'Hasarius adansoni',
        'Helpis minitabunda',
        'Hentzia palmarum',
        'Lyssomanes viridis',
        'Maevia inclemens',
        'Menemerus semilimbatus',
        'Naphrys pulex',
        'Opisthoncus polyphemus',
        'Paraphidippus fartilis',
        'Pelegrina galathea',
        'Phidippus audax',
        'Phidippus clarus',
        'Philaeus chrysops',
        'Plexippus paykulli',
        'Salticus scenicus',
        'Sassacus vitis',
        'Sitticus fasciger',
        'Trite auricoma',
        'Heteropoda venatoria',
        'Stiphidion facetum',
        'Leucauge argyra',
        'Leucauge venusta',
        'Aphonopelma chalcodes',
        'Aphonopelma hentzi',
        'Brachypelma emilia',
        'Cryptachaea veruculata',
        'Enoplognatha ovata',
        'Latrodectus geometricus',
        'Latrodectus hesperus',
        'Latrodectus katipo',
        'Latrodectus mactans',
        'Latrodectus variolus',
        'Steatoda nobilis',
        'Diaea ambara',
        'Misumena vatia',
        'Misumenoides formosipes',
        'Sidymella trapezia',
        'Synema globosum',
        'Thomisus onustus',
        'Uliodon albopunctatus',
        'Amblyomma maculatum',
        'Dermacentor occidentalis',
        'Ixodes pacificus',
        'Leiobunum vittatum',
        'Centruroides gracilis',
        'Centruroides sculpturatus',
        'Centruroides vittatus',
        'Hadrurus arizonensis',
        'Paravaejovis spinigerus',
        'Paruroctonus silvestrii',
        'Uroctonus mordax',
        'Vaejovis carolinianus',
        'Aculops rhois',
        'Tetranychus lintearius',
        'Trombidium holosericeum',
        'Scolopendra subspinipes',
        'Scutigera coleoptrata',
        'Pseudopolydesmus serratus',
        'Apheloria virginiensis',
        'Harpaphe haydeniana',
        'Rudiloria trimaculata',
        'Xystocheir dissecta',
        'Anadenobolus monilicornis',
        'Narceus annularis',
        'Trigoniulus corallinus',
        'Orthoporus ornatus',
        'Lepas anatifera',
        'Pollicipes polymerus',
        'Semibalanus balanoides',
        'Semibalanus cariosus',
        'Austrominius modestus',
        'Balanus nubilus',
        'Megabalanus californicus',
        'Chamaesipho columna',
        'Epopella plicata',
        'Tetraclita rubescens',
        'Pycnoscelus surinamensis',
        'Blatta orientalis',
        'Drymaplaneta semivitta',
        'Periplaneta australasiae',
        'Periplaneta fuliginosa',
        'Shelfordella lateralis',
        'Blattella germanica',
        'Parcoblatta americana',
        'Xylobiops basilaris',
        'Arrhenodes minutus',
        'Acmaeodera flavomarginata',
        'Agrilus ruficollis',
        'Atalantycha bilineata',
        'Cantharis rustica',
        'Chauliognathus pensylvanicus',
        'Calathus ruficollis',
        'Calosoma sayi',
        'Carabus coriaceus',
        'Cicindela aurulenta',
        'Cicindela campestris',
        'Cicindela duodecimguttata',
        'Cicindela formosa',
        'Cicindela hirticollis',
        'Cicindela longilabris',
        'Cicindela ocellata',
        'Cicindela repanda',
        'Cicindela rufiventris',
        'Cicindela scutellaris',
        'Cicindela sedecimpunctata',
        'Cicindela sexguttata',
        'Cicindela tranquebarica',
        'Cicindela trifasciata',
        'Colliuris pensylvanica',
        'Laemostenus complanatus',
        'Scaphinotus angusticollis',
        'Analeptura lineola',
        'Anastrangalia laetifica',
        'Arhopalus ferus',
        'Desmocerus palliatus',
        'Eburia quadrigeminata',
        'Evodinus monticola',
        'Knulliana cincta',
        'Megacyllene robiniae',
        'Monochamus notatus',
        'Morimus asper',
        'Neoclytus acuminatus',
        'Obrium maculatum',
        'Oemona hirta',
        'Plinthocoelium suaveolens',
        'Rosalia funebris',
        'Saperda tridentata',
        'Sphaenothecus bilineatus',
        'Stictoleptura canadensis',
        'Stictoleptura rubra',
        'Strangalepta abbreviata',
        'Tetraopes femoratus',
        'Tetraopes tetrophthalmus',
        'Trachyderes mandibularis',
        'Trichocnemis spiculatus',
        'Xylotrechus colonus',
        'Asphaera lustrans',
        'Bruchidius villosus',
        'Calligrapha multipunctata',
        'Calligrapha serpentina',
        'Charidotella sexpunctata',
        'Chrysochus auratus',
        'Chrysolina americana',
        'Chrysolina fastuosa',
        'Chrysomela populi',
        'Chrysomela scripta',
        'Diabrotica balteata',
        'Diabrotica undecimpunctata',
        'Disonycha glabrata',
        'Helocassis clavata',
        'Labidomera clivicollis',
        'Lema daturaphila',
        'Leptinotarsa juncta',
        'Lilioceris lilii',
        'Omophoita cyanipennis',
        'Paranapiacaba tricincta',
        'Paropsis charybdis',
        'Paropsisterna m-fuscum',
        'Plagiodera versicolora',
        'Timarcha tenebricosa',
        'Trirhabda flavolimbata',
        'Xanthogaleruca luteola',
        'Zygogramma signatipennis',
        'Trichodes apiarius',
        'Anatis labiculata',
        'Anatis mali',
        'Axion plagiatum',
        'Azya orbigera',
        'Calvia quatuordecimguttata',
        'Cheilomenes sexmaculata',
        'Chilocorus cacti',
        'Chilocorus stigma',
        'Coccinella novemnotata',
        'Coccinella septempunctata',
        'Coccinella transversalis',
        'Coccinella trifasciata',
        'Coccinella undecimpunctata',
        'Coelophora inaequalis',
        'Coleomegilla maculata',
        'Cryptolaemus montrouzieri',
        'Cycloneda sanguinea',
        'Halyzia sedecimguttata',
        'Harmonia axyridis',
        'Harmonia conformis',
        'Hippodamia convergens',
        'Hippodamia parenthesis',
        'Illeis galbula',
        'Mulsantina picta',
        'Olla v-nigrum',
        'Psyllobora renifer',
        'Psyllobora vigintiduopunctata',
        'Cucujus clavipes',
        'Cyrtepistomus castaneus',
        'Diaprepes abbreviatus',
        'Eudiagogus pulcher',
        'Dermestes lardarius',
        'Rhodobaenus quinquepunctatus',
        'Rhynchophorus ferrugineus',
        'Thermonectus marmoratus',
        'Alaus lusciosus',
        'Alaus oculatus',
        'Ischyrus quadripunctatus',
        'Megalodacne heros',
        'Anoplotrupes stercorosus',
        'Lucidota atra',
        'Lucanus cervus',
        'Lucanus elaphus',
        'Calopteron terminale',
        'Cysteodemus armatus',
        'Epicauta pennsylvanica',
        'Lytta magister',
        'Oedemera nobilis',
        'Odontotaenius disjunctus',
        'Pyrochroa coccinea',
        'Chrysina beyeri',
        'Chrysina gloriosa',
        'Cotinis mutabilis',
        'Cotinis nitida',
        'Digitonthophagus gazella',
        'Euphoria inda',
        'Euphoria sepulcralis',
        'Exomala orientalis',
        'Gymnetis caseyi',
        'Melolontha melolontha',
        'Osmoderma scabra',
        'Pelidnota punctata',
        'Phyllopertha horticola',
        'Polyphylla decemlineata',
        'Popillia japonica',
        'Trichiotinus texanus',
        'Valgus hemipterus',
        'Necrophila americana',
        'Oiceoptoma noveboracense',
        'Phosphuga atrata',
        'Bolitotherus cornutus',
        'Eleodes osculans',
        'Lagria hirta',
        'Zopherus nodulosus',
        'Forficula auricularia',
        'Labidura riparia',
        'Dioctria hyalipennis',
        'Efferia albibarbis',
        'Laphria flavicollis',
        'Laphria macquarti',
        'Laphria thoracica',
        'Mallophora leschenaulti',
        'Microstylum morosum',
        'Promachus bastardii',
        'Promachus hinei',
        'Promachus rufipes',
        'Stichopogon trifasciatus',
        'Dilophus nigrostigma',
        'Bombylius major',
        'Poecilanthrax lucifer',
        'Calliphora vicina',
        'Asphondylia auripila',
        'Rhopalomyia californica',
        'Aedes aegypti',
        'Aedes albopictus',
        'Culex quinquefasciatus',
        'Arachnocampa luminosa',
        'Rainieria antennaepes',
        'Pogonortalis doclea',
        'Pyrgota undata',
        'Exaireta spinigera',
        'Eristalinus taeniops',
        'Eristalis arbustorum',
        'Eristalis dimidiata',
        'Eristalis tenax',
        'Eristalis transversa',
        'Eupeodes americanus',
        'Eupeodes volucris',
        'Helophilus fasciatus',
        'Helophilus pendulus',
        'Merodon equestris',
        'Meromacrus acutus',
        'Milesia virginiensis',
        'Myathropa florea',
        'Ocyptamus fascipennis',
        'Ocyptamus fuscipennis',
        'Palpada furcata',
        'Palpada vinetorum',
        'Paragus haemorrhous',
        'Scaeva pyrastri',
        'Toxomerus marginatus',
        'Toxomerus occidentalis',
        'Volucella pellucens',
        'Volucella zonaria',
        'Trichopoda lanipes',
        'Trigonospila brevifacies',
        'Ceratitis capitata',
        'Eurosta solidaginis',
        'Delphinia picta',
        'Hexagenia limbata',
        'Acanalonia bivittata',
        'Acanthosoma haemorrhoidale',
        'Elasmostethus cruciatus',
        'Alydus eurinus',
        'Megalotomus quinquespinosus',
        'Riptortus pedestris',
        'Glycaspis brimblecombei',
        'Pachypsylla celtidismamma',
        'Pachypsylla venusta',
        'Aphis nerii',
        'Brevicoryne brassicae',
        'Hormaphis hamamelidis',
        'Tamalia coweni',
        'Tuberolachnus salignus',
        'Philaenus spumarius',
        'Lethocerus americanus',
        'Cercopis vulnerata',
        'Huaina inca',
        'Prosapia bicincta',
        'Colladonus clitellarius',
        'Cuerna costalis',
        'Graphocephala coccinea',
        'Graphocephala versuta',
        'Homalodisca vitripennis',
        'Paraulacizes irrorata',
        'Sobara palliolata',
        'Sophonia orientalis',
        'Texananus excultus',
        'Tylozygus bifidus',
        'Tylozygus geometricus',
        'Cyclochila australasiae',
        'Henicopsaltria eydouxii',
        'Neotibicen canicularis',
        'Neotibicen pruinosus',
        'Neotibicen resh',
        'Neotibicen tibicen',
        'Pacarina puella',
        'Acanthocephala alata',
        'Acanthocephala femorata',
        'Acanthocephala terminalis',
        'Acanthocephala thomasi',
        'Anisoscelis luridus',
        'Chariesterus antennator',
        'Euthochtha galeator',
        'Leptoglossus occidentalis',
        'Leptoglossus phyllopus',
        'Mozena lunata',
        'Narnia femorata',
        'Sagotylus confluens',
        'Thasus acutangulus',
        'Sehirus cinctus',
        'Cryptococcus fagisuga',
        'Flatormenis proxima',
        'Metcalfa pruinosa',
        'Lycorma delicatula',
        'Gelastocoris oculatus',
        'Largus californicus',
        'Stenomacra marginella',
        'Lygaeus equestris',
        'Lygaeus kalmii',
        'Melacoryphus lateralis',
        'Melanopleurus belfragei',
        'Neacoryphus bicrucis',
        'Oncopeltus fasciatus',
        'Spilostethus pandurus',
        'Spilostethus saxatilis',
        'Acutalis tartarea',
        'Entylia carinata',
        'Platycotis vittata',
        'Adelphocoris lineolatus',
        'Closterotomus norwegicus',
        'Leptopterna dolabrata',
        'Metriorrhynchomiris dislocatus',
        'Stenotus binotatus',
        'Taylorilygus apicalis',
        'Icerya purchasi',
        'Agonoscelis puberula',
        'Bagrada hilaris',
        'Banasa calva',
        'Banasa dimidiata',
        'Banasa euchlora',
        'Carpocoris mediterraneus',
        'Chinavia hilaris',
        'Chlorochroa ligata',
        'Dolycoris baccarum',
        'Erthesina fullo',
        'Eurydema ornata',
        'Euschistus servus',
        'Euschistus tristigmus',
        'Euthyrhynchus floridanus',
        'Glaucias amyoti',
        'Graphosoma italicum',
        'Halyomorpha halys',
        'Menecles insertus',
        'Monteithiella humeralis',
        'Mormidea lugens',
        'Murgantia histrionica',
        'Nezara viridula',
        'Oebalus pugnax',
        'Podisus maculiventris',
        'Proxys punctulatus',
        'Rhaphigaster nebulosa',
        'Stiretrus anchorago',
        'Thyanta custator',
        'Dindymus sanguineus',
        'Dysdercus bimaculatus',
        'Dysdercus mimulus',
        'Apiomerus flaviventris',
        'Arilus cristatus',
        'Melanolestes picipes',
        'Phymata americana',
        'Phymata fasciata',
        'Pselliopus barberi',
        'Rasahus hamatus',
        'Reduvius personatus',
        'Repipta taurus',
        'Zelus cervicalis',
        'Zelus renardii',
        'Zelus tetracanthus',
        'Boisea trivittata',
        'Corizus hyoscyami',
        'Jadera haematoloma',
        'Liorhyssus hyalinus',
        'Niesthrea louisianica',
        'Myodocha serripes',
        'Xanthochilus saturnius',
        'Apis cerana',
        'Apis dorsata',
        'Apis mellifera',
        'Bombus auricomus',
        'Bombus borealis',
        'Bombus californicus',
        'Bombus ephippiatus',
        'Bombus fraternus',
        'Bombus griseocollis',
        'Bombus hypnorum',
        'Bombus impatiens',
        'Bombus lapidarius',
        'Bombus melanopygus',
        'Bombus mixtus',
        'Bombus pascuorum',
        'Bombus pensylvanicus',
        'Bombus perplexus',
        'Bombus ruderatus',
        'Bombus rufocinctus',
        'Bombus ternarius',
        'Bombus terrestris',
        'Bombus terricola',
        'Bombus vosnesenskii',
        'Eulaema polychroma',
        'Melissodes bimaculatus',
        'Tetragonisca angustula',
        'Trigona fulviventris',
        'Xylocopa californica',
        'Xylocopa mexicanorum',
        'Xylocopa micans',
        'Xylocopa tabaniformis',
        'Xylocopa violacea',
        'Xylocopa virginica',
        'Chrysis angolensis',
        'Bembix americana',
        'Bicyrtes quadrifasciatus',
        'Philanthus gibbosus',
        'Sphecius speciosus',
        'Trypoxylon politum',
        'Andricus confertus',
        'Andricus fullawayi',
        'Andricus gigas',
        'Andricus parmula',
        'Antron quercusechinus',
        'Callirhytis quercuspomiformis',
        'Callirhytis seminator',
        'Cynips douglasii',
        'Diplolepis polita',
        'Diplolepis rosae',
        'Disholcaspis plumbella',
        'Dryocosmus minusculus',
        'Diprion similis',
        'Evania appendigaster',
        'Anoplolepis gracilipes',
        'Atta mexicana',
        'Camponotus sericeiventris',
        'Crematogaster scutellaris',
        'Dorymyrmex bicolor',
        'Formica obscuripes',
        'Neoponera villosa',
        'Paraponera clavata',
        'Pogonomyrmex californicus',
        'Pogonomyrmex rugosus',
        'Tapinoma sessile',
        'Veromessor pergandei',
        'Agapostemon splendens',
        'Agapostemon virescens',
        'Augochlora pura',
        'Augochloropsis metallica',
        'Halictus ligatus',
        'Enicospilus purgatus',
        'Megarhyssa atrata',
        'Megarhyssa macrurus',
        'Megachile xylocopoides',
        'Dasymutilla aureola',
        'Dasymutilla sackenii',
        'Pepsis thisbe',
        'Tachypompilus ferrugineus',
        'Megascolia maculata',
        'Scolia bicincta',
        'Scolia nobilitata',
        'Ammophila pictipennis',
        'Chlorion aerarium',
        'Sceliphron caementarium',
        'Sphex ichneumoneus',
        'Sphex nudus',
        'Myzinum quinquecinctum',
        'Dolichovespula arenaria',
        'Dolichovespula maculata',
        'Euodynerus hidalgo',
        'Euodynerus pratensis',
        'Mischocyttarus flavitarsis',
        'Pachodynerus erynnis',
        'Parapolybia indica',
        'Polistes annularis',
        'Polistes aurifer',
        'Polistes bellicosus',
        'Polistes carolina',
        'Polistes comanchus',
        'Polistes dominula',
        'Polistes dorsalis',
        'Polistes exclamans',
        'Polistes instabilis',
        'Polistes major',
        'Polistes metricus',
        'Vespa crabro',
        'Vespa velutina',
        'Vespula maculifrons',
        'Adela septentrionella',
        'Adela trigrapha',
        'Anthela ocellata',
        'Argyresthia oreasella',
        'Atteva aurea',
        'Oegoconia novimundi',
        'Blastobasis glandulella',
        'Trilocha varians',
        'Limnaecia phragmitella',
        'Cossus cossus',
        'Achyra affinitalis',
        'Achyra rantalis',
        'Agriphila ruricolella',
        'Argyria lacteella',
        'Chrysoteuchia topiarius',
        'Cnaphalocrocis medinalis',
        'Cnaphalocrocis poeyalis',
        'Cnaphalocrocis trapezalis',
        'Condylorrhiza vestigialis',
        'Crambus agitatellus',
        'Crambus albellus',
        'Crambus girardellus',
        'Crambus laqueatellus',
        'Crambus praefectellus',
        'Crambus saltuellus',
        'Culladia cuneiferellus',
        'Diacme adipaloides',
        'Diaphania costata',
        'Diaphania indica',
        'Diathrausta harlequinalis',
        'Elophila gyralis',
        'Elophila icciusalis',
        'Elophila obliteralis',
        'Euchromius ocelleus',
        'Eudonia strigalis',
        'Eustixia pupula',
        'Evergestis rimosalis',
        'Fissicrambus mutabilis',
        'Fumibotys fumalis',
        'Glaphyria sequistrialis',
        'Glyphodes onychinalis',
        'Hellula rogatalis',
        'Herpetogramma abdominalis',
        'Herpetogramma bipunctalis',
        'Herpetogramma licarsisalis',
        'Herpetogramma phaeopteralis',
        'Hileithia magualis',
        'Hydriris ornatalis',
        'Hygraula nitens',
        'Hymenia perspectalis',
        'Lamprosema victoriae',
        'Leucinodes cordalis',
        'Lineodes integra',
        'Lineodes interrupta',
        'Loxostege cereralis',
        'Lygropia rivulalis',
        'Maruca vitrata',
        'Metoeca foedalis',
        'Microcrambus biguttellus',
        'Microcrambus elegans',
        'Microtheoris ophionalis',
        'Mimoschinia rufofascialis',
        'Nacoleia rhoeoalis',
        'Nomophila nearctica',
        'Nomophila noctuella',
        'Ommatospila narcaeusalis',
        'Ostrinia nubilalis',
        'Palpita atrisquamalis',
        'Palpita flegia',
        'Palpita quadristigmalis',
        'Parapoynx allionealis',
        'Parapoynx badiusalis',
        'Parapoynx diminutalis',
        'Parapoynx maculalis',
        'Patania ruralis',
        'Pediasia trisecta',
        'Perispasta caeculalis',
        'Petrophila bifascialis',
        'Petrophila jaliscalis',
        'Pilocrocis ramentalis',
        'Pyrausta bicoloralis',
        'Pyrausta californicalis',
        'Pyrausta despicata',
        'Pyrausta laticlavia',
        'Pyrausta purpuralis',
        'Pyrausta rubricalis',
        'Pyrausta signatalis',
        'Samea baccatalis',
        'Samea ecclesialis',
        'Samea multiplicalis',
        'Sameodes cancellalis',
        'Saucrobotys futilalis',
        'Scoparia biplagialis',
        'Spoladea recurvalis',
        'Syngamia florella',
        'Udea rubigalis',
        'Uresiphita maorialis',
        'Uresiphita reversalis',
        'Urola nivalis',
        'Agonopterix alstroemeriana',
        'Bibarrambla allenella',
        'Ethmia delliella',
        'Machimia tentoriferella',
        'Drepana arcuata',
        'Drepana bilineata',
        'Thyatira batis',
        'Allotria elonympha',
        'Amata huebneri',
        'Amata phegea',
        'Anticarsia gemmatalis',
        'Apantesis arge',
        'Apantesis incorrupta',
        'Apantesis ornata',
        'Apantesis parthenice',
        'Apantesis proxima',
        'Apantesis virgo',
        'Apantesis virguncula',
        'Arachnis picta',
        'Arctia virginalis',
        'Ascalapha odorata',
        'Asota caricae',
        'Asota heliconia',
        'Asota plaginota',
        'Barsine striata',
        'Brunia antica',
        'Brunia replana',
        'Bulia deducta',
        'Caenurgia chloropha',
        'Caenurgina erechtea',
        'Calliteara pudibunda',
        'Catocala amatrix',
        'Catocala cara',
        'Catocala cerogama',
        'Catocala lineella',
        'Catocala relicta',
        'Catocala vidua',
        'Chalciope mygdon',
        'Cisseps fulvicollis',
        'Cisthene packardii',
        'Cisthene plumbea',
        'Cisthene tenuifascia',
        'Clemensia albata',
        'Cosmosoma myrodora',
        'Crambidia pallida',
        'Creatonotos transiens',
        'Ctenucha rubroscapus',
        'Ctenucha venosa',
        'Ctenucha virginica',
        'Cycnia collaris',
        'Cyligramma latona',
        'Dasypodia cymatodes',
        'Diacrisia sannio',
        'Diatenes aglossoides',
        'Dysschema howardi',
        'Ercheia cyllaria',
        'Erebus ephesperis',
        'Estigmene acrea',
        'Eublemma minima',
        'Eublemma recta',
        'Eubolina impartialis',
        'Euchaetes egle',
        'Euclidia cuspidea',
        'Euclidia glyphica',
        'Eudocima phalonia',
        'Euparthenos nubilis',
        'Euplagia quadripunctaria',
        'Gnophaela latipennis',
        'Goniapteryx servia',
        'Halone sejuncta',
        'Halysidota tessellaris',
        'Haploa clymene',
        'Haploa lecontei',
        'Horama panthalon',
        'Hypena abalienalis',
        'Hypena baltimoralis',
        'Hypena bijugalis',
        'Hypena deceptalis',
        'Hypena manalis',
        'Hypena minualis',
        'Hypena palparia',
        'Hypena proboscidalis',
        'Hypena scabra',
        'Hypercompe scribonia',
        'Hyperstrotia pervertens',
        'Hyperstrotia secta',
        'Hypoprepia fucosa',
        'Hypoprepia miniata',
        'Hypsoropha hormos',
        'Idia aemula',
        'Idia americalis',
        'Idia julia',
        'Idia rotundalis',
        'Isogona tenuis',
        'Lascoria ambigualis',
        'Laspeyria concavata',
        'Leptocneria reducta',
        'Lopharthrum comprimens',
        'Lophocampa annulosa',
        'Lophocampa argentata',
        'Lophocampa caryae',
        'Lophocampa maculata',
        'Lyclene structa',
        'Lymantria dispar',
        'Macrochilo absorptalis',
        'Melipotis acontioides',
        'Melipotis cellaris',
        'Melipotis fasciolaris',
        'Melipotis jucunda',
        'Metalectra discalis',
        'Metria amella',
        'Meyrickella torquesauria',
        'Mocis frugalis',
        'Mocis latipes',
        'Mocis marcida',
        'Mocis texana',
        'Mocis undata',
        'Nigetia formosalis',
        'Nyctemera adversata',
        'Nyctemera amicus',
        'Nyctemera baulus',
        'Orgyia definita',
        'Orgyia detrita',
        'Orgyia leucostigma',
        'Orvasca subnotata',
        'Palthis angulalis',
        'Palthis asopialis',
        'Panopoda carneicosta',
        'Pantydia sparsa',
        'Parallelia bistriaris',
        'Pareuchaetes insulata',
        'Perina nuda',
        'Phalaenostola larentioides',
        'Philenora aspectalella',
        'Phoberia atomaris',
        'Phyprosopus callitrichoides',
        'Pseudohemihyalea edwardsii',
        'Ptichodis vinculum',
        'Pyrrharctia isabella',
        'Renia discoloralis',
        'Renia flavipunctalis',
        'Rhapsa scotosialis',
        'Rivula propinqualis',
        'Rivula sericealis',
        'Schistophleps bipuncta',
        'Scolecocampa liburna',
        'Selenisa sueroides',
        'Spilosoma congrua',
        'Spilosoma curvata',
        'Spilosoma lubricipeda',
        'Spilosoma virginica',
        'Sympis rufibasis',
        'Syntomeida epilais',
        'Syntomeida melanthus',
        'Tathorhynchus exsiccata',
        'Termessa gratiosa',
        'Tetanolita floridana',
        'Tetanolita mynesalis',
        'Trigonodes hyppasia',
        'Tyria jacobaeae',
        'Utetheisa pulchella',
        'Virbia aurantiaca',
        'Virbia laeta',
        'Zale lunata',
        'Zale minerea',
        'Zanclognatha cruralis',
        'Zanclognatha laevigata',
        'Marathyssa basalis',
        'Marathyssa inficita',
        'Paectes abrostoloides',
        'Paectes oculatrix',
        'Anacampsis fullonella',
        'Chionodes mediofuscella',
        'Dichomeris flavocostella',
        'Stegasta bosqueella',
        'Acronyctodes mexicanaria',
        'Alsophila pometaria',
        'Anachloris subochraria',
        'Anticlea vasiliata',
        'Arhodia lasiocamparia',
        'Besma endropiaria',
        'Biston betularia',
        'Cabera erythemaria',
        'Campaea perlata',
        'Camptogramma bilineata',
        'Caripeta divisata',
        'Caripeta piniata',
        'Celenna festivaria',
        'Chiasmia clathrata',
        'Chiasmia emersaria',
        'Chlorochlamys chloroleucaria',
        'Chloroclystis filata',
        'Chrysolarentia mecynata',
        'Cingilia catenaria',
        'Circopetes obtusata',
        'Cladara atroliturata',
        'Cladara limitaria',
        'Cleora sublunaria',
        'Comostola laesaria',
        'Costaconvexa centrostrigaria',
        'Crypsiphona ocultaria',
        'Cyclophora nanaria',
        'Cyclophora packardi',
        'Dichorda illustraria',
        'Dichorda iridaria',
        'Didymoctenia exsuperata',
        'Digrammia continuata',
        'Digrammia gnophosaria',
        'Digrammia ocellinata',
        'Disclisioprocta stellata',
        'Dissomorphia australiaria',
        'Dysphania militaris',
        'Ecliptopera silaceata',
        'Ectropis bhurmitra',
        'Ematurga atomaria',
        'Ennomos subsignaria',
        'Epicyme rubropunctaria',
        'Epiphryne verriculata',
        'Epirrhoe alternata',
        'Epirrita autumnata',
        'Epyaxa rosearia',
        'Erannis defoliaria',
        'Erastria cruentaria',
        'Eubaphe unicolor',
        'Euchlaena amoenaria',
        'Euchlaena serrata',
        'Eufidonia notataria',
        'Eugonobapta nivosaria',
        'Eulithis explanata',
        'Euphyia intermediata',
        'Eusarca confusaria',
        'Eutrapela clemataria',
        'Fernaldella fimetaria',
        'Gastrophora henricaria',
        'Gellonia dejectaria',
        'Glena quinquelinearia',
        'Glenoides texanaria',
        'Gymnoscelis rufifasciata',
        'Haematopis grataria',
        'Hammaptera parinotata',
        'Hemithea aestivaria',
        'Heteralex unilinea',
        'Hethemia pistasciaria',
        'Homodotis megaspilata',
        'Horisme intestinata',
        'Hydrelia inornata',
        'Hypagyrtis unipunctata',
        'Hyposidra talaca',
        'Idaea aversata',
        'Idaea demissaria',
        'Idaea dimidiata',
        'Idaea furciferata',
        'Idaea seriata',
        'Idaea tacturata',
        'Ilexia intractata',
        'Iridopsis defectaria',
        'Iridopsis larvaria',
        'Isturgia dislocaria',
        'Lambdina fiscellaria',
        'Leptostales pannaria',
        'Lobocleta ossularia',
        'Lobophora nivigerata',
        'Lomaspilis marginata',
        'Lomographa glomeraria',
        'Lophosis labeculata',
        'Lycia ypsilon',
        'Lytrosis unitaria',
        'Macaria bisignata',
        'Melanodes anthracitaria',
        'Melanolophia canadaria',
        'Melanolophia imitata',
        'Melanolophia signataria',
        'Mellilla xanthometata',
        'Mesoleuca ruficillata',
        'Milionia zonea',
        'Nematocampa resistaria',
        'Nemoria bistriaria',
        'Nemoria lixaria',
        'Nemoria mimosaria',
        'Neoalcis californiaria',
        'Nothomiza flavicosta',
        'Obeidia tigrata',
        'Operophtera bruceata',
        'Operophtera brumata',
        'Opisthograptis luteolata',
        'Orthoclydon praefectata',
        'Orthonama obstipata',
        'Paleacrita vernata',
        'Pasiphila rectangulata',
        'Patalene olyzonaria',
        'Peribatodes rhomboidaria',
        'Pero morrisonaria',
        'Phelotis cognata',
        'Phigalia denticulata',
        'Phigalia strigataria',
        'Pholodes sinistraria',
        'Phrissogonus laticostatus',
        'Plagodis alcoolaria',
        'Plagodis pulveraria',
        'Plagodis serinaria',
        'Pleuroprucha insulsaria',
        'Poecilasthena pulchraria',
        'Prasinocyma rhodocosma',
        'Prasinocyma semicrocea',
        'Prochoerodes lineola',
        'Psamatodes abydata',
        'Pseudocoremia suavis',
        'Pseudopanthera macularia',
        'Rheumaptera prunivorata',
        'Rhodometra sacraria',
        'Scopula inductata',
        'Scopula limboundata',
        'Scopula umbilicata',
        'Scotopteryx chenopodiata',
        'Sicya macularia',
        'Sigilliclystis insigillata',
        'Siona lineata',
        'Spargania magnoliata',
        'Speranza pustularia',
        'Sphacelodes vulneraria',
        'Synchlora aerata',
        'Synchlora frondaria',
        'Syneora hemeropa',
        'Timandra amaturaria',
        'Timandra comae',
        'Traminda aventiaria',
        'Trichodezia albovittata',
        'Triphosa haesitata',
        'Xanthorhoe ferrugata',
        'Xanthorhoe fluctuata',
        'Xanthorhoe labradorensis',
        'Triodia sylvina',
        'Amblyscirtes aesculapius',
        'Amblyscirtes celia',
        'Amblyscirtes hegon',
        'Amblyscirtes nysa',
        'Amblyscirtes vialis',
        'Anatrytone logan',
        'Ancyloxypha numitor',
        'Antigonus erosus',
        'Apyrrothrix araxes',
        'Asbolis capucinus',
        'Astraptes anaphus',
        'Astraptes fulgerator',
        'Atalopedes campestris',
        'Atrytonopsis hianna',
        'Cabares potrillo',
        'Carcharodus alceae',
        'Celotes nessus',
        'Chioides albofasciatus',
        'Chioides zilpa',
        'Chiomara georgina',
        'Cogia calchas',
        'Copaeodes aurantiaca',
        'Copaeodes minima',
        'Cymaenes trebius',
        'Eantis thraso',
        'Epargyreus clarus',
        'Erynnis baptisiae',
        'Erynnis brizo',
        'Erynnis funeralis',
        'Erynnis horatius',
        'Erynnis juvenalis',
        'Erynnis propertius',
        'Erynnis tages',
        'Erynnis tristis',
        'Euphyes dion',
        'Euphyes vestris',
        'Heliopetes ericetorum',
        'Heliopetes macaira',
        'Heliopyrgus domicella',
        'Hesperia colorado',
        'Hesperia comma',
        'Hesperia leonardus',
        'Hesperia sassacus',
        'Hylephila phyleus',
        'Iambrix salsala',
        'Lerema accius',
        'Lerodea eufala',
        'Mylon pelopidas',
        'Ochlodes agricola',
        'Ochlodes sylvanoides',
        'Ochlodes sylvanus',
        'Panoquina ocola',
        'Panoquina panoquin',
        'Panoquina panoquinoides',
        'Pellicia arina',
        'Phocides pigmalion',
        'Poanes hobomok',
        'Poanes melane',
        'Poanes taxiles',
        'Poanes zabulon',
        'Polites mystic',
        'Polites peckius',
        'Polygonus leo',
        'Pompeius verna',
        'Potanthus omaha',
        'Pyrgus communis',
        'Pyrgus malvae',
        'Pyrgus malvoides',
        'Pyrgus oileus',
        'Pyrgus philetas',
        'Pyrgus ruralis',
        'Spathilepia clonius',
        'Staphylus ceos',
        'Staphylus hayhurstii',
        'Staphylus mazans',
        'Suastus gremius',
        'Systasea pulverulenta',
        'Thorybes bathyllus',
        'Thorybes pylades',
        'Thymelicus lineola',
        'Thymelicus sylvestris',
        'Urbanus dorantes',
        'Urbanus procne',
        'Urbanus proteus',
        'Vettius fantasos',
        'Wallengrenia egeremet',
        'Xenophanes tryxus',
        'Zopyrion sandace',
        'Lactura subfervens',
        'Artace cribrarius',
        'Euthrix potatoria',
        'Lasiocampa quercus',
        'Macrothylacia rubi',
        'Malacosoma americanum',
        'Malacosoma disstria',
        'Tolype laricis',
        'Trabala vishnou',
        'Apoda biguttata',
        'Doratifera pinguis',
        'Euclea incisa',
        'Isochaetes beutenmuelleri',
        'Packardia geminata',
        'Phobetron pithecium',
        'Tortricidia testacea',
        'Acytolepis puspa',
        'Anthene emolus',
        'Arawacus jada',
        'Aricia agestis',
        'Aricia cramera',
        'Atlides halesus',
        'Brephidium exilis',
        'Callophrys augustinus',
        'Callophrys dumetorum',
        'Callophrys eryphon',
        'Callophrys gryneus',
        'Callophrys mossii',
        'Callophrys rubi',
        'Callophrys viridis',
        'Callophrys xami',
        'Calycopis cecrops',
        'Calycopis isobeon',
        'Castalius rosimon',
        'Celastrina echo',
        'Celastrina neglecta',
        'Chilades pandava',
        'Chlorostrymon simaethis',
        'Cupido amyntula',
        'Cupido comyntas',
        'Cupido minimus',
        'Cyaniris semiargus',
        'Drupadia ravindra',
        'Echinargus isola',
        'Eooxylides tharis',
        'Euchrysops cnejus',
        'Eumaeus atala',
        'Eumaeus childrenae',
        'Euphilotes bernardino',
        'Everes argiades',
        'Glaucopsyche lygdamus',
        'Habrodais grunus',
        'Hemiargus ceraunus',
        'Hypolycaena erylus',
        'Icaricia acmon',
        'Icaricia icarioides',
        'Icaricia lupini',
        'Icaricia saepiolus',
        'Jamides celeno',
        'Leptotes cassius',
        'Leptotes marina',
        'Leptotes pirithous',
        'Leptotes plinius',
        'Lycaena alciphron',
        'Lycaena arota',
        'Lycaena boldenarum',
        'Lycaena dispar',
        'Lycaena dorcas',
        'Lycaena editha',
        'Lycaena epixanthe',
        'Lycaena gorgon',
        'Lycaena helloides',
        'Lycaena heteronea',
        'Lycaena hippothoe',
        'Lycaena hyllus',
        'Lycaena phlaeas',
        'Lycaena rubidus',
        'Lycaena salustius',
        'Lycaena tityrus',
        'Lycaena virgaureae',
        'Lycaena xanthoides',
        'Ministrymon clytie',
        'Ministrymon leda',
        'Panthiades bathildis',
        'Phaeostrymon alcestis',
        'Philotes sonorensis',
        'Plebejus argus',
        'Plebejus idas',
        'Polyommatus bellargus',
        'Polyommatus coridon',
        'Polyommatus icarus',
        'Pseudolycaena damo',
        'Pseudozizeeria maha',
        'Rapala pheretima',
        'Satyrium behrii',
        'Satyrium calanus',
        'Satyrium edwardsii',
        'Satyrium favonius',
        'Satyrium spini',
        'Satyrium tetra',
        'Strymon albata',
        'Strymon bazochii',
        'Strymon melinus',
        'Strymon rufofusca',
        'Zizeeria knysna',
        'Zizina labradus',
        'Zizina otis',
        'Zizina oxleyi',
        'Zizula hylax',
        'Lagoa crispata',
        'Achatia distincta',
        'Acontia trabealis',
        'Acronicta dactylina',
        'Acronicta fragilis',
        'Acronicta hasta',
        'Acronicta innotata',
        'Acronicta insularis',
        'Acronicta lobeliae',
        'Acronicta morula',
        'Acronicta oblinita',
        'Acronicta rubricoma',
        'Acronicta rumicis',
        'Agnorisma badinodis',
        'Agrotis exclamationis',
        'Agrotis infusa',
        'Agrotis malefida',
        'Agrotis munda',
        'Agrotis porphyricollis',
        'Agrotis venerabilis',
        'Allagrapha aerea',
        'Alypia octomaculata',
        'Amphipoea americana',
        'Amphipyra pyramidoides',
        'Amyna stricta',
        'Anaplectoides prasina',
        'Anathix ralla',
        'Anicla infecta',
        'Argyrogramma verruca',
        'Athetis tarda',
        'Athetis tenuis',
        'Autographa precationis',
        'Bagisara repanda',
        'Balsa tristrigella',
        'Bryolymnia viridata',
        'Callopistria cordata',
        'Callopistria floridensis',
        'Chloridea virescens',
        'Choephora fungorum',
        'Chrysodeixis argentifera',
        'Chrysodeixis eriosoma',
        'Chrysodeixis includens',
        'Colocasia propinquilinea',
        'Condica vecors',
        'Condica videns',
        'Conistra vaccinii',
        'Copivaleria grotei',
        'Cosmia calami',
        'Cosmodes elegans',
        'Ctenoplusia oxygramma',
        'Cucullia asteroides',
        'Dargida diffusa',
        'Dargida procinctus',
        'Diachrysia balluca',
        'Diastema tigris',
        'Elaphria chalcedonia',
        'Elaphria versicolor',
        'Emarginea percara',
        'Eudryas grata',
        'Euplexia benesimilis',
        'Feltia subterranea',
        'Feralia comstocki',
        'Galgula partita',
        'Helicoverpa armigera',
        'Helicoverpa zea',
        'Hyppa xylinoides',
        'Lacanobia oleracea',
        'Lacinipolia erecta',
        'Lacinipolia laudabilis',
        'Lacinipolia olivacea',
        'Lacinipolia renigera',
        'Leucania adjuta',
        'Leucania diatrecta',
        'Leucania incognita',
        'Leuconycta diphteroides',
        'Leuconycta lepidula',
        'Magusa divaricata',
        'Maliattha concinnimacula',
        'Maliattha synochitis',
        'Marimatha nigrofimbria',
        'Micrathetis triplex',
        'Morrisonia mucens',
        'Mythimna albipuncta',
        'Mythimna convecta',
        'Mythimna oxygala',
        'Mythimna unipuncta',
        'Nedra ramosula',
        'Neoligia exhausta',
        'Neumoegenia poetica',
        'Noctua pronuba',
        'Ochropleura implecta',
        'Ogdoconta cinereola',
        'Orthodes furtiva',
        'Orthodes majuscula',
        'Orthosia cerasi',
        'Orthosia gothica',
        'Orthosia hibisci',
        'Panthea acronyctoides',
        'Panthea virginarius',
        'Papaipema inquaesita',
        'Peridroma saucia',
        'Phlogophora iris',
        'Phosphila miselioides',
        'Phosphila turbulenta',
        'Plusia putnami',
        'Polia nimbosa',
        'Ponometia candefacta',
        'Ponometia semiflava',
        'Proteuxoa hypochalchis',
        'Protodeltote albidula',
        'Protodeltote muscosula',
        'Psaphida rolandi',
        'Pseudeustrotia carneola',
        'Pseudohermonassa bicarnea',
        'Rachiplusia ou',
        'Schinia chrysellus',
        'Schinia rivulosa',
        'Schinia trifascia',
        'Spodoptera dolichos',
        'Spodoptera exigua',
        'Spodoptera frugiperda',
        'Spodoptera mauritia',
        'Spodoptera ornithogalli',
        'Spragueia dama',
        'Spragueia guttata',
        'Spragueia jaguaralis',
        'Tarache aprica',
        'Tarache quadriplaga',
        'Thysanoplusia orichalcea',
        'Tricholita signata',
        'Trichoplusia ni',
        'Trichordestra legitima',
        'Tripudia quadrifera',
        'Tyta luctuosa',
        'Xerociris wilsonii',
        'Xestia c-nigrum',
        'Xestia dolosa',
        'Xestia smithii',
        'Xestia xanthographa',
        'Baileya australis',
        'Baileya ophthalmica',
        'Meganola minuscula',
        'Meganola phylla',
        'Nola cereella',
        'Uraba lugens',
        'Clostera apicalis',
        'Datana integerrima',
        'Datana major',
        'Destolmia lineata',
        'Ecnomodes sagittaria',
        'Furcula borealis',
        'Furcula cinerea',
        'Heterocampa guttivitta',
        'Heterocampa obliqua',
        'Heterocampa umbrata',
        'Hyperaeschra georgica',
        'Lochmaeus bilineata',
        'Lochmaeus manteo',
        'Macrurocampa marthesia',
        'Nadata gibbosa',
        'Neola semiaurata',
        'Nerice bidentata',
        'Ochrogaster lunifer',
        'Oligocentria lignicolor',
        'Oligocentria semirufescens',
        'Peridea basitriens',
        'Phalera bucephala',
        'Pheosia rimosa',
        'Schizura concinna',
        'Schizura ipomaeae',
        'Schizura leptinoides',
        'Schizura unicornis',
        'Acraea terpsicore',
        'Adelpha basiloides',
        'Adelpha californica',
        'Adelpha eulalia',
        'Adelpha fessonia',
        'Adelpha iphicleola',
        'Aglais io',
        'Aglais milberti',
        'Aglais urticae',
        'Agraulis vanillae',
        'Anaea andria',
        'Anartia amathea',
        'Anartia fatima',
        'Anartia jatrophae',
        'Anthanassa texana',
        'Anthanassa tulcis',
        'Apatura iris',
        'Aphantopus hyperantus',
        'Argynnis adippe',
        'Argynnis aglaja',
        'Argynnis pandora',
        'Argynnis paphia',
        'Argyreus hyperbius',
        'Asterocampa celtis',
        'Asterocampa clyton',
        'Asterocampa leilia',
        'Boloria bellona',
        'Boloria epithore',
        'Boloria euphrosyne',
        'Brenthis daphne',
        'Brenthis ino',
        'Brintesia circe',
        'Caligo telamonius',
        'Catonephele numilia',
        'Cercyonis oetus',
        'Cercyonis pegala',
        'Cercyonis sthenele',
        'Chlosyne acastus',
        'Chlosyne californica',
        'Chlosyne gabbii',
        'Chlosyne harrisii',
        'Chlosyne hippodrome',
        'Chlosyne lacinia',
        'Chlosyne nycteis',
        'Chlosyne palla',
        'Chlosyne rosita',
        'Coenonympha pamphilus',
        'Coenonympha tullia',
        'Colobura dirce',
        'Cupha erymanthis',
        'Danaus genutia',
        'Danaus gilippus',
        'Danaus petilia',
        'Danaus plexippus',
        'Diaethria clymena',
        'Dione moneta',
        'Doleschallia bisaltide',
        'Doxocopa laure',
        'Doxocopa pavon',
        'Dryadula phaetusa',
        'Dryas iulia',
        'Dynamine postverta',
        'Elymnias hypermnestra',
        'Epiphile adrasta',
        'Erebia ligea',
        'Eresia phillyra',
        'Eueides isabella',
        'Eunica monima',
        'Eunica tatila',
        'Euphydryas aurinia',
        'Euphydryas chalcedona',
        'Euphydryas phaeton',
        'Euploea core',
        'Euploea mulciber',
        'Euptoieta claudia',
        'Euptoieta hegesia',
        'Euthalia aconthea',
        'Faunis canens',
        'Greta morgane',
        'Hamadryas amphinome',
        'Hamadryas feronia',
        'Hamadryas glauconome',
        'Hamadryas guatemalena',
        'Hamanumida daedalus',
        'Heliconius charithonia',
        'Heliconius hortense',
        'Hermeuptychia hermes',
        'Hermeuptychia hermybius',
        'Heteronympha merope',
        'Hipparchia fidia',
        'Hipparchia semele',
        'Hipparchia statilinus',
        'Historis odius',
        'Hypanartia lethe',
        'Hypolimnas anomala',
        'Hypolimnas misippus',
        'Ideopsis vulgaris',
        'Issoria lathonia',
        'Junonia atlites',
        'Junonia coenia',
        'Junonia hedonia',
        'Junonia hierta',
        'Junonia iphita',
        'Junonia lemonias',
        'Junonia oenone',
        'Junonia orithya',
        'Junonia terea',
        'Junonia villida',
        'Kaniska canace',
        'Lasiommata maera',
        'Lasiommata megera',
        'Lethe anthedon',
        'Lethe eurydice',
        'Lethe portlandia',
        'Lexias pardalis',
        'Libytheana carinenta',
        'Limenitis archippus',
        'Limenitis arthemis',
        'Limenitis camilla',
        'Limenitis lorquini',
        'Manataria hercyna',
        'Maniola jurtina',
        'Marpesia petreus',
        'Megisto cymela',
        'Melanargia galathea',
        'Melitaea athalia',
        'Melitaea cinxia',
        'Melitaea phoebe',
        'Mestra amymone',
        'Minois dryas',
        'Moduza procris',
        'Morpho helenor',
        'Morpho polyphemus',
        'Mycalesis mineus',
        'Mycalesis perseoides',
        'Myscelia cyananthe',
        'Myscelia ethusa',
        'Neptis hylas',
        'Nymphalis antiopa',
        'Nymphalis californica',
        'Nymphalis l-album',
        'Nymphalis polychloros',
        'Orsotriaena medus',
        'Pararge aegeria',
        'Pareuptychia ocirrhoe',
        'Phaedyma columella',
        'Phalanta phalantha',
        'Phyciodes cocyta',
        'Phyciodes mylitta',
        'Phyciodes phaon',
        'Phyciodes pulchella',
        'Phyciodes tharos',
        'Polygonia c-album',
        'Polygonia c-aureum',
        'Polygonia comma',
        'Polygonia interrogationis',
        'Polygonia satyrus',
        'Pyronia bathseba',
        'Pyronia cecilia',
        'Pyrrhogyra neaerea',
        'Satyrus ferula',
        'Siderone galanthis',
        'Siproeta epaphus',
        'Siproeta stelenes',
        'Smyrna blomfildia',
        'Speyeria atlantis',
        'Speyeria callippe',
        'Speyeria coronis',
        'Speyeria cybele',
        'Tanaecia pelea',
        'Tirumala limniace',
        'Vanessa annabella',
        'Vanessa atalanta',
        'Vanessa cardui',
        'Vanessa gonerilla',
        'Vanessa itea',
        'Vanessa virginiensis',
        'Vindula dejone',
        'Ypthima baldus',
        'Ypthima horsfieldii',
        'Ypthima huebneri',
        'Agriophara confertella',
        'Endrosis sarcitrella',
        'Hofmannophila pseudospretella',
        'Olbonoma triptycha',
        'Promalactis suzukiella',
        'Battus philenor',
        'Eurytides marcellus',
        'Eurytides philolaus',
        'Graphium agamemnon',
        'Graphium sarpedon',
        'Iphiclides feisthamelii',
        'Pachliopta aristolochiae',
        'Papilio aegeus',
        'Papilio anchisiades',
        'Papilio canadensis',
        'Papilio clytia',
        'Papilio cresphontes',
        'Papilio demoleus',
        'Papilio eurymedon',
        'Papilio garamas',
        'Papilio glaucus',
        'Papilio machaon',
        'Papilio memnon',
        'Papilio multicaudata',
        'Papilio palamedes',
        'Papilio paris',
        'Papilio polyxenes',
        'Papilio rogeri',
        'Papilio rumiko',
        'Papilio rutulus',
        'Papilio thoas',
        'Papilio troilus',
        'Papilio xuthus',
        'Papilio zelicaon',
        'Parides photinus',
        'Parnassius apollo',
        'Parnassius smintheus',
        'Protographium epidaus',
        'Troides helena',
        'Zerynthia cassandra',
        'Zerynthia polyxena',
        'Zerynthia rumina',
        'Abaeis nicippe',
        'Anthocharis sara',
        'Aphrissa statira',
        'Appias libythea',
        'Ascia monuste',
        'Belenois aurota',
        'Belenois java',
        'Catasticta teutila',
        'Catopsilia florella',
        'Catopsilia pomona',
        'Catopsilia pyranthe',
        'Cepora nerissa',
        'Colias croceus',
        'Colias eurytheme',
        'Colias philodice',
        'Delias hyparete',
        'Delias pasithoe',
        'Euchloe ausonides',
        'Eurema blanda',
        'Eurema salome',
        'Ganyra josephina',
        'Glutophrissa drusilla',
        'Gonepteryx cleopatra',
        'Gonepteryx rhamni',
        'Kricogonia lyside',
        'Leptophobia aripa',
        'Leptosia nina',
        'Nathalis iole',
        'Phoebis agarithe',
        'Phoebis argante',
        'Phoebis sennae',
        'Pieris brassicae',
        'Pieris canidia',
        'Pieris marginalis',
        'Pieris napi',
        'Pieris rapae',
        'Pieris virginiensis',
        'Pontia beckerii',
        'Pontia daplidice',
        'Pontia edusa',
        'Pontia occidentalis',
        'Pontia protodice',
        'Pyrisitia lisa',
        'Pyrisitia nise',
        'Pyrisitia proterpia',
        'Zerene cesonia',
        'Zerene eurydice',
        'Plutella xylostella',
        'Psyche casta',
        'Thyridopteryx ephemeraeformis',
        'Anstenoptilia marmarodactyla',
        'Emmelina monodactyla',
        'Gillmeria pallidactyla',
        'Lantanophaga pusillidactylus',
        'Platyptilia carduidactylus',
        'Pterophorus pentadactyla',
        'Sphenarches anisodactylus',
        'Aglossa costiferalis',
        'Aglossa disciferalis',
        'Arta olivalis',
        'Arta statalis',
        'Clydonopteron sacculana',
        'Condylolomia participalis',
        'Elasmopalpus lignosellus',
        'Endotricha flammealis',
        'Endotricha mesenterialis',
        'Endotricha pyrosalis',
        'Epipaschia superatalis',
        'Eulogia ochrifrontella',
        'Galasa nigrinodis',
        'Hypsopygia albidalis',
        'Hypsopygia binodulalis',
        'Hypsopygia costalis',
        'Hypsopygia olinalis',
        'Moodna ostrinella',
        'Oneida lunulalis',
        'Pococera asperatella',
        'Scenedra decoratalis',
        'Spectrotrota fimbrialis',
        'Tosale oviplagalis',
        'Abisara echerius',
        'Anteros carausius',
        'Apodemia mejicanus',
        'Apodemia mormo',
        'Apodemia palmerii',
        'Apodemia virgulti',
        'Calephelis nemesis',
        'Calephelis perditalis',
        'Calephelis virginiensis',
        'Caria ino',
        'Emesis emesia',
        'Hamearis lucina',
        'Lasaia sula',
        'Melanis acroleuca',
        'Melanis cephise',
        'Rhetus arcius',
        'Zemeros flegyas',
        'Actias luna',
        'Anisota stigma',
        'Antheraea polyphemus',
        'Arsenura armida',
        'Attacus atlas',
        'Automeris io',
        'Callosamia angulifera',
        'Dryocampa rubicunda',
        'Eacles imperialis',
        'Eupackardia calleta',
        'Hemileuca maia',
        'Hyalophora cecropia',
        'Opodiphthera eucalypti',
        'Saturnia pyri',
        'Sphingicampa hubbardi',
        'Scythris trivinctella',
        'Synanthedon acerni',
        'Acherontia atropos',
        'Amorpha juglandis',
        'Ceratomia amyntor',
        'Ceratomia undulosa',
        'Darapsa myron',
        'Deilephila elpenor',
        'Dolba hyloeus',
        'Enyo lugubris',
        'Erinnyis ello',
        'Erinnyis obscura',
        'Eumorpha labruscae',
        'Hemaris diffinis',
        'Hemaris thetis',
        'Hemaris thysbe',
        'Hippotion celerio',
        'Hippotion scrofa',
        'Hyles euphorbiae',
        'Hyles lineata',
        'Laothoe populi',
        'Macroglossum stellatarum',
        'Manduca sexta',
        'Mimas tiliae',
        'Pachylia ficus',
        'Paonias excaecata',
        'Paonias myops',
        'Paratrea plebeja',
        'Smerinthus jamaicensis',
        'Sphecodina abbottii',
        'Sphinx chersis',
        'Sphinx dollii',
        'Sphinx perelegans',
        'Theretra latreillii',
        'Theretra nessus',
        'Xylophanes falco',
        'Thyris maculata',
        'Acrolophus popeanella',
        'Homostinea curviliniella',
        'Monopis pavlovskii',
        'Tinea apicimaculella',
        'Acleris forsskaleana',
        'Acleris nivisellana',
        'Acleris semipurpurana',
        'Amorbia humerosana',
        'Archips argyrospila',
        'Archips cerasivorana',
        'Archips grisea',
        'Archips purpurana',
        'Archips semiferana',
        'Argyrotaenia alisellana',
        'Argyrotaenia mariana',
        'Argyrotaenia quercifoliana',
        'Argyrotaenia velutinana',
        'Cenopis pettitana',
        'Chimoptesis pennsylvaniana',
        'Choristoneura houstonana',
        'Clepsis melaleucana',
        'Clepsis peritana',
        'Clepsis persicana',
        'Clepsis virescana',
        'Cnephasia jactatana',
        'Coelostathma discopunctana',
        'Crocidosema plebejana',
        'Cydia caryana',
        'Cydia latiferreana',
        'Cydia pomonella',
        'Ecdytolopha insiticiana',
        'Ecdytolopha mana',
        'Endothenia hebesana',
        'Epiblema otiosana',
        'Epiblema scudderiana',
        'Epiblema strenuana',
        'Epiphyas postvittana',
        'Eucosma ochroterminana',
        'Eucosma parmatana',
        'Eucosma tomonana',
        'Eumarozia malachitana',
        'Gymnandrosoma punctidiscanum',
        'Isotenes miserana',
        'Olethreutes bipartitana',
        'Olethreutes fasciatana',
        'Pandemis limitata',
        'Pelochrista derelicta',
        'Pelochrista dorsisignatana',
        'Pelochrista similiana',
        'Platynota exasperatana',
        'Platynota flavedana',
        'Platynota rostrana',
        'Proteoteras aesculana',
        'Sparganothis sulfureana',
        'Sparganothoides lentiginosana',
        'Syndemis afflictana',
        'Callizzia amorata',
        'Micronia aculeata',
        'Urania fulgens',
        'Ypsolopha dentella',
        'Cyclosia papilionaris',
        'Zygaena filipendulae',
        'Empusa pennata',
        'Gonatista grisea',
        'Hierodula patellifera',
        'Mantis religiosa',
        'Miomantis caffra',
        'Phyllovates chlorophaea',
        'Stagmomantis californica',
        'Stagmomantis carolina',
        'Stagmomantis limbata',
        'Tenodera sinensis',
        'Iris oratoria',
        'Thesprotia graminis',
        'Archichauliodes diversus',
        'Chauliodes pectinicornis',
        'Chauliodes rastricornis',
        'Libelloides coccajus',
        'Chrysoperla carnea',
        'Micromus tasmaniae',
        'Dicromantispa interrupta',
        'Dicromantispa sayi',
        'Zeugomantispa minuta',
        'Adversaeschna brevistyla',
        'Aeshna clepsydra',
        'Aeshna eremita',
        'Aeshna juncea',
        'Aeshna tuberculifera',
        'Aeshna umbrosa',
        'Aeshna verticalis',
        'Aeshna walkeri',
        'Anax junius',
        'Anax longipes',
        'Anax parthenope',
        'Coryphaeschna ingens',
        'Epiaeschna heros',
        'Gynacantha nervosa',
        'Nasiaeschna pentacantha',
        'Rhionaeschna multicolor',
        'Rhionaeschna mutata',
        'Austroargiolestes icteromelas',
        'Calopteryx aequabilis',
        'Calopteryx dimidiata',
        'Calopteryx haemorrhoidalis',
        'Calopteryx maculata',
        'Calopteryx splendens',
        'Calopteryx virgo',
        'Hetaerina americana',
        'Hetaerina titia',
        'Hetaerina vulnerata',
        'Acanthagrion quadratum',
        'Amphiagrion abbreviatum',
        'Amphiagrion saucium',
        'Argia agrioides',
        'Argia alberta',
        'Argia anceps',
        'Argia apicalis',
        'Argia bipunctulata',
        'Argia emma',
        'Argia fumipennis',
        'Argia hinei',
        'Argia immunda',
        'Argia lugens',
        'Argia moesta',
        'Argia nahuana',
        'Argia oenea',
        'Argia pallens',
        'Argia plana',
        'Argia rhoadsi',
        'Argia sedula',
        'Argia tibialis',
        'Argia translata',
        'Argia vivida',
        'Ceriagrion cerinorubellum',
        'Chromagrion conditum',
        'Coenagrion resolutum',
        'Enallagma annexum',
        'Enallagma antennatum',
        'Enallagma aspersum',
        'Enallagma basidens',
        'Enallagma boreale',
        'Enallagma carunculatum',
        'Enallagma civile',
        'Enallagma divagans',
        'Enallagma durum',
        'Enallagma ebrium',
        'Enallagma exsulans',
        'Enallagma hageni',
        'Enallagma novaehispaniae',
        'Enallagma praevarum',
        'Enallagma signatum',
        'Enallagma traviatum',
        'Hesperagrion heterodoxum',
        'Ischnura aurora',
        'Ischnura barberi',
        'Ischnura cervula',
        'Ischnura demorsa',
        'Ischnura elegans',
        'Ischnura erratica',
        'Ischnura hastata',
        'Ischnura heterosticta',
        'Ischnura kellicotti',
        'Ischnura perparva',
        'Ischnura posita',
        'Ischnura ramburii',
        'Ischnura senegalensis',
        'Ischnura verticalis',
        'Nehalennia gracilis',
        'Protoneura cara',
        'Pseudagrion microcephalum',
        'Telebasis salva',
        'Xanthocnemis zealandica',
        'Cordulegaster boltonii',
        'Cordulegaster maculata',
        'Cordulegaster obliqua',
        'Epitheca cynosura',
        'Epitheca petechialis',
        'Epitheca semiaquea',
        'Epitheca spinigera',
        'Hemicordulia australiae',
        'Somatochlora linearis',
        'Somatochlora tenebrosa',
        'Somatochlora walshii',
        'Aphylla angustifolia',
        'Aphylla williamsoni',
        'Arigomphus furcifer',
        'Arigomphus submedianus',
        'Dromogomphus spinosus',
        'Dromogomphus spoliatus',
        'Erpetogomphus compositus',
        'Erpetogomphus designatus',
        'Erpetogomphus eutainia',
        'Gomphurus externus',
        'Gomphurus fraternus',
        'Gomphurus vastus',
        'Hagenius brevistylus',
        'Onychogomphus forcipatus',
        'Onychogomphus uncatus',
        'Ophiogomphus bison',
        'Ophiogomphus severus',
        'Phanogomphus borealis',
        'Phanogomphus exilis',
        'Phanogomphus graslinellus',
        'Phanogomphus kurilis',
        'Phanogomphus lividus',
        'Phanogomphus militaris',
        'Phanogomphus oklahomensis',
        'Phanogomphus spicatus',
        'Phyllogomphoides albrighti',
        'Phyllogomphoides stigmatus',
        'Progomphus borealis',
        'Progomphus obscurus',
        'Stylogomphus albistylus',
        'Stylurus plagiatus',
        'Archilestes californicus',
        'Archilestes grandis',
        'Chalcolestes viridis',
        'Lestes alacer',
        'Lestes australis',
        'Lestes eurinus',
        'Lestes forcipatus',
        'Lestes forficula',
        'Lestes inaequalis',
        'Lestes unguiculatus',
        'Lestes vigilax',
        'Sympecma fusca',
        'Acisoma panorpoides',
        'Brachydiplax chalybea',
        'Brachymesia furcata',
        'Brachymesia gravida',
        'Brachythemis contaminata',
        'Brachythemis leucosticta',
        'Brechmorhoga mendax',
        'Cannaphila insularis',
        'Celithemis amanda',
        'Celithemis elisa',
        'Celithemis eponina',
        'Celithemis ornata',
        'Crocothemis erythraea',
        'Crocothemis servilia',
        'Diplacodes bipunctata',
        'Dythemis fugax',
        'Dythemis nigrescens',
        'Dythemis sterilis',
        'Dythemis velox',
        'Erythemis attala',
        'Erythemis collocata',
        'Erythemis peruviana',
        'Erythemis plebeja',
        'Erythemis simplicicollis',
        'Erythemis vesiculosa',
        'Erythrodiplax basifusca',
        'Erythrodiplax berenice',
        'Erythrodiplax funerea',
        'Erythrodiplax umbrata',
        'Ladona deplanata',
        'Ladona julia',
        'Lathrecista asiatica',
        'Leucorrhinia glacialis',
        'Leucorrhinia intacta',
        'Libellula auripennis',
        'Libellula axilena',
        'Libellula comanche',
        'Libellula croceipennis',
        'Libellula cyanea',
        'Libellula depressa',
        'Libellula flavida',
        'Libellula forensis',
        'Libellula fulva',
        'Libellula incesta',
        'Libellula luctuosa',
        'Libellula needhami',
        'Libellula pulchella',
        'Libellula quadrimaculata',
        'Libellula saturata',
        'Libellula semifasciata',
        'Libellula vibrans',
        'Macrodiplax balteata',
        'Miathyria marcella',
        'Micrathyria aequalis',
        'Micrathyria hagenii',
        'Neurothemis fluctuans',
        'Neurothemis fulvia',
        'Neurothemis tullia',
        'Orthemis discolor',
        'Orthemis ferruginea',
        'Orthetrum albistylum',
        'Orthetrum brunneum',
        'Orthetrum caledonicum',
        'Orthetrum chrysis',
        'Orthetrum chrysostigma',
        'Orthetrum coerulescens',
        'Orthetrum glaucum',
        'Orthetrum luzonicum',
        'Orthetrum pruinosum',
        'Orthetrum testaceum',
        'Pachydiplax longipennis',
        'Palpopleura lucia',
        'Paltothemis lineatipes',
        'Pantala flavescens',
        'Pantala hymenaea',
        'Perithemis domitia',
        'Perithemis intensa',
        'Perithemis tenera',
        'Plathemis lydia',
        'Plathemis subornata',
        'Potamarcha congener',
        'Pseudoleon superbus',
        'Rhyothemis phyllis',
        'Rhyothemis variegata',
        'Sympetrum ambiguum',
        'Sympetrum corruptum',
        'Sympetrum illotum',
        'Sympetrum meridionale',
        'Sympetrum obtrusum',
        'Sympetrum pallipes',
        'Sympetrum pedemontanum',
        'Sympetrum rubicundulum',
        'Sympetrum semicinctum',
        'Sympetrum vicinum',
        'Tholymis tillarga',
        'Tramea calverti',
        'Tramea lacerata',
        'Tramea onusta',
        'Trithemis arteriosa',
        'Trithemis aurora',
        'Trithemis festiva',
        'Trithemis kirbyi',
        'Tyriobapta torrida',
        'Didymops transversa',
        'Macromia illinoiensis',
        'Macromia taeniolata',
        'Tachopteryx thoreyi',
        'Uropetala carovei',
        'Platycnemis pennipes',
        'Acrida cinerea',
        'Amblytropidia mysteca',
        'Camnula pellucida',
        'Chimarocephala pacifica',
        'Chorthippus brunneus',
        'Chortophaga australior',
        'Chortophaga viridifasciata',
        'Dichromorpha viridis',
        'Dissosteira pictipennis',
        'Hesperotettix speciosus',
        'Hesperotettix viridis',
        'Hippiscus ocelote',
        'Lactista gibbosus',
        'Locusta migratoria',
        'Melanoplus bivittatus',
        'Melanoplus differentialis',
        'Melanoplus femurrubrum',
        'Melanoplus punctulatus',
        'Oedipoda caerulescens',
        'Paroxya clavuliger',
        'Pezotettix giornae',
        'Pseudochorthippus curtipennis',
        'Pseudochorthippus parallelus',
        'Schistocerca lineata',
        'Schistocerca nitens',
        'Syrbula montezuma',
        'Trimerotropis fontana',
        'Trimerotropis maritima',
        'Trimerotropis pallidipennis',
        'Trimerotropis verruculata',
        'Valanga nigricornis',
        'Xenocatantops humilis',
        'Hemideina crassidens',
        'Gryllodes sigillatus',
        'Gryllus campestris',
        'Gryllus pennsylvanicus',
        'Hapithus agitator',
        'Neoxabea bipunctata',
        'Nisitrus vittatus',
        'Oecanthus nigricornis',
        'Oecanthus niveus',
        'Orocharis saltator',
        'Velarifictorus micado',
        'Sphenarium purpurascens',
        'Diestrammena asynamora',
        'Chromacris colorata',
        'Dracotettix monstrosus',
        'Romalea microptera',
        'Taeniopoda tamaulipensis',
        'Tropidacris cristata',
        'Stenopelmatus fuscus',
        'Anabrus simplex',
        'Conocephalus fasciatus',
        'Conocephalus fuscus',
        'Conocephalus strictus',
        'Decticus verrucivorus',
        'Leptophyes punctatissima',
        'Meconema meridionale',
        'Meconema thalassinum',
        'Microcentrum retinerve',
        'Neoconocephalus triops',
        'Orchelimum nigripes',
        'Orchelimum pulchellum',
        'Pholidoptera griseoaptera',
        'Roeseliana roeselii',
        'Scudderia mexicana',
        'Scudderia septentrionalis',
        'Carausius morosus',
        'Clitarchus hookeri',
        'Cerastipsocus venosus',
        'Graphopsocus cruciatus',
        'Lepisma saccharina',
        'Blepharipoda occidentalis',
        'Orconectes virilis',
        'Procambarus acutus',
        'Procambarus clarkii',
        'Cancer irroratus',
        'Cancer productus',
        'Glebocarcinus oregonensis',
        'Romaleon antennarium',
        'Coenobita compressus',
        'Clibanarius vittatus',
        'Pugettia producta',
        'Grapsus tenuicrustatus',
        'Leptograpsus variegatus',
        'Pachygrapsus crassipes',
        'Emerita analoga',
        'Emerita talpoida',
        'Stenorhynchus seticornis',
        'Menippe mercenaria',
        'Homarus americanus',
        'Leptuca pugilator',
        'Minuca pugnax',
        'Pagurus granosimanus',
        'Pagurus hirsutiusculus',
        'Pagurus novizealandiae',
        'Palaemon affinis',
        'Jasus edwardsii',
        'Panulirus argus',
        'Lophopanopeus bellus',
        'Paranephrops planifrons',
        'Loxorhynchus grandis',
        'Guinusia chabrus',
        'Petrolisthes elongatus',
        'Arenaeus cribrarius',
        'Callinectes sapidus',
        'Ovalipes catharus',
        'Ovalipes ocellatus',
        'Portunus pelagicus',
        'Parribacus antarcticus',
        'Aratus pisonii',
        'Stenopus hispidus',
        'Hemigrapsus crenulatus',
        'Hemigrapsus nudus',
        'Hemigrapsus sanguineus',
        'Hemigrapsus sexdentatus',
        'Armadillidium nasatum',
        'Armadillidium vulgare',
        'Idotea resecata',
        'Pentidotea wosnesenskii',
        'Ligia exotica',
        'Porcellio dilatatus',
        'Porcellio laevis',
        'Porcellio spinicornis',
        'Isocladus armatus',
        'Trachelipus rathkii',
        'Limulus polyphemus',
        'Integripelta bilabiata',
        'Anguilla australis',
        'Anguilla dieffenbachii',
        'Heteroconger hassi',
        'Echidna nebulosa',
        'Gymnothorax eurostus',
        'Gymnothorax flavimarginatus',
        'Gymnothorax funebris',
        'Gymnothorax javanicus',
        'Gymnothorax meleagris',
        'Gymnothorax prasinus',
        'Muraena helena',
        'Rhinomuraena quaesita',
        'Latropiscis purpurissatus',
        'Synodus variegatus',
        'Strongylura marina',
        'Holocentrus rufus',
        'Neoniphon sammara',
        'Dorosoma cepedianum',
        'Catostomus commersonii',
        'Hypentelium nigricans',
        'Campostoma anomalum',
        'Cyprinus carpio',
        'Luxilus chrysocephalus',
        'Notemigonus crysoleucas',
        'Rhinichthys atratulus',
        'Semotilus atromaculatus',
        'Fundulus diaphanus',
        'Fundulus heteroclitus',
        'Fundulus notatus',
        'Gambusia affinis',
        'Esox americanus',
        'Esox niger',
        'Umbra limi',
        'Culaea inconstans',
        'Lepisosteus platyrhincus',
        'Antennarius commerson',
        'Antennarius striatus',
        'Mugil cephalus',
        'Acanthurus achilles',
        'Acanthurus bahianus',
        'Acanthurus chirurgus',
        'Acanthurus dussumieri',
        'Acanthurus lineatus',
        'Acanthurus nigricans',
        'Acanthurus nigrofuscus',
        'Acanthurus olivaceus',
        'Acanthurus triostegus',
        'Acanthurus xanthopterus',
        'Ctenochaetus striatus',
        'Ctenochaetus strigosus',
        'Naso brevirostris',
        'Naso unicornis',
        'Prionurus microlepidotus',
        'Zebrasoma flavescens',
        'Zebrasoma scopas',
        'Zebrasoma velifer',
        'Aplodactylus lophodon',
        'Ostorhinchus limenus',
        'Exallias brevis',
        'Plagiotremus rhinorhynchos',
        'Plagiotremus tapeinosoma',
        'Caesio caerulaurea',
        'Eocallionymus papilio',
        'Caranx ignobilis',
        'Caranx ruber',
        'Caranx sexfasciatus',
        'Pseudocaranx georgianus',
        'Seriola lalandi',
        'Trachurus novaezelandiae',
        'Ambloplites rupestris',
        'Lepomis auritus',
        'Lepomis cyanellus',
        'Lepomis gulosus',
        'Lepomis macrochirus',
        'Lepomis microlophus',
        'Micropterus punctulatus',
        'Micropterus salmoides',
        'Pomoxis annularis',
        'Centropomus undecimalis',
        'Chaetodon auriga',
        'Chaetodon baronessa',
        'Chaetodon citrinellus',
        'Chaetodon ephippium',
        'Chaetodon flavirostris',
        'Chaetodon fremblii',
        'Chaetodon kleinii',
        'Chaetodon lineolatus',
        'Chaetodon lunula',
        'Chaetodon lunulatus',
        'Chaetodon melannotus',
        'Chaetodon ocellatus',
        'Chaetodon ornatissimus',
        'Chaetodon quadrimaculatus',
        'Chaetodon rafflesii',
        'Chaetodon striatus',
        'Chaetodon trifascialis',
        'Chaetodon ulietensis',
        'Chaetodon unimaculatus',
        'Chaetodon vagabundus',
        'Chelmonops truncatus',
        'Forcipiger flavissimus',
        'Hemitaurichthys polylepis',
        'Heniochus acuminatus',
        'Heniochus chrysostomus',
        'Heniochus diphreutes',
        'Heniochus varius',
        'Cheilodactylus fuscus',
        'Cheilodactylus nigripes',
        'Cheilodactylus vestitus',
        'Dactylophora nigricans',
        'Chironemus marmoratus',
        'Herichthys cyanoguttatus',
        'Oreochromis aureus',
        'Cirrhitichthys aprinus',
        'Cirrhitichthys falco',
        'Cirrhitichthys oxycephalus',
        'Paracirrhites arcatus',
        'Paracirrhites forsteri',
        'Coryphaena hippurus',
        'Dinolestes lewini',
        'Echeneis naucrates',
        'Cymatogaster aggregata',
        'Enoplosus armatus',
        'Platax teira',
        'Gerres subfasciatus',
        'Amblygobius phalaena',
        'Istigobius hoesei',
        'Neogobius melanostomus',
        'Padogobius bonelli',
        'Valenciennea strigata',
        'Anisotremus virginicus',
        'Haemulon chrysargyreum',
        'Haemulon flavolineatum',
        'Plectorhinchus chaetodonoides',
        'Plectorhinchus vittatus',
        'Atypichthys strigatus',
        'Girella nigricans',
        'Girella tricuspidata',
        'Girella zebra',
        'Microcanthus strigatus',
        'Scorpis lineolata',
        'Tilodon sexfasciatus',
        'Achoerodus viridis',
        'Bodianus rufus',
        'Cheilinus fasciatus',
        'Cheilinus undulatus',
        'Cheilio inermis',
        'Coris gaimard',
        'Coris julis',
        'Coris picta',
        'Epibulus insidiator',
        'Eupetrichthys angustipes',
        'Gomphosus varius',
        'Halichoeres bivittatus',
        'Halichoeres garnoti',
        'Halichoeres hortulanus',
        'Hemigymnus fasciatus',
        'Hemigymnus melapterus',
        'Labroides dimidiatus',
        'Macropharyngodon meleagris',
        'Notolabrus gymnogenis',
        'Novaculichthys taeniourus',
        'Ophthalmolepis lineolata',
        'Oxycheilinus bimaculatus',
        'Oxycheilinus digramma',
        'Pictilabrus laticlavius',
        'Pseudolabrus guentheri',
        'Semicossyphus pulcher',
        'Stethojulis bandanensis',
        'Stethojulis interrupta',
        'Thalassoma amblycephalum',
        'Thalassoma duperrey',
        'Thalassoma hardwicke',
        'Thalassoma lucasanum',
        'Thalassoma lunare',
        'Thalassoma lutescens',
        'Thalassoma pavo',
        'Thalassoma trilobatum',
        'Lethrinus nebulosus',
        'Monotaxis grandoculis',
        'Lutjanus apodus',
        'Lutjanus bohar',
        'Lutjanus fulvus',
        'Lutjanus kasmira',
        'Lutjanus synagris',
        'Macolor macularis',
        'Ocyurus chrysurus',
        'Nemateleotris magnifica',
        'Ptereleotris evides',
        'Monodactylus argenteus',
        'Schuettea scalaripinnis',
        'Morone americana',
        'Morone chrysops',
        'Mulloidichthys flavolineatus',
        'Mulloidichthys martinicus',
        'Mulloidichthys vanicolensis',
        'Mullus surmuletus',
        'Parupeneus cyclostomus',
        'Parupeneus multifasciatus',
        'Parupeneus spilurus',
        'Upeneichthys lineatus',
        'Upeneichthys vlamingii',
        'Upeneus tragula',
        'Scolopsis bilineata',
        'Heteroscarus acroptilus',
        'Pempheris affinis',
        'Pempheris compressa',
        'Etheostoma caeruleum',
        'Perca fluviatilis',
        'Sander vitreus',
        'Trachinops taeniatus',
        'Centropyge bicolor',
        'Centropyge bispinosa',
        'Genicanthus melanospilos',
        'Holacanthus ciliaris',
        'Holacanthus passer',
        'Pomacanthus arcuatus',
        'Pomacanthus imperator',
        'Pomacanthus paru',
        'Pomacanthus semicirculatus',
        'Pygoplites diacanthus',
        'Abudefduf sexfasciatus',
        'Abudefduf troschelii',
        'Abudefduf vaigiensis',
        'Amphiprion bicinctus',
        'Amphiprion chrysopterus',
        'Amphiprion clarkii',
        'Amphiprion melanopus',
        'Amphiprion ocellaris',
        'Amphiprion perideraion',
        'Chromis chromis',
        'Chromis cyanea',
        'Chromis hypsilepis',
        'Chromis multilineata',
        'Chromis viridis',
        'Dascyllus albisella',
        'Dascyllus aruanus',
        'Dascyllus reticulatus',
        'Dascyllus trimaculatus',
        'Hypsypops rubicundus',
        'Mecaenichthys immaculatus',
        'Microspathodon chrysurus',
        'Microspathodon dorsalis',
        'Parma microlepis',
        'Parma unifasciata',
        'Parma victoriae',
        'Plectroglyphidodon lacrymatus',
        'Pomacentrus coelestis',
        'Pomacentrus moluccensis',
        'Premnas biaculeatus',
        'Stegastes partitus',
        'Pomatomus saltatrix',
        'Priacanthus hamrur',
        'Cetoscarus ocellatus',
        'Chlorurus microrhinos',
        'Chlorurus sordidus',
        'Scarus frenatus',
        'Scarus ghobban',
        'Scarus iseri',
        'Scarus psittacus',
        'Sparisoma aurofrenatum',
        'Aplodinotus grunniens',
        'Argyrosomus japonicus',
        'Cynoscion nebulosus',
        'Pogonias cromis',
        'Sciaenops ocellatus',
        'Cephalopholis argus',
        'Cephalopholis miniata',
        'Epinephelus fasciatus',
        'Epinephelus merra',
        'Hypoplectrodes maccullochi',
        'Paralabrax clathratus',
        'Plectropomus leopardus',
        'Pseudanthias squamipinnis',
        'Sillago ciliata',
        'Acanthopagrus australis',
        'Archosargus probatocephalus',
        'Diplodus sargus',
        'Diplodus vulgaris',
        'Lagodon rhomboides',
        'Pagrus auratus',
        'Rhabdosargus sarba',
        'Sarpa salpa',
        'Sphyraena barracuda',
        'Sphyraena jello',
        'Bellapiscis medius',
        'Enneapterygius atrogulare',
        'Forsterygion lapillum',
        'Zanclus cornutus',
        'Oncorhynchus clarkii',
        'Oncorhynchus gorbuscha',
        'Oncorhynchus keta',
        'Oncorhynchus kisutch',
        'Oncorhynchus mykiss',
        'Oncorhynchus tshawytscha',
        'Salmo trutta',
        'Salvelinus fontinalis',
        'Scorpaenichthys marmoratus',
        'Hexagrammos decagrammus',
        'Platycephalus fuscus',
        'Dendrochirus brachypterus',
        'Dendrochirus zebra',
        'Pterois antennata',
        'Pterois miles',
        'Scorpaena jacksoniensis',
        'Scorpaena plumieri',
        'Taenianotus triacanthus',
        'Centropogon australis',
        'Ariopsis felis',
        'Ameiurus natalis',
        'Ameiurus nebulosus',
        'Pylodictis olivaris',
        'Cnidoglanis macrocephalus',
        'Plotosus lineatus',
        'Aulostomus chinensis',
        'Aulostomus maculatus',
        'Solenostomus paradoxus',
        'Hippocampus abdominalis',
        'Hippocampus bargibanti',
        'Hippocampus breviceps',
        'Hippocampus denise',
        'Hippocampus erectus',
        'Hippocampus guttulatus',
        'Hippocampus histrix',
        'Hippocampus kuda',
        'Hippocampus pontohi',
        'Hippocampus reidi',
        'Hippocampus spinosissimus',
        'Hippocampus whitei',
        'Phyllopteryx taeniolatus',
        'Anoplocapros inermis',
        'Balistapus undulatus',
        'Balistoides conspicillum',
        'Balistoides viridescens',
        'Melichthys niger',
        'Melichthys vidua',
        'Odonus niger',
        'Rhinecanthus aculeatus',
        'Sufflamen bursa',
        'Sufflamen chrysopterum',
        'Allomycterus pilatus',
        'Chilomycterus schoepfii',
        'Dicotylichthys punctulatus',
        'Diodon nicthemerus',
        'Acanthaluteres vittiger',
        'Aluterus scriptus',
        'Brachaluteres jacksonianus',
        'Cantherhines dumerilii',
        'Cantherhines macrocerus',
        'Eubalichthys mosaicus',
        'Meuschenia freycineti',
        'Meuschenia hippocrepis',
        'Meuschenia trachylepis',
        'Monacanthus chinensis',
        'Oxymonacanthus longirostris',
        'Scobinichthys granulatus',
        'Acanthostracion quadricornis',
        'Lactophrys triqueter',
        'Lactoria fornasini',
        'Ostracion cubicus',
        'Ostracion meleagris',
        'Arothron hispidus',
        'Arothron meleagris',
        'Arothron nigropunctatus',
        'Canthigaster jactator',
        'Canthigaster rostrata',
        'Canthigaster valentini',
        'Sphoeroides testudineus',
        'Tetractenos glaber',
        'Ascaphus truei',
        'Bombina orientalis',
        'Bombina variegata',
        'Anaxyrus americanus',
        'Anaxyrus boreas',
        'Anaxyrus californicus',
        'Anaxyrus cognatus',
        'Anaxyrus fowleri',
        'Anaxyrus punctatus',
        'Anaxyrus speciosus',
        'Anaxyrus woodhousii',
        'Bufo bankorensis',
        'Bufo bufo',
        'Bufo gargarizans',
        'Bufo spinosus',
        'Bufotes variabilis',
        'Epidalea calamita',
        'Incilius alvarius',
        'Incilius marmoreus',
        'Incilius mazatlanensis',
        'Incilius nebulifer',
        'Incilius occidentalis',
        'Phrynoidis asper',
        'Rhaebo haematiticus',
        'Rhinella horribilis',
        'Rhinella marina',
        'Sclerophrys gutturalis',
        'Oophaga pumilio',
        'Fejervarya kawamurai',
        'Fejervarya limnocharis',
        'Eleutherodactylus coqui',
        'Acris blanchardi',
        'Dendropsophus microcephalus',
        'Hyla arborea',
        'Hyla arenicolor',
        'Hyla chrysoscelis',
        'Hyla cinerea',
        'Hyla eximia',
        'Hyla femoralis',
        'Hyla gratiosa',
        'Hyla immaculata',
        'Hyla japonica',
        'Hyla meridionalis',
        'Hyla versicolor',
        'Hyla wrightorum',
        'Osteopilus septentrionalis',
        'Pseudacris cadaverina',
        'Pseudacris clarkii',
        'Pseudacris crucifer',
        'Pseudacris feriarum',
        'Pseudacris fouquettei',
        'Pseudacris hypochondriaca',
        'Pseudacris regilla',
        'Pseudacris sierra',
        'Pseudacris triseriata',
        'Rheohyla miotympanum',
        'Scinax staufferi',
        'Smilisca baudinii',
        'Smilisca phaeota',
        'Tlalocohyla smithii',
        'Hyperolius marmoratus',
        'Leiopelma archeyi',
        'Leiopelma hochstetteri',
        'Leptodactylus melanonotus',
        'Leptodactylus savagei',
        'Gastrophryne carolinensis',
        'Gastrophryne olivacea',
        'Kaloula pulchra',
        'Pelobates fuscus',
        'Litoria ewingii',
        'Ranoidea aurea',
        'Ranoidea caerulea',
        'Ranoidea moorei',
        'Ranoidea raniformis',
        'Xenopus laevis',
        'Amietia fuscigula',
        'Chalcorana chalconota',
        'Glandirana emeljanovi',
        'Hylarana erythraea',
        'Hylarana latouchii',
        'Lithobates berlandieri',
        'Lithobates catesbeianus',
        'Lithobates clamitans',
        'Lithobates grylio',
        'Lithobates magnaocularis',
        'Lithobates palustris',
        'Lithobates pipiens',
        'Lithobates septentrionalis',
        'Lithobates sphenocephalus',
        'Lithobates sylvaticus',
        'Pelophylax bedriagae',
        'Pelophylax chosenicus',
        'Pelophylax nigromaculatus',
        'Rana cascadae',
        'Rana dalmatina',
        'Rana draytonii',
        'Rana italica',
        'Rana latastei',
        'Rana luteiventris',
        'Rana temporaria',
        'Rana uenoi',
        'Chiromantis xerampelina',
        'Polypedates leucomystax',
        'Polypedates megacephalus',
        'Rhinophrynus dorsalis',
        'Scaphiopus couchii',
        'Ambystoma californiense',
        'Ambystoma maculatum',
        'Ambystoma opacum',
        'Ambystoma talpoideum',
        'Dicamptodon ensatus',
        'Hynobius leechii',
        'Hynobius quelpaertensis',
        'Onychodactylus koreanus',
        'Aneides aeneus',
        'Aneides flavipunctatus',
        'Aneides lugubris',
        'Batrachoseps attenuatus',
        'Batrachoseps gavilanensis',
        'Desmognathus conanti',
        'Desmognathus monticola',
        'Desmognathus quadramaculatus',
        'Ensatina eschscholtzii',
        'Eurycea bislineata',
        'Eurycea guttolineata',
        'Eurycea quadridigitata',
        'Eurycea troglodytes',
        'Eurycea wilderae',
        'Karsenia koreana',
        'Plethodon albagula',
        'Plethodon cinereus',
        'Plethodon cylindraceus',
        'Plethodon dunni',
        'Plethodon serratus',
        'Speleomantes strinatii',
        'Lissotriton vulgaris',
        'Notophthalmus viridescens',
        'Salamandra lanzai',
        'Salamandra salamandra',
        'Salamandrina perspicillata',
        'Taricha granulosa',
        'Taricha torosa',
        'Triturus carnifex',
        'Siren intermedia',
        'Botrylloides violaceus',
        'Botryllus schlosseri',
        'Cnemidocarpa finmarkiensis',
        'Styela montereyensis',
        'Accipiter badius',
        'Accipiter cooperii',
        'Accipiter striatus',
        'Aegypius monachus',
        'Aquila audax',
        'Aquila chrysaetos',
        'Aquila nipalensis',
        'Aquila rapax',
        'Buteo augur',
        'Buteo buteo',
        'Buteo galapagoensis',
        'Buteo jamaicensis',
        'Buteo lagopus',
        'Buteo lineatus',
        'Buteo plagiatus',
        'Buteo platypterus',
        'Buteo regalis',
        'Buteo rufinus',
        'Buteo swainsoni',
        'Buteogallus anthracinus',
        'Buteogallus meridionalis',
        'Buteogallus urubitinga',
        'Chondrohierax uncinatus',
        'Circaetus cinereus',
        'Circaetus pectoralis',
        'Circus approximans',
        'Circus cyaneus',
        'Circus hudsonius',
        'Circus pygargus',
        'Elanus caeruleus',
        'Elanus leucurus',
        'Geranoaetus albicaudatus',
        'Geranoaetus melanoleucus',
        'Geranoaetus polyosoma',
        'Geranospiza caerulescens',
        'Gypaetus barbatus',
        'Gypohierax angolensis',
        'Gyps rueppelli',
        'Haliaeetus albicilla',
        'Haliaeetus leucocephalus',
        'Haliastur sphenurus',
        'Hieraaetus pennatus',
        'Ictinia mississippiensis',
        'Ictinia plumbea',
        'Kaupifalco monogrammicus',
        'Lophaetus occipitalis',
        'Melierax canorus',
        'Milvus migrans',
        'Milvus milvus',
        'Necrosyrtes monachus',
        'Neophron percnopterus',
        'Parabuteo unicinctus',
        'Pernis apivorus',
        'Pernis ptilorhynchus',
        'Polemaetus bellicosus',
        'Polyboroides typus',
        'Pseudastur albicollis',
        'Rostrhamus sociabilis',
        'Rupornis magnirostris',
        'Spilornis cheela',
        'Spizaetus ornatus',
        'Torgos tracheliotos',
        'Trigonoceps occipitalis',
        'Pandion haliaetus',
        'Aix sponsa',
        'Alopochen aegyptiaca',
        'Amazonetta brasiliensis',
        'Anas acuta',
        'Anas castanea',
        'Anas crecca',
        'Anas erythrorhyncha',
        'Anas flavirostris',
        'Anas fulvigula',
        'Anas georgica',
        'Anas platyrhynchos',
        'Anas poecilorhyncha',
        'Anas rubripes',
        'Anas superciliosa',
        'Anas undulata',
        'Anas zonorhyncha',
        'Anser albifrons',
        'Anser anser',
        'Anser brachyrhynchus',
        'Anser caerulescens',
        'Anser cygnoides',
        'Anser fabalis',
        'Anser rossii',
        'Anser serrirostris',
        'Aythya affinis',
        'Aythya americana',
        'Aythya australis',
        'Aythya collaris',
        'Aythya ferina',
        'Aythya fuligula',
        'Aythya marila',
        'Aythya nyroca',
        'Aythya valisineria',
        'Branta bernicla',
        'Branta canadensis',
        'Branta hutchinsii',
        'Bucephala albeola',
        'Bucephala clangula',
        'Bucephala islandica',
        'Cairina moschata',
        'Chenonetta jubata',
        'Chloephaga picta',
        'Clangula hyemalis',
        'Coscoroba coscoroba',
        'Cygnus atratus',
        'Cygnus buccinator',
        'Cygnus columbianus',
        'Cygnus cygnus',
        'Cygnus melancoryphus',
        'Cygnus olor',
        'Dendrocygna autumnalis',
        'Dendrocygna javanica',
        'Histrionicus histrionicus',
        'Lophodytes cucullatus',
        'Mareca americana',
        'Mareca falcata',
        'Mareca penelope',
        'Mareca sibilatrix',
        'Mareca strepera',
        'Melanitta americana',
        'Melanitta perspicillata',
        'Merganetta armata',
        'Mergellus albellus',
        'Mergus merganser',
        'Mergus serrator',
        'Oxyura jamaicensis',
        'Plectropterus gambensis',
        'Sarkidiornis melanotos',
        'Somateria mollissima',
        'Somateria spectabilis',
        'Spatula clypeata',
        'Spatula cyanoptera',
        'Spatula discors',
        'Spatula querquedula',
        'Spatula rhynchotis',
        'Tadorna tadornoides',
        'Tadorna variegata',
        'Chauna torquata',
        'Anseranas semipalmata',
        'Anthracoceros albirostris',
        'Lophoceros alboterminatus',
        'Lophoceros fasciatus',
        'Lophoceros nasutus',
        'Tockus erythrorhynchus',
        'Tockus leucomelas',
        'Tockus rufirostris',
        'Bucorvus abyssinicus',
        'Phoeniculus purpureus',
        'Upupa epops',
        'Aeronautes saxatalis',
        'Chaetura pelagica',
        'Streptoprocne zonaris',
        'Antrostomus carolinensis',
        'Antrostomus vociferus',
        'Chordeiles minor',
        'Nyctibius grandis',
        'Nyctibius griseus',
        'Podargus strigoides',
        'Amazilia beryllina',
        'Amazilia candida',
        'Amazilia cyanocephala',
        'Amazilia rutila',
        'Amazilia violiceps',
        'Amazilia yucatanensis',
        'Anthracothorax nigricollis',
        'Anthracothorax prevostii',
        'Archilochus alexandri',
        'Archilochus colubris',
        'Calypte anna',
        'Calypte costae',
        'Campylopterus curvipennis',
        'Chlorostilbon canivetii',
        'Chlorostilbon lucidus',
        'Chlorostilbon ricordii',
        'Colibri coruscans',
        'Colibri cyanotus',
        'Colibri thalassinus',
        'Cynanthus latirostris',
        'Eugenes fulgens',
        'Eugenes spectabilis',
        'Heliodoxa jacula',
        'Heliomaster constantii',
        'Lampornis calolaemus',
        'Ocreatus underwoodii',
        'Orthorhyncus cristatus',
        'Panterpe insignis',
        'Phaethornis guy',
        'Phaethornis longirostris',
        'Selasphorus platycercus',
        'Selasphorus rufus',
        'Selasphorus sasin',
        'Thalurania colombica',
        'Cariama cristata',
        'Dromaius novaehollandiae',
        'Cathartes aura',
        'Coragyps atratus',
        'Gymnogyps californianus',
        'Vultur gryphus',
        'Brachyramphus marmoratus',
        'Cepphus columba',
        'Fratercula corniculata',
        'Ptychoramphus aleuticus',
        'Synthliboramphus antiquus',
        'Uria aalge',
        'Burhinus bistriatus',
        'Burhinus capensis',
        'Burhinus grallarius',
        'Burhinus senegalensis',
        'Burhinus vermiculatus',
        'Charadrius alexandrinus',
        'Charadrius bicinctus',
        'Charadrius collaris',
        'Charadrius marginatus',
        'Charadrius melodus',
        'Charadrius nivosus',
        'Charadrius obscurus',
        'Charadrius ruficapillus',
        'Charadrius semipalmatus',
        'Charadrius tricollaris',
        'Charadrius vociferus',
        'Charadrius wilsonia',
        'Elseyornis melanops',
        'Pluvialis apricaria',
        'Pluvialis fulva',
        'Pluvialis squatarola',
        'Vanellus albiceps',
        'Vanellus chilensis',
        'Vanellus coronatus',
        'Vanellus indicus',
        'Vanellus senegallus',
        'Vanellus vanellus',
        'Haematopus bachmani',
        'Haematopus fuliginosus',
        'Haematopus longirostris',
        'Haematopus palliatus',
        'Haematopus unicolor',
        'Jacana spinosa',
        'Anous minutus',
        'Chlidonias hybrida',
        'Chlidonias leucopterus',
        'Chlidonias niger',
        'Chroicocephalus bulleri',
        'Chroicocephalus cirrocephalus',
        'Chroicocephalus genei',
        'Chroicocephalus maculipennis',
        'Chroicocephalus novaehollandiae',
        'Chroicocephalus philadelphia',
        'Chroicocephalus ridibundus',
        'Chroicocephalus scopulinus',
        'Creagrus furcatus',
        'Hydrocoloeus minutus',
        'Hydroprogne caspia',
        'Ichthyaetus melanocephalus',
        'Larosterna inca',
        'Larus argentatus',
        'Larus belcheri',
        'Larus californicus',
        'Larus canus',
        'Larus crassirostris',
        'Larus delawarensis',
        'Larus dominicanus',
        'Larus fuscus',
        'Larus glaucescens',
        'Larus glaucoides',
        'Larus heermanni',
        'Larus hyperboreus',
        'Larus marinus',
        'Larus michahellis',
        'Larus occidentalis',
        'Larus pacificus',
        'Larus schistisagus',
        'Leucophaeus atricilla',
        'Leucophaeus pipixcan',
        'Onychoprion anaethetus',
        'Phaetusa simplex',
        'Rissa tridactyla',
        'Rynchops niger',
        'Sterna forsteri',
        'Sterna hirundo',
        'Sterna paradisaea',
        'Sterna striata',
        'Sternula albifrons',
        'Sternula antillarum',
        'Thalasseus elegans',
        'Thalasseus maximus',
        'Thalasseus sandvicensis',
        'Himantopus himantopus',
        'Himantopus leucocephalus',
        'Himantopus mexicanus',
        'Recurvirostra americana',
        'Rostratula benghalensis',
        'Actitis hypoleucos',
        'Actitis macularius',
        'Arenaria interpres',
        'Arenaria melanocephala',
        'Calidris acuminata',
        'Calidris alba',
        'Calidris alpina',
        'Calidris mauri',
        'Calidris melanotos',
        'Calidris minuta',
        'Calidris minutilla',
        'Calidris ptilocnemis',
        'Calidris pusilla',
        'Calidris ruficollis',
        'Calidris virgata',
        'Gallinago delicata',
        'Limnodromus griseus',
        'Limosa fedoa',
        'Numenius americanus',
        'Numenius madagascariensis',
        'Numenius phaeopus',
        'Phalaropus lobatus',
        'Phalaropus tricolor',
        'Tringa erythropus',
        'Tringa flavipes',
        'Tringa incana',
        'Tringa melanoleuca',
        'Tringa semipalmata',
        'Tringa solitaria',
        'Tringa stagnatilis',
        'Xenus cinereus',
        'Stercorarius antarcticus',
        'Stercorarius longicaudus',
        'Stercorarius maccormicki',
        'Anastomus oscitans',
        'Ciconia boyciana',
        'Ciconia ciconia',
        'Ciconia episcopus',
        'Ciconia maguari',
        'Ciconia nigra',
        'Leptoptilos javanicus',
        'Mycteria americana',
        'Mycteria leucocephala',
        'Colius striatus',
        'Chalcophaps indica',
        'Columba guinea',
        'Columba livia',
        'Columba oenas',
        'Columba palumbus',
        'Columbina inca',
        'Columbina passerina',
        'Columbina talpacoti',
        'Geopelia humeralis',
        'Geopelia placida',
        'Geopelia striata',
        'Hemiphaga novaeseelandiae',
        'Leptotila verreauxi',
        'Oena capensis',
        'Patagioenas cayennensis',
        'Patagioenas fasciata',
        'Patagioenas flavirostris',
        'Patagioenas maculosa',
        'Patagioenas picazuro',
        'Phaps chalcoptera',
        'Streptopelia capicola',
        'Streptopelia chinensis',
        'Streptopelia decaocto',
        'Streptopelia decipiens',
        'Streptopelia orientalis',
        'Streptopelia roseogrisea',
        'Streptopelia semitorquata',
        'Streptopelia senegalensis',
        'Treron calvus',
        'Treron vernans',
        'Turtur chalcospilos',
        'Zenaida asiatica',
        'Zenaida aurita',
        'Zenaida galapagoensis',
        'Zenaida macroura',
        'Alcedo atthis',
        'Chloroceryle amazona',
        'Chloroceryle americana',
        'Corythornis cristatus',
        'Halcyon albiventris',
        'Halcyon chelicuti',
        'Halcyon leucocephala',
        'Halcyon senegalensis',
        'Ispidina picta',
        'Megaceryle alcyon',
        'Megaceryle maxima',
        'Megaceryle torquata',
        'Pelargopsis capensis',
        'Todiramphus sanctus',
        'Coracias abyssinicus',
        'Coracias naevius',
        'Eurystomus glaucurus',
        'Eurystomus orientalis',
        'Merops bullockoides',
        'Merops nubicoides',
        'Merops ornatus',
        'Merops philippinus',
        'Baryphthengus martii',
        'Electron platyrhynchum',
        'Eumomota superciliosa',
        'Momotus aequatorialis',
        'Momotus coeruliceps',
        'Momotus mexicanus',
        'Centropus senegalensis',
        'Centropus sinensis',
        'Centropus superciliosus',
        'Chrysococcyx lucidus',
        'Coccyzus americanus',
        'Crotophaga major',
        'Crotophaga sulcirostris',
        'Eudynamys scolopaceus',
        'Geococcyx californianus',
        'Guira guira',
        'Piaya cayana',
        'Corythaixoides concolor',
        'Eurypyga helias',
        'Caracara cheriway',
        'Falco cenchroides',
        'Falco columbarius',
        'Falco naumanni',
        'Falco peregrinus',
        'Falco rusticolus',
        'Falco sparverius',
        'Falco tinnunculus',
        'Micrastur semitorquatus',
        'Milvago chimango',
        'Malacoptila panamensis',
        'Monasa nigrifrons',
        'Notharchus hyperrhynchus',
        'Ortalis cinereiceps',
        'Ortalis vetula',
        'Ortalis wagleri',
        'Megapodius reinwardt',
        'Callipepla californica',
        'Callipepla douglasii',
        'Callipepla gambelii',
        'Callipepla squamata',
        'Colinus virginianus',
        'Oreortyx pictus',
        'Alectoris chukar',
        'Bonasa umbellus',
        'Centrocercus urophasianus',
        'Dendroperdix sephaena',
        'Gallus gallus',
        'Lagopus leucura',
        'Lagopus muta',
        'Meleagris gallopavo',
        'Pavo cristatus',
        'Perdix perdix',
        'Phasianus colchicus',
        'Pternistis afer',
        'Tympanuchus phasianellus',
        'Gavia immer',
        'Gavia pacifica',
        'Gavia stellata',
        'Aramus guarauna',
        'Anthropoides paradiseus',
        'Antigone canadensis',
        'Grus grus',
        'Grus monacha',
        'Heliornis fulica',
        'Aramides albiventris',
        'Aramides cajaneus',
        'Fulica alai',
        'Fulica americana',
        'Fulica armillata',
        'Fulica atra',
        'Fulica cristata',
        'Gallinula chloropus',
        'Gallinula galeata',
        'Gallinula tenebrosa',
        'Gallirallus australis',
        'Gallirallus philippensis',
        'Porphyrio martinica',
        'Porphyrio melanotus',
        'Porzana carolina',
        'Rallus aquaticus',
        'Rallus elegans',
        'Rallus limicola',
        'Zapornia flavirostra',
        'Eupodotis afraoides',
        'Lissotis melanogaster',
        'Acanthisitta chloris',
        'Xenicus gilviventris',
        'Gerygone igata',
        'Sericornis frontalis',
        'Acrocephalus arundinaceus',
        'Acrocephalus schoenobaenus',
        'Acrocephalus scirpaceus',
        'Aegithalos caudatus',
        'Psaltriparus minimus',
        'Aegithina tiphia',
        'Alauda arvensis',
        'Eremophila alpestris',
        'Galerida cristata',
        'Lullula arborea',
        'Artamus leucorynchus',
        'Bombycilla cedrorum',
        'Bombycilla garrulus',
        'Buphagus africanus',
        'Buphagus erythrorhynchus',
        'Calcarius ornatus',
        'Plectrophenax nivalis',
        'Callaeas wilsoni',
        'Coracina novaehollandiae',
        'Cardinalis cardinalis',
        'Cardinalis sinuatus',
        'Habia fuscicauda',
        'Passerina amoena',
        'Passerina caerulea',
        'Passerina ciris',
        'Passerina cyanea',
        'Pheucticus chrysopeplus',
        'Pheucticus ludovicianus',
        'Pheucticus melanocephalus',
        'Piranga ludoviciana',
        'Piranga olivacea',
        'Piranga rubra',
        'Spiza americana',
        'Certhia americana',
        'Certhia brachydactyla',
        'Certhia familiaris',
        'Cinclus mexicanus',
        'Camaroptera brachyura',
        'Cisticola juncidis',
        'Orthotomus sutorius',
        'Prinia socialis',
        'Prinia subflava',
        'Aphelocoma californica',
        'Aphelocoma insularis',
        'Aphelocoma wollweberi',
        'Aphelocoma woodhouseii',
        'Corvus brachyrhynchos',
        'Corvus corax',
        'Corvus cornix',
        'Corvus corone',
        'Corvus coronoides',
        'Corvus cryptoleucus',
        'Corvus frugilegus',
        'Corvus macrorhynchos',
        'Corvus monedula',
        'Corvus sinaloae',
        'Cyanocitta cristata',
        'Cyanocitta stelleri',
        'Cyanocorax beecheii',
        'Cyanocorax chrysops',
        'Cyanocorax sanblasianus',
        'Cyanocorax yncas',
        'Cyanocorax yucatanicus',
        'Cyanopica cooki',
        'Cyanopica cyanus',
        'Dendrocitta formosae',
        'Dendrocitta vagabunda',
        'Garrulus glandarius',
        'Gymnorhinus cyanocephalus',
        'Nucifraga caryocatactes',
        'Nucifraga columbiana',
        'Perisoreus canadensis',
        'Pica hudsonia',
        'Pica nuttalli',
        'Pica pica',
        'Psilorhinus morio',
        'Pyrrhocorax pyrrhocorax',
        'Urocissa erythroryncha',
        'Cracticus nigrogularis',
        'Cracticus torquatus',
        'Gymnorhina tibicen',
        'Strepera graculina',
        'Dicaeum cruentatum',
        'Dicrurus leucophaeus',
        'Dicrurus macrocercus',
        'Dicrurus paradiseus',
        'Donacobius atricapilla',
        'Emberiza cia',
        'Emberiza cirlus',
        'Emberiza citrinella',
        'Emberiza elegans',
        'Emberiza melanocephala',
        'Emberiza rustica',
        'Euodice malabarica',
        'Lagonosticta senegala',
        'Lonchura atricapilla',
        'Lonchura punctulata',
        'Neochmia temporalis',
        'Spermestes cucullata',
        'Uraeginthus angolensis',
        'Uraeginthus bengalus',
        'Acanthis cabaret',
        'Acanthis flammea',
        'Acanthis hornemanni',
        'Carduelis carduelis',
        'Chloris chloris',
        'Chloris sinica',
        'Coccothraustes coccothraustes',
        'Coccothraustes vespertinus',
        'Crithagra mozambica',
        'Euphonia gouldi',
        'Euphonia laniirostris',
        'Fringilla coelebs',
        'Haemorhous cassinii',
        'Haemorhous mexicanus',
        'Haemorhous purpureus',
        'Himatione sanguinea',
        'Leucosticte tephrocotis',
        'Linaria cannabina',
        'Loxia curvirostra',
        'Pyrrhula pyrrhula',
        'Spinus lawrencei',
        'Spinus magellanicus',
        'Spinus notatus',
        'Spinus pinus',
        'Spinus psaltria',
        'Spinus tristis',
        'Furnarius leucopus',
        'Furnarius rufus',
        'Lepidocolaptes souleyetii',
        'Sittasomus griseicapillus',
        'Cecropis abyssinica',
        'Cecropis daurica',
        'Delichon urbicum',
        'Hirundo neoxena',
        'Hirundo rustica',
        'Hirundo smithii',
        'Hirundo tahitica',
        'Petrochelidon pyrrhonota',
        'Progne chalybea',
        'Progne subis',
        'Progne tapera',
        'Ptyonoprogne rupestris',
        'Pygochelidon cyanoleuca',
        'Riparia riparia',
        'Stelgidopteryx ruficollis',
        'Stelgidopteryx serripennis',
        'Tachycineta albilinea',
        'Tachycineta albiventer',
        'Tachycineta bicolor',
        'Tachycineta thalassina',
        'Agelaioides badius',
        'Agelaius phoeniceus',
        'Agelaius tricolor',
        'Cacicus cela',
        'Cassiculus melanicterus',
        'Dolichonyx oryzivorus',
        'Euphagus carolinus',
        'Euphagus cyanocephalus',
        'Icterus abeillei',
        'Icterus auratus',
        'Icterus bullockii',
        'Icterus chrysater',
        'Icterus cucullatus',
        'Icterus galbula',
        'Icterus gularis',
        'Icterus mesomelas',
        'Icterus nigrogularis',
        'Icterus parisorum',
        'Icterus pectoralis',
        'Icterus prosthemelas',
        'Icterus pustulatus',
        'Icterus spurius',
        'Molothrus aeneus',
        'Molothrus ater',
        'Molothrus bonariensis',
        'Psarocolius decumanus',
        'Psarocolius montezuma',
        'Psarocolius wagleri',
        'Quiscalus lugubris',
        'Quiscalus major',
        'Quiscalus mexicanus',
        'Quiscalus quiscula',
        'Sturnella loyca',
        'Sturnella magna',
        'Sturnella neglecta',
        'Xanthocephalus xanthocephalus',
        'Icteria virens',
        'Corvinella melanoleuca',
        'Lanius borealis',
        'Lanius bucephalus',
        'Lanius collaris',
        'Lanius collurio',
        'Lanius cristatus',
        'Lanius excubitor',
        'Lanius ludovicianus',
        'Lanius meridionalis',
        'Lanius minor',
        'Lanius schach',
        'Lanius senator',
        'Leiothrix lutea',
        'Turdoides striata',
        'Megalurus punctatus',
        'Malurus splendens',
        'Acanthorhynchus tenuirostris',
        'Anthochaera carunculata',
        'Anthochaera chrysoptera',
        'Anthornis melanura',
        'Entomyzon cyanotis',
        'Gavicalis virescens',
        'Lichmera indistincta',
        'Meliphaga lewinii',
        'Phylidonyris novaehollandiae',
        'Prosthemadera novaeseelandiae',
        'Dumetella carolinensis',
        'Margarops fuscatus',
        'Melanoptila glabrirostris',
        'Melanotis caerulescens',
        'Mimus gilvus',
        'Mimus longicaudatus',
        'Mimus parvulus',
        'Mimus polyglottos',
        'Mimus saturninus',
        'Toxostoma crissale',
        'Toxostoma curvirostre',
        'Toxostoma longirostre',
        'Toxostoma redivivum',
        'Toxostoma rufum',
        'Mohoua albicilla',
        'Hypothymis azurea',
        'Terpsiphone mutata',
        'Terpsiphone viridis',
        'Anthus cervinus',
        'Anthus novaeseelandiae',
        'Anthus petrosus',
        'Anthus rubescens',
        'Anthus rufulus',
        'Anthus spinoletta',
        'Anthus spragueii',
        'Anthus trivialis',
        'Macronyx croceus',
        'Motacilla alba',
        'Motacilla capensis',
        'Motacilla cinerea',
        'Motacilla maderaspatensis',
        'Cercomela familiaris',
        'Copsychus fulicatus',
        'Cossypha caffra',
        'Cossypha heuglini',
        'Erithacus rubecula',
        'Luscinia megarhynchos',
        'Monticola saxatilis',
        'Muscicapa dauurica',
        'Myophonus caeruleus',
        'Oenanthe hispanica',
        'Oenanthe isabellina',
        'Oenanthe oenanthe',
        'Phoenicurus auroreus',
        'Phoenicurus fuliginosus',
        'Phoenicurus ochruros',
        'Phoenicurus phoenicurus',
        'Saxicola caprata',
        'Saxicola maurus',
        'Saxicola rubetra',
        'Saxicola torquatus',
        'Aethopyga siparaja',
        'Anthreptes malacensis',
        'Chalcomitra senegalensis',
        'Cinnyris asiaticus',
        'Cinnyris osea',
        'Cinnyris venustus',
        'Hedydipna collaris',
        'Oriolus chinensis',
        'Oriolus oriolus',
        'Sphecotheres vieilloti',
        'Colluricincla harmonica',
        'Pachycephala pectoralis',
        'Pachycephala rufiventris',
        'Chamaea fasciata',
        'Sinosuthora webbiana',
        'Baeolophus atricristatus',
        'Baeolophus bicolor',
        'Baeolophus inornatus',
        'Baeolophus ridgwayi',
        'Baeolophus wollweberi',
        'Cyanistes caeruleus',
        'Lophophanes cristatus',
        'Parus major',
        'Parus minor',
        'Periparus ater',
        'Poecile atricapillus',
        'Poecile carolinensis',
        'Poecile gambeli',
        'Poecile hudsonicus',
        'Poecile montanus',
        'Poecile palustris',
        'Poecile rufescens',
        'Poecile sclateri',
        'Sittiparus varius',
        'Basileuterus culicivorus',
        'Basileuterus rufifrons',
        'Cardellina pusilla',
        'Cardellina rubrifrons',
        'Geothlypis poliocephala',
        'Geothlypis tolmiei',
        'Geothlypis trichas',
        'Limnothlypis swainsonii',
        'Mniotilta varia',
        'Oporornis agilis',
        'Oreothlypis celata',
        'Oreothlypis luciae',
        'Oreothlypis peregrina',
        'Oreothlypis ruficapilla',
        'Oreothlypis superciliosa',
        'Oreothlypis virginiae',
        'Parkesia motacilla',
        'Parkesia noveboracensis',
        'Protonotaria citrea',
        'Seiurus aurocapilla',
        'Setophaga americana',
        'Setophaga caerulescens',
        'Setophaga citrina',
        'Setophaga coronata',
        'Setophaga discolor',
        'Setophaga dominica',
        'Setophaga fusca',
        'Setophaga graciae',
        'Setophaga kirtlandii',
        'Setophaga magnolia',
        'Setophaga nigrescens',
        'Setophaga occidentalis',
        'Setophaga palmarum',
        'Setophaga pensylvanica',
        'Setophaga petechia',
        'Setophaga pinus',
        'Setophaga pitiayumi',
        'Setophaga ruticilla',
        'Setophaga striata',
        'Setophaga townsendi',
        'Setophaga virens',
        'Aimophila rufescens',
        'Aimophila ruficeps',
        'Ammodramus henslowii',
        'Ammodramus leconteii',
        'Ammodramus savannarum',
        'Amphispiza bilineata',
        'Arremonops rufivirgatus',
        'Artemisiospiza nevadensis',
        'Atlapetes pileatus',
        'Chlorospingus flavopectus',
        'Chondestes grammacus',
        'Junco hyemalis',
        'Junco phaeonotus',
        'Melospiza georgiana',
        'Melospiza lincolnii',
        'Melospiza melodia',
        'Melozone albicollis',
        'Melozone crissalis',
        'Melozone fusca',
        'Melozone kieneri',
        'Passerculus sandwichensis',
        'Passerella iliaca',
        'Peucaea aestivalis',
        'Peucaea botterii',
        'Peucaea carpalis',
        'Peucaea cassinii',
        'Pipilo chlorurus',
        'Pipilo erythrophthalmus',
        'Pipilo maculatus',
        'Pooecetes gramineus',
        'Spizella breweri',
        'Spizella pallida',
        'Spizella passerina',
        'Spizella pusilla',
        'Spizelloides arborea',
        'Zonotrichia albicollis',
        'Zonotrichia atricapilla',
        'Zonotrichia capensis',
        'Zonotrichia leucophrys',
        'Zonotrichia querula',
        'Passer diffusus',
        'Passer domesticus',
        'Passer griseus',
        'Passer hispaniolensis',
        'Passer melanurus',
        'Passer montanus',
        'Petronia petronia',
        'Eopsaltria australis',
        'Petroica australis',
        'Petroica boodang',
        'Petroica macrocephala',
        'Peucedramus taeniatus',
        'Phylloscopus collybita',
        'Phylloscopus trochilus',
        'Ceratopipra mentalis',
        'Manacus candei',
        'Euplectes franciscanus',
        'Euplectes orix',
        'Plocepasser mahali',
        'Ploceus baglafecht',
        'Ploceus cucullatus',
        'Ploceus philippinus',
        'Ploceus velatus',
        'Quelea quelea',
        'Polioptila albiloris',
        'Polioptila caerulea',
        'Polioptila californica',
        'Promerops cafer',
        'Prunella collaris',
        'Prunella modularis',
        'Phainopepla nitens',
        'Ptiliogonys cinereus',
        'Ptilonorhynchus violaceus',
        'Hypsipetes amaurotis',
        'Pycnonotus aurigaster',
        'Pycnonotus goiavier',
        'Pycnonotus leucotis',
        'Pycnonotus sinensis',
        'Pycnonotus xanthopygos',
        'Regulus calendula',
        'Regulus satrapa',
        'Auriparus flaviceps',
        'Remiz pendulinus',
        'Rhipidura albiscapa',
        'Rhipidura fuliginosa',
        'Rhipidura javanica',
        'Rhipidura leucophrys',
        'Sitta canadensis',
        'Sitta carolinensis',
        'Sitta europaea',
        'Sitta pygmaea',
        'Spindalis zena',
        'Acridotheres cristatellus',
        'Acridotheres fuscus',
        'Acridotheres javanicus',
        'Acridotheres tristis',
        'Aplonis panayensis',
        'Cinnyricinclus leucogaster',
        'Gracupica contra',
        'Gracupica nigricollis',
        'Lamprotornis chalybaeus',
        'Lamprotornis nitens',
        'Lamprotornis superbus',
        'Spodiopsar cineraceus',
        'Sturnus unicolor',
        'Sturnus vulgaris',
        'Sylvia atricapilla',
        'Sylvia communis',
        'Sylvia curruca',
        'Sylvia melanocephala',
        'Coereba flaveola',
        'Dacnis cayana',
        'Embernagra platensis',
        'Eucometis penicillata',
        'Geospiza fuliginosa',
        'Loxigilla noctis',
        'Paroaria coronata',
        'Paroaria gularis',
        'Ramphocelus dimidiatus',
        'Ramphocelus flammigerus',
        'Ramphocelus sanguinolentus',
        'Saltator aurantiirostris',
        'Saltator coerulescens',
        'Saltator maximus',
        'Sicalis flaveola',
        'Sporophila caerulescens',
        'Sporophila corvina',
        'Sporophila minuta',
        'Sporophila nigricollis',
        'Sporophila torqueola',
        'Tachyphonus rufus',
        'Tangara cyanicollis',
        'Tangara gyrola',
        'Tangara larvata',
        'Tangara vitriolina',
        'Thraupis abbas',
        'Thraupis episcopus',
        'Thraupis sayaca',
        'Tiaris bicolor',
        'Volatinia jacarina',
        'Pachyramphus aglaiae',
        'Tityra inquisitor',
        'Tityra semifasciata',
        'Campylorhynchus brunneicapillus',
        'Campylorhynchus griseus',
        'Campylorhynchus gularis',
        'Campylorhynchus zonatus',
        'Catherpes mexicanus',
        'Cistothorus palustris',
        'Salpinctes obsoletus',
        'Thryomanes bewickii',
        'Thryophilus pleurostictus',
        'Thryophilus sinaloa',
        'Thryothorus ludovicianus',
        'Troglodytes aedon',
        'Troglodytes pacificus',
        'Catharus aurantiirostris',
        'Catharus fuscescens',
        'Catharus guttatus',
        'Catharus ustulatus',
        'Hylocichla mustelina',
        'Ixoreus naevius',
        'Myadestes occidentalis',
        'Myadestes townsendi',
        'Sialia currucoides',
        'Sialia mexicana',
        'Sialia sialis',
        'Turdus amaurochalinus',
        'Turdus assimilis',
        'Turdus eunomus',
        'Turdus falcklandii',
        'Turdus fuscater',
        'Turdus grayi',
        'Turdus ignobilis',
        'Turdus iliacus',
        'Turdus leucomelas',
        'Turdus libonyana',
        'Turdus merula',
        'Turdus migratorius',
        'Turdus philomelos',
        'Turdus pilaris',
        'Turdus rufiventris',
        'Turdus rufopalliatus',
        'Turdus torquatus',
        'Arundinicola leucocephala',
        'Contopus cooperi',
        'Contopus sordidulus',
        'Contopus virens',
        'Elaenia flavogaster',
        'Empidonax alnorum',
        'Empidonax difficilis',
        'Empidonax flaviventris',
        'Empidonax oberholseri',
        'Empidonax virescens',
        'Empidonax wrightii',
        'Fluvicola nengeta',
        'Hymenops perspicillatus',
        'Machetornis rixosa',
        'Megarynchus pitangua',
        'Myiarchus cinerascens',
        'Myiarchus crinitus',
        'Myiarchus magnirostris',
        'Myiodynastes luteiventris',
        'Myiodynastes maculatus',
        'Myiozetetes cayanensis',
        'Myiozetetes similis',
        'Pitangus sulphuratus',
        'Pyrocephalus rubinus',
        'Sayornis nigricans',
        'Sayornis phoebe',
        'Sayornis saya',
        'Todirostrum cinereum',
        'Tolmomyias sulphurescens',
        'Tyrannus caudifasciatus',
        'Tyrannus forficatus',
        'Tyrannus melancholicus',
        'Tyrannus tyrannus',
        'Tyrannus verticalis',
        'Tyrannus vociferans',
        'Prionops plumatus',
        'Vidua macroura',
        'Cyclarhis gujanensis',
        'Vireo altiloquus',
        'Vireo atricapilla',
        'Vireo flavoviridis',
        'Vireo gilvus',
        'Vireo griseus',
        'Vireo huttoni',
        'Vireo olivaceus',
        'Vireo pallens',
        'Vireo solitarius',
        'Zosterops lateralis',
        'Zosterops palpebrosus',
        'Ardea alba',
        'Ardea cinerea',
        'Ardea goliath',
        'Ardea herodias',
        'Ardea intermedia',
        'Ardea melanocephala',
        'Ardeola bacchus',
        'Ardeola grayii',
        'Ardeola ralloides',
        'Botaurus lentiginosus',
        'Botaurus stellaris',
        'Bubulcus ibis',
        'Butorides striata',
        'Butorides virescens',
        'Egretta caerulea',
        'Egretta garzetta',
        'Egretta novaehollandiae',
        'Egretta rufescens',
        'Egretta thula',
        'Egretta tricolor',
        'Ixobrychus minutus',
        'Ixobrychus sinensis',
        'Nyctanassa violacea',
        'Nycticorax caledonicus',
        'Nycticorax nycticorax',
        'Pilherodius pileatus',
        'Syrigma sibilatrix',
        'Tigrisoma lineatum',
        'Tigrisoma mexicanum',
        'Pelecanus conspicillatus',
        'Pelecanus crispus',
        'Pelecanus erythrorhynchos',
        'Pelecanus occidentalis',
        'Pelecanus rufescens',
        'Pelecanus thagus',
        'Eudocimus albus',
        'Mesembrinibis cayennensis',
        'Phimosus infuscatus',
        'Platalea ajaja',
        'Plegadis chihi',
        'Plegadis falcinellus',
        'Theristicus caudatus',
        'Theristicus melanopis',
        'Threskiornis aethiopicus',
        'Threskiornis molucca',
        'Threskiornis spinicollis',
        'Phaethon lepturus',
        'Phaethon rubricauda',
        'Phoeniconaias minor',
        'Phoenicopterus chilensis',
        'Phoenicopterus roseus',
        'Phoenicopterus ruber',
        'Eubucco bourcierii',
        'Lybius torquatus',
        'Trachyphonus vaillantii',
        'Psilopogon haemacephalus',
        'Campephilus melanoleucos',
        'Colaptes auratus',
        'Colaptes campestris',
        'Colaptes chrysoides',
        'Colaptes melanochloros',
        'Colaptes punctigula',
        'Dendrocopos major',
        'Dendrocopos medius',
        'Dendrocopos minor',
        'Dendrocopos syriacus',
        'Dendropicos fuscescens',
        'Dinopium benghalense',
        'Dryocopus lineatus',
        'Dryocopus martius',
        'Dryocopus pileatus',
        'Jynx torquilla',
        'Melanerpes aurifrons',
        'Melanerpes carolinus',
        'Melanerpes chrysogenys',
        'Melanerpes erythrocephalus',
        'Melanerpes formicivorus',
        'Melanerpes hoffmannii',
        'Melanerpes hypopolius',
        'Melanerpes pygmaeus',
        'Melanerpes rubricapillus',
        'Melanerpes striatus',
        'Melanerpes uropygialis',
        'Picoides arizonae',
        'Picoides borealis',
        'Picoides nuttallii',
        'Picoides pubescens',
        'Picoides scalaris',
        'Picoides villosus',
        'Picus canus',
        'Sphyrapicus ruber',
        'Sphyrapicus varius',
        'Aulacorhynchus prasinus',
        'Ramphastos toco',
        'Aechmophorus clarkii',
        'Aechmophorus occidentalis',
        'Podiceps auritus',
        'Podiceps cristatus',
        'Podiceps grisegena',
        'Podiceps major',
        'Podiceps nigricollis',
        'Podilymbus podiceps',
        'Poliocephalus rufopectus',
        'Rollandia rolland',
        'Tachybaptus dominicus',
        'Tachybaptus novaehollandiae',
        'Tachybaptus ruficollis',
        'Diomedea epomophora',
        'Diomedea exulans',
        'Phoebastria nigripes',
        'Thalassarche cauta',
        'Thalassarche melanophris',
        'Oceanodroma furcata',
        'Ardenna bulleri',
        'Ardenna creatopus',
        'Ardenna grisea',
        'Ardenna pacifica',
        'Calonectris diomedea',
        'Daption capense',
        'Fulmarus glacialis',
        'Macronectes giganteus',
        'Macronectes halli',
        'Puffinus opisthomelas',
        'Cacatua sanguinea',
        'Cacatua tenuirostris',
        'Calyptorhynchus banksii',
        'Calyptorhynchus funereus',
        'Eolophus roseicapilla',
        'Amazona autumnalis',
        'Amazona finschi',
        'Amazona ochrocephala',
        'Ara ararauna',
        'Ara chloropterus',
        'Brotogeris chiriri',
        'Brotogeris jugularis',
        'Eupsittula canicularis',
        'Eupsittula pertinax',
        'Forpus conspicillatus',
        'Myiopsitta monachus',
        'Pionus menstruus',
        'Pionus senilis',
        'Psittacara leucophthalmus',
        'Psittacara mitratus',
        'Agapornis roseicollis',
        'Barnardius zonarius',
        'Psephotus haematonotus',
        'Psittacula eupatria',
        'Psittacula krameri',
        'Trichoglossus haematodus',
        'Nestor meridionalis',
        'Rhea pennata',
        'Aptenodytes patagonicus',
        'Pygoscelis adeliae',
        'Pygoscelis papua',
        'Spheniscus magellanicus',
        'Spheniscus mendiculus',
        'Asio otus',
        'Athene brama',
        'Athene cunicularia',
        'Bubo africanus',
        'Bubo bubo',
        'Bubo lacteus',
        'Bubo scandiacus',
        'Bubo virginianus',
        'Glaucidium brasilianum',
        'Glaucidium perlatum',
        'Megascops asio',
        'Megascops choliba',
        'Megascops trichopsis',
        'Micrathene whitneyi',
        'Ninox novaeseelandiae',
        'Strix aluco',
        'Strix occidentalis',
        'Strix varia',
        'Tyto alba',
        'Anhinga anhinga',
        'Fregata magnificens',
        'Microcarbo africanus',
        'Microcarbo melanoleucos',
        'Microcarbo niger',
        'Microcarbo pygmeus',
        'Phalacrocorax aristotelis',
        'Phalacrocorax auritus',
        'Phalacrocorax brasilianus',
        'Phalacrocorax capensis',
        'Phalacrocorax carbo',
        'Phalacrocorax pelagicus',
        'Phalacrocorax penicillatus',
        'Phalacrocorax punctatus',
        'Phalacrocorax varius',
        'Morus bassanus',
        'Sula leucogaster',
        'Sula nebouxii',
        'Sula sula',
        'Sula variegata',
        'Trogon elegans',
        'Trogon melanocephalus',
        'Trogon mexicanus',
        'Trogon personatus',
        'Trogon rufus',
        'Carcharhinus amblyrhynchos',
        'Carcharhinus melanopterus',
        'Triaenodon obesus',
        'Heterodontus francisci',
        'Heterodontus portusjacksoni',
        'Carcharodon carcharias',
        'Carcharias taurus',
        'Dasyatis brevicaudata',
        'Dasyatis sabina',
        'Himantura fai',
        'Pastinachus ater',
        'Taeniura lymma',
        'Taeniurops meyeni',
        'Aetobatus ocellatus',
        'Manta alfredi',
        'Myliobatis tenuicaudatus',
        'Trygonoptera imitata',
        'Trygonoptera testacea',
        'Urolophus halleri',
        'Urobatis jamaicensis',
        'Orectolobus halei',
        'Orectolobus maculatus',
        'Stegostoma fasciatum',
        'Rhynchobatus australiae',
        'Trygonorrhina dumerilii',
        'Trygonorrhina fasciata',
        'Antilocapra americana',
        'Aepyceros melampus',
        'Alcelaphus buselaphus',
        'Antidorcas marsupialis',
        'Bison bison',
        'Boselaphus tragocamelus',
        'Capra ibex',
        'Cephalophus rufilatus',
        'Connochaetes taurinus',
        'Damaliscus lunatus',
        'Damaliscus pygargus',
        'Eudorcas thomsonii',
        'Gazella gazella',
        'Hippotragus equinus',
        'Hippotragus niger',
        'Kobus kob',
        'Madoqua kirkii',
        'Nanger granti',
        'Oreamnos americanus',
        'Oreotragus oreotragus',
        'Oryx gazella',
        'Ourebia ourebi',
        'Ovibos moschatus',
        'Ovis canadensis',
        'Ovis dalli',
        'Raphicerus campestris',
        'Redunca redunca',
        'Sylvicapra grimmia',
        'Syncerus caffer',
        'Tragelaphus angasii',
        'Tragelaphus oryx',
        'Lama guanicoe',
        'Alces alces',
        'Alces americanus',
        'Capreolus capreolus',
        'Cervus canadensis',
        'Cervus elaphus',
        'Hydropotes inermis',
        'Odocoileus hemionus',
        'Odocoileus virginianus',
        'Rusa unicolor',
        'Giraffa camelopardalis',
        'Hippopotamus amphibius',
        'Phacochoerus africanus',
        'Sus scrofa',
        'Pecari tajacu',
        'Canis adustus',
        'Canis aureus',
        'Canis latrans',
        'Cerdocyon thous',
        'Lycalopex griseus',
        'Otocyon megalotis',
        'Urocyon cinereoargenteus',
        'Urocyon littoralis',
        'Vulpes lagopus',
        'Vulpes macrotis',
        'Vulpes vulpes',
        'Caracal caracal',
        'Felis catus',
        'Felis silvestris',
        'Leopardus wiedii',
        'Leptailurus serval',
        'Lynx rufus',
        'Panthera leo',
        'Panthera tigris',
        'Puma concolor',
        'Cynictis penicillata',
        'Helogale parvula',
        'Herpestes auropunctatus',
        'Herpestes ichneumon',
        'Herpestes sanguineus',
        'Ichneumia albicauda',
        'Mungos mungo',
        'Crocuta crocuta',
        'Mephitis macroura',
        'Mephitis mephitis',
        'Spilogale gracilis',
        'Eira barbara',
        'Enhydra lutris',
        'Lontra canadensis',
        'Lontra longicaudis',
        'Lutra lutra',
        'Martes americana',
        'Martes foina',
        'Martes martes',
        'Martes pennanti',
        'Meles meles',
        'Mellivora capensis',
        'Mustela nivalis',
        'Mustela putorius',
        'Neovison vison',
        'Pteronura brasiliensis',
        'Arctocephalus forsteri',
        'Arctocephalus pusillus',
        'Otaria byronia',
        'Phocarctos hookeri',
        'Zalophus californianus',
        'Zalophus wollebaeki',
        'Hydrurga leptonyx',
        'Mirounga angustirostris',
        'Phoca vitulina',
        'Bassariscus astutus',
        'Nasua narica',
        'Nasua nasua',
        'Procyon lotor',
        'Ursus americanus',
        'Ursus arctos',
        'Ursus maritimus',
        'Civettictis civetta',
        'Genetta maculata',
        'Balaenoptera acutorostrata',
        'Balaenoptera musculus',
        'Balaenoptera physalus',
        'Megaptera novaeangliae',
        'Delphinus capensis',
        'Grampus griseus',
        'Lagenorhynchus obliquidens',
        'Orcinus orca',
        'Stenella longirostris',
        'Tursiops truncatus',
        'Eschrichtius robustus',
        'Phocoenoides dalli',
        'Physeter macrocephalus',
        'Taphozous mauritianus',
        'Artibeus jamaicensis',
        'Artibeus lituratus',
        'Eidolon helvum',
        'Epomophorus gambianus',
        'Pteropus poliocephalus',
        'Corynorhinus townsendii',
        'Lasiurus blossevillii',
        'Myotis lucifugus',
        'Perimyotis subflavus',
        'Dasypus novemcinctus',
        'Galeopterus variegatus',
        'Didelphis marsupialis',
        'Didelphis virginiana',
        'Philander opossum',
        'Macropus agilis',
        'Macropus fuliginosus',
        'Macropus rufogriseus',
        'Thylogale billardierii',
        'Wallabia bicolor',
        'Trichosurus vulpecula',
        'Pseudocheirus peregrinus',
        'Vombatus ursinus',
        'Erinaceus europaeus',
        'Notiosorex crawfordi',
        'Neurotrichus gibbsii',
        'Parascalops breweri',
        'Talpa europaea',
        'Lepus alleni',
        'Lepus americanus',
        'Lepus californicus',
        'Lepus europaeus',
        'Lepus townsendii',
        'Oryctolagus cuniculus',
        'Sylvilagus aquaticus',
        'Sylvilagus audubonii',
        'Sylvilagus bachmani',
        'Sylvilagus cunicularius',
        'Sylvilagus floridanus',
        'Sylvilagus nuttallii',
        'Ochotona princeps',
        'Equus grevyi',
        'Equus quagga',
        'Ceratotherium simum',
        'Diceros bicornis',
        'Tapirus bairdii',
        'Alouatta caraya',
        'Callithrix jacchus',
        'Saimiri sciureus',
        'Cercopithecus mitis',
        'Chlorocebus pygerythrus',
        'Chlorocebus tantalus',
        'Colobus guereza',
        'Erythrocebus patas',
        'Papio cynocephalus',
        'Trachypithecus obscurus',
        'Pan troglodytes',
        'Loxodonta africana',
        'Castor canadensis',
        'Castor fiber',
        'Myodes glareolus',
        'Ondatra zibethicus',
        'Peromyscus fraterculus',
        'Cuniculus paca',
        'Dasyprocta mexicana',
        'Coendou mexicanus',
        'Erethizon dorsatum',
        'Thomomys bottae',
        'Glis glis',
        'Dipodomys merriami',
        'Hystrix cristata',
        'Mus musculus',
        'Myocastor coypus',
        'Callosciurus erythraeus',
        'Callosciurus notatus',
        'Callospermophilus lateralis',
        'Callospermophilus saturatus',
        'Cynomys ludovicianus',
        'Cynomys mexicanus',
        'Glaucomys sabrinus',
        'Marmota flaviventris',
        'Marmota monax',
        'Otospermophilus beecheyi',
        'Otospermophilus variegatus',
        'Paraxerus cepapi',
        'Sciurus alleni',
        'Sciurus arizonensis',
        'Sciurus aureogaster',
        'Sciurus carolinensis',
        'Sciurus colliaei',
        'Sciurus granatensis',
        'Sciurus griseus',
        'Sciurus niger',
        'Sciurus vulgaris',
        'Sciurus yucatanensis',
        'Tamias amoenus',
        'Tamias merriami',
        'Tamias sibiricus',
        'Tamias sonomae',
        'Tamias speciosus',
        'Tamias striatus',
        'Tamias townsendii',
        'Tamias umbrinus',
        'Tamiasciurus douglasii',
        'Tamiasciurus hudsonicus',
        'Urocitellus parryii',
        'Urocitellus richardsonii',
        'Xerospermophilus tereticaudus',
        'Xerus inauris',
        'Trichechus manatus',
        'Orycteropus afer',
        'Alligator mississippiensis',
        'Caiman yacare',
        'Crocodylus acutus',
        'Crocodylus palustris',
        'Crocodylus porosus',
        'Sphenodon punctatus',
        'Acanthocercus atricollis',
        'Agama agama',
        'Agama atra',
        'Agama lionotus',
        'Bronchocela cristatella',
        'Bronchocela jubata',
        'Japalura swinhonis',
        'Stellagama stellio',
        'Anguis fragilis',
        'Anguis veronensis',
        'Elgaria multicarinata',
        'Ophisaurus attenuatus',
        'Ophisaurus ventralis',
        'Anniella stebbinsi',
        'Boa imperator',
        'Lichanura trivirgata',
        'Calumma parsonii',
        'Chamaeleo chamaeleon',
        'Furcifer oustaleti',
        'Trioceros jacksonii',
        'Ahaetulla prasina',
        'Arizona elegans',
        'Cemophora coccinea',
        'Chionactis occipitalis',
        'Coluber bilineatus',
        'Coluber constrictor',
        'Coluber flagellum',
        'Coluber fuliginosus',
        'Coluber lateralis',
        'Coluber mentovarius',
        'Coluber schotti',
        'Coniophanes imperialis',
        'Conopsis lineata',
        'Conopsis nasus',
        'Crotaphopeltis hotamboeia',
        'Dendrelaphis pictus',
        'Diadophis punctatus',
        'Drymarchon melanurus',
        'Farancia abacura',
        'Haldea striatula',
        'Heterodon kennerlyi',
        'Heterodon platirhinos',
        'Hierophis viridiflavus',
        'Hypsiglena chlorophaea',
        'Hypsiglena jani',
        'Hypsiglena ochrorhyncha',
        'Imantodes cenchoa',
        'Lampropeltis alterna',
        'Lampropeltis annulata',
        'Lampropeltis californiae',
        'Lampropeltis calligaster',
        'Lampropeltis gentilis',
        'Lampropeltis holbrooki',
        'Lampropeltis nigra',
        'Lampropeltis polyzona',
        'Lampropeltis splendida',
        'Lampropeltis triangulum',
        'Lampropeltis zonata',
        'Leptodeira septentrionalis',
        'Leptophis ahaetulla',
        'Leptophis diplotropis',
        'Leptophis mexicanus',
        'Liodytes rigida',
        'Natrix maura',
        'Natrix natrix',
        'Nerodia clarkii',
        'Nerodia erythrogaster',
        'Nerodia fasciata',
        'Nerodia rhombifer',
        'Nerodia sipedon',
        'Ninia sebae',
        'Opheodrys aestivus',
        'Pantherophis alleghaniensis',
        'Pantherophis emoryi',
        'Pantherophis obsoletus',
        'Pantherophis vulpinus',
        'Phyllorhynchus decurtatus',
        'Pituophis catenifer',
        'Rhabdophis tigrinus',
        'Rhinocheilus lecontei',
        'Salvadora grahamiae',
        'Senticolis triaspis',
        'Spilotes pullatus',
        'Storeria dekayi',
        'Storeria occipitomaculata',
        'Storeria storerioides',
        'Tantilla gracilis',
        'Thamnophis cyrtopsis',
        'Thamnophis elegans',
        'Thamnophis eques',
        'Thamnophis marcianus',
        'Thamnophis proximus',
        'Thamnophis sirtalis',
        'Trimorphodon lyrophanes',
        'Trimorphodon paucimaculatus',
        'Trimorphodon tau',
        'Tropidoclonion lineatum',
        'Virginia valeriae',
        'Basiliscus basiliscus',
        'Basiliscus plumifrons',
        'Basiliscus vittatus',
        'Corytophanes cristatus',
        'Crotaphytus collaris',
        'Anolis carolinensis',
        'Anolis cristatellus',
        'Anolis cybotes',
        'Anolis distichus',
        'Anolis roquet',
        'Anolis sagrei',
        'Mokopirirakau granulatus',
        'Woodworthia maculatus',
        'Micrurus tener',
        'Coleonyx brevis',
        'Cyrtodactylus marmoratus',
        'Gehyra mutilata',
        'Gekko chinensis',
        'Gekko japonicus',
        'Hemidactylus frenatus',
        'Hemidactylus platyurus',
        'Hemidactylus turcicus',
        'Lepidodactylus lugubris',
        'Lygodactylus capensis',
        'Heloderma horridum',
        'Conolophus subcristatus',
        'Ctenosaura acanthura',
        'Ctenosaura hemilopha',
        'Ctenosaura macrolopha',
        'Ctenosaura pectinata',
        'Ctenosaura similis',
        'Dipsosaurus dorsalis',
        'Iguana iguana',
        'Sauromalus ater',
        'Lacerta agilis',
        'Lacerta viridis',
        'Podarcis muralis',
        'Podarcis siculus',
        'Podarcis tiliguerta',
        'Psammodromus algirus',
        'Teira dugesii',
        'Timon lepidus',
        'Zootoca vivipara',
        'Pseudaspis cana',
        'Rena dulcis',
        'Rena humilis',
        'Callisaurus draconoides',
        'Cophosaurus texanus',
        'Holbrookia elegans',
        'Holbrookia lacerata',
        'Petrosaurus mearnsi',
        'Phrynosoma blainvillii',
        'Phrynosoma cornutum',
        'Phrynosoma hernandesi',
        'Phrynosoma mcallii',
        'Phrynosoma modestum',
        'Phrynosoma solare',
        'Phrynosoma taurus',
        'Sceloporus chrysostictus',
        'Sceloporus consobrinus',
        'Sceloporus cowlesi',
        'Sceloporus formosus',
        'Sceloporus graciosus',
        'Sceloporus horridus',
        'Sceloporus malachiticus',
        'Sceloporus minor',
        'Sceloporus mucronatus',
        'Sceloporus nelsoni',
        'Sceloporus occidentalis',
        'Sceloporus olivaceus',
        'Sceloporus orcutti',
        'Sceloporus poinsettii',
        'Sceloporus siniferus',
        'Sceloporus torquatus',
        'Sceloporus tristichus',
        'Sceloporus undulatus',
        'Sceloporus uniformis',
        'Sceloporus variabilis',
        'Sceloporus zosteromus',
        'Urosaurus bicarinatus',
        'Urosaurus nigricaudus',
        'Urosaurus ornatus',
        'Uta stansburiana',
        'Phyllodactylus unctus',
        'Tarentola mauritanica',
        'Thecadactylus rapicauda',
        'Morelia spilota',
        'Eulamprus quoyii',
        'Eutropis multifasciata',
        'Oligosoma polychroma',
        'Plestiodon fasciatus',
        'Plestiodon gilberti',
        'Plestiodon laticeps',
        'Plestiodon obsoletus',
        'Plestiodon skiltonianus',
        'Scincella lateralis',
        'Sphenomorphus indicus',
        'Tiliqua rugosa',
        'Tiliqua scincoides',
        'Trachylepis margaritifera',
        'Trachylepis quinquetaeniata',
        'Trachylepis striata',
        'Gonatodes albogularis',
        'Ameiva ameiva',
        'Aspidoscelis costata',
        'Aspidoscelis exsanguis',
        'Aspidoscelis gularis',
        'Aspidoscelis hyperythra',
        'Aspidoscelis inornata',
        'Aspidoscelis lineattissima',
        'Aspidoscelis scalaris',
        'Aspidoscelis sexlineata',
        'Aspidoscelis sonorae',
        'Aspidoscelis tigris',
        'Aspidoscelis uniparens',
        'Cnemidophorus lemniscatus',
        'Holcosus festivus',
        'Holcosus undulatus',
        'Pholidoscelis exsul',
        'Salvator merianae',
        'Microlophus albemarlensis',
        'Varanus nebulosus',
        'Agkistrodon contortrix',
        'Agkistrodon piscivorus',
        'Bitis arietans',
        'Crotalus adamanteus',
        'Crotalus aquilus',
        'Crotalus atrox',
        'Crotalus basiliscus',
        'Crotalus cerastes',
        'Crotalus horridus',
        'Crotalus lepidus',
        'Crotalus molossus',
        'Crotalus oreganus',
        'Crotalus ornatus',
        'Crotalus pricei',
        'Crotalus pyrrhus',
        'Crotalus ravus',
        'Crotalus ruber',
        'Crotalus scutulatus',
        'Crotalus tigris',
        'Crotalus viridis',
        'Sistrurus catenatus',
        'Sistrurus tergeminus',
        'Trimeresurus albolabris',
        'Vipera aspis',
        'Xantusia henshawi',
        'Chelonia mydas',
        'Eretmochelys imbricata',
        'Chelydra serpentina',
        'Macrochelys temminckii',
        'Actinemys marmorata',
        'Chrysemys picta',
        'Deirochelys reticularia',
        'Emydoidea blandingii',
        'Graptemys pseudogeographica',
        'Graptemys versa',
        'Pseudemys rubriventris',
        'Pseudemys texana',
        'Terrapene carolina',
        'Terrapene ornata',
        'Trachemys scripta',
        'Trachemys venusta',
        'Mauremys caspica',
        'Rhinoclemmys areolata',
        'Rhinoclemmys pulcherrima',
        'Kinosternon baurii',
        'Kinosternon integrum',
        'Kinosternon sonoriense',
        'Sternotherus carinatus',
        'Sternotherus odoratus',
        'Chersina angulata',
        'Gopherus agassizii',
        'Gopherus berlandieri',
        'Gopherus morafkai',
        'Gopherus polyphemus',
        'Testudo graeca',
        'Testudo hermanni',
        'Apalone ferox',
        'Apalone spinifera',
        'Actinia equina',
        'Actinia tenebrosa',
        'Anthopleura artemisia',
        'Anthopleura elegantissima',
        'Anthopleura sola',
        'Anthopleura xanthogrammica',
        'Condylactis gigantea',
        'Epiactis prolifera',
        'Isactinia olivacea',
        'Oulactis magna',
        'Oulactis muscosa',
        'Phlyctenactis tuberculosa',
        'Urticina coriacea',
        'Urticina crassicornis',
        'Urticina grebelnyi',
        'Urticina lofotensis',
        'Diadumene lineata',
        'Metridium farcimen',
        'Anthothoe albocincta',
        'Corynactis californica',
        'Balanophyllia elegans',
        'Tubastraea coccinea',
        'Pseudodiploria strigosa',
        'Velella velella',
        'Eucheilota bakeri',
        'Physalia physalis',
        'Chrysaora fuscescens',
        'Phacellophora camtschatica',
        'Asterias rubens',
        'Coscinasterias muricata',
        'Evasterias troschelii',
        'Leptasterias hexactis',
        'Pisaster giganteus',
        'Pisaster ochraceus',
        'Pycnopodia helianthoides',
        'Stichaster australis',
        'Acanthaster planci',
        'Henricia leviuscula',
        'Henricia pumila',
        'Patiria chilensis',
        'Patiria miniata',
        'Patiriella regularis',
        'Dermasterias imbricata',
        'Linckia multifora',
        'Culcita novaeguineae',
        'Oreaster reticulatus',
        'Protoreaster nodosus',
        'Colobocentrotus atratus',
        'Evechinus chloroticus',
        'Heterocentrotus mamillatus',
        'Strongylocentrotus franciscanus',
        'Strongylocentrotus purpuratus',
        'Tripneustes gratilla',
        'Tripneustes ventricosus',
        'Dendraster excentricus',
        'Diadema antillarum',
        'Diadema setosum',
        'Holothuria atra',
        'Lissothuria nutriens',
        'Eupentacta quinquesemita',
        'Ophionereis fasciata',
        'Ophiothrix spiculata',
        'Eontia ponderosa',
        'Nuttallia obscurata',
        'Macoma nasuta',
        'Mya arenaria',
        'Platyodon cancellatus',
        'Cyrtopleura costata',
        'Mytilus californianus',
        'Mytilus galloprovincialis',
        'Perna canaliculus',
        'Xenostrobus neozelanicus',
        'Anomia simplex',
        'Pododesmus macrochisma',
        'Magallana gigas',
        'Argopecten gibbus',
        'Argopecten irradians',
        'Argopecten ventricosus',
        'Crassadoma gigantea',
        'Pecten novaezelandiae',
        'Amblema plicata',
        'Elliptio complanata',
        'Lampsilis cardium',
        'Lampsilis teres',
        'Leptodea fragilis',
        'Pyganodon grandis',
        'Dinocardium robustum',
        'Tridacna crocea',
        'Chama arcana',
        'Corbicula fluminea',
        'Donax gouldii',
        'Donax variabilis',
        'Spisula solidissima',
        'Ensis directus',
        'Siliqua patula',
        'Tagelus californianus',
        'Austrovenus stutchburyi',
        'Chione californiensis',
        'Chione elevata',
        'Dosinia anus',
        'Ruditapes philippinarum',
        'Saxidomus nuttalli',
        'Tivela stultorum',
        'Octopus bimaculoides',
        'Octopus cyanea',
        'Octopus rubescens',
        'Octopus tetricus',
        'Octopus vulgaris',
        'Sepia apama',
        'Sepia officinalis',
        'Spirula spirula',
        'Aplysia californica',
        'Bursatella leachii',
        'Bulla gouldiana',
        'Haminoea japonica',
        'Nerita melanotragus',
        'Nerita picea',
        'Lymnaea stagnalis',
        'Siphonaria australis',
        'Crepidula onyx',
        'Crepipatella lingulata',
        'Crucibulum spinosum',
        'Cypraea tigris',
        'Hipponix antiquatus',
        'Hipponix tumens',
        'Austrolittorina antipodum',
        'Littoraria irrorata',
        'Littorina keenae',
        'Littorina obtusata',
        'Littorina scutulata',
        'Euspira heros',
        'Neverita reclusiana',
        'Cyphoma gibbosum',
        'Pomatias elegans',
        'Strombus alatus',
        'Hespererato vitellina',
        'Acmaea mitra',
        'Calliostoma canaliculatum',
        'Epitonium tinctum',
        'Janthina janthina',
        'Opalia wroblewskyi',
        'Diodora aspera',
        'Fissurellidea bimaculata',
        'Scutus breviculus',
        'Haliotis cracherodii',
        'Haliotis fulgens',
        'Haliotis iris',
        'Haliotis rufescens',
        'Lottia gigantea',
        'Lottia insessa',
        'Lottia limatula',
        'Lottia scabra',
        'Lottia scutum',
        'Testudinalia testudinalis',
        'Cellana denticulata',
        'Cellana ornata',
        'Cellana radians',
        'Melanoides tuberculata',
        'Diloma aethiops',
        'Norrisia norrisii',
        'Tegula aureotincta',
        'Tegula eiseni',
        'Tegula funebralis',
        'Cookia sulcata',
        'Lunella smaragdus',
        'Maoricolpus roseus',
        'Busycon carica',
        'Busycotypus canaliculatus',
        'Cominella adspersa',
        'Cominella maculosa',
        'Cominella virgata',
        'Lirabuccinum dirum',
        'Sinistrofulgur sinistrum',
        'Sinistrofulgur pulleyi',
        'Alia carinata',
        'Amphissa columbiana',
        'Amphissa versicolor',
        'Californiconus californicus',
        'Triplofusus giganteus',
        'Mitra idae',
        'Acanthinucella punctulata',
        'Acanthinucella spirata',
        'Ceratostoma foliatum',
        'Dicathais orbita',
        'Haustrum scobina',
        'Nucella lapillus',
        'Pteropurpura festiva',
        'Roperia poulsoni',
        'Nassarius fossatus',
        'Tritia obsoleta',
        'Americoliva sayana',
        'Hallaxa chani',
        'Aeolidia loui',
        'Anteaeolidiella oliviae',
        'Aldisa sanguinea',
        'Cadlina flavomaculata',
        'Cadlina luteomarginata',
        'Cadlina modesta',
        'Ceratosoma amoenum',
        'Doriprismatica atromarginata',
        'Felimare picta',
        'Felimare tricolor',
        'Felimare villafranca',
        'Glossodoris rufomarginata',
        'Hypselodoris imperialis',
        'Thorunna daniellae',
        'Corambe pacifica',
        'Cuthona divae',
        'Doriopsilla albopunctata',
        'Doriopsilla fulva',
        'Dendronotus albus',
        'Dendronotus subramosus',
        'Dendronotus venustus',
        'Dirona albolineata',
        'Dirona picta',
        'Diaulula odonoghuei',
        'Diaulula sandiegensis',
        'Geitodoris heathi',
        'Peltodoris nobilis',
        'Doris montereyensis',
        'Doto amyra',
        'Doto columbiana',
        'Doto kya',
        'Eubranchus rustyus',
        'Hermissenda crassicornis',
        'Hermissenda opalescens',
        'Phidiana hiltoni',
        'Abronica abronia',
        'Flabellina cooperi',
        'Flabellina exoptata',
        'Flabellina rubrolineata',
        'Flabellina trilineata',
        'Pteraeolidia ianthina',
        'Pteraeolidia semperi',
        'Ancula gibbosa',
        'Goniodoris castanea',
        'Okenia rosacea',
        'Janolus barbarensis',
        'Jorunna funebris',
        'Aegires albopunctatus',
        'Acanthodoris hudsoni',
        'Acanthodoris lutea',
        'Acanthodoris nanaimoensis',
        'Acanthodoris rhodoceras',
        'Diaphorodoris lirulatocauda',
        'Onchidoris bilamellata',
        'Phyllidia ocellata',
        'Phyllidia varicosa',
        'Phyllidiella pustulosa',
        'Crimora coneja',
        'Polycera atra',
        'Polycera hedgpethi',
        'Polycera quadrilineata',
        'Tambja morosa',
        'Triopha catalinae',
        'Triopha maculata',
        'Limacia cockerelli',
        'Limacia mcdonaldi',
        'Rostanga pulchra',
        'Melibe leonina',
        'Tenellia albocrusta',
        'Tenellia flavovulta',
        'Tenellia lagunae',
        'Zelentia fulgens',
        'Tritonia festiva',
        'Berthella californica',
        'Elysia crispata',
        'Elysia hedgpethi',
        'Deroceras reticulatum',
        'Ariolimax columbianus',
        'Arion hortensis',
        'Prophysaon andersoni',
        'Megaustenia imperator',
        'Athoracophorus bitentaculatus',
        'Monadenia infumata',
        'Rabdotus mooreanus',
        'Zachrysia provisoria',
        'Cochlicopa lubrica',
        'Zonitoides arboreus',
        'Haplotrema minimum',
        'Haplotrema vancouverense',
        'Arianta arbustorum',
        'Cantareus apertus',
        'Cepaea nemoralis',
        'Cornu aspersum',
        'Eobania vermiculata',
        'Helix pomatia',
        'Otala lactea',
        'Helminthoglypta arrosa',
        'Helminthoglypta nickliniana',
        'Helminthoglypta sequoicola',
        'Helminthoglypta traskii',
        'Helminthoglypta tudiculata',
        'Cochlicella barbara',
        'Limax maximus',
        'Glyptostoma newberryanum',
        'Milax gagates',
        'Liguus fasciatus',
        'Meghimatium bilineatum',
        'Philomycus carolinianus',
        'Linisa texasiana',
        'Mesodon thyroidus',
        'Neohelix albolabris',
        'Patera roemeri',
        'Polygyra cereolus',
        'Vespericola columbianus',
        'Paryphanta busbyi',
        'Rumina decollata',
        'Acanthochitona zelandica',
        'Acanthopleura granulata',
        'Chiton glaucus',
        'Sypharochiton pelliserpentis',
        'Sypharochiton sinclairi',
        'Ischnochiton maorianus',
        'Lepidozona cooperi',
        'Lepidozona mertensii',
        'Stenoplax conspicua',
        'Cyanoplax hartwegii',
        'Cryptochiton stelleri',
        'Katharina tunicata',
        'Mopalia hindsii',
        'Mopalia muscosa',
        'Tonicella lineata',
        'Tonicella lokii',
        'Pseudobiceros ferrugineus',
        'Phascolosoma agassizii',
        'Nostoc commune',
        'Desmarestia ligulata',
        'Leathesia marina',
        'Ascophyllum nodosum',
        'Fucus distichus',
        'Pelvetiopsis limitata',
        'Silvetia compressa',
        'Hormosira banksii',
        'Carpophyllum maschalocarpum',
        'Carpophyllum plumosum',
        'Cystophora retroflexa',
        'Stephanocystis osmundacea',
        'Alaria marginata',
        'Ecklonia radiata',
        'Egregia menziesii',
        'Eisenia arborea',
        'Pterygophora californica',
        'Undaria pinnatifida',
        'Costaria costata',
        'Laminaria setchellii',
        'Laminaria sinclairii',
        'Saccharina latissima',
        'Macrocystis pyrifera',
        'Nereocystis luetkeana',
        'Postelsia palmaeformis',
        'Colpomenia sinuosa',
        'Herpothallon rubrocinctum',
        'Chrysothrix candelaris',
        'Candelaria concolor',
        'Cladia retipora',
        'Cladonia chlorophaea',
        'Cladonia confusa',
        'Cladonia coniocraea',
        'Cladonia furcata',
        'Cladonia macilenta',
        'Cladonia squamosa',
        'Cladonia stellaris',
        'Pilophorus acicularis',
        'Protoparmeliopsis pinguis',
        'Rhizoplaca chrysoleuca',
        'Dolichousnea longissima',
        'Evernia mesomorpha',
        'Evernia prunastri',
        'Flavoparmelia caperata',
        'Flavopunctelia flaventior',
        'Hypogymnia imshaugii',
        'Hypogymnia physodes',
        'Hypogymnia tubulosa',
        'Melanelixia subaurifera',
        'Parmelia squarrosa',
        'Parmotrema hypotropum',
        'Platismatia glauca',
        'Platismatia herrei',
        'Platismatia tuckermanii',
        'Pseudevernia consocians',
        'Punctelia rudecta',
        'Tuckermannopsis orbata',
        'Usnea intermedia',
        'Usnea rubicunda',
        'Usnea strigosa',
        'Vulpicida pinastri',
        'Heterodermia leucomela',
        'Phaeophyscia rubropulchra',
        'Physcia adscendens',
        'Physcia millegrana',
        'Physcia stellaris',
        'Niebla cephalota',
        'Niebla combeoides',
        'Niebla homalea',
        'Ramalina canariensis',
        'Ramalina celastri',
        'Ramalina leptocarpha',
        'Ramalina menziesii',
        'Ramalina pollinaria',
        'Ramalina subleptocarpha',
        'Stereocaulon ramulosum',
        'Porpidia albocaerulescens',
        'Graphis scripta',
        'Scytinium palmatum',
        'Lobaria anomala',
        'Lobaria anthraspis',
        'Lobaria pulmonaria',
        'Lobaria quercizans',
        'Lobaria scrobiculata',
        'Pseudocyphellaria crocata',
        'Sticta fuliginosa',
        'Sticta limbata',
        'Peltigera canina',
        'Peltigera collina',
        'Peltigera membranacea',
        'Dibaeis arcuata',
        'Dibaeis baeomyces',
        'Icmadophila ericetorum',
        'Polycauliona candelaria',
        'Rusavskia elegans',
        'Teloschistes chrysophthalmus',
        'Teloschistes exilis',
        'Teloschistes flavicans',
        'Xanthomendoza fallax',
        'Xanthomendoza hasseana',
        'Xanthoria parietina',
        'Lasallia papulosa',
        'Umbilicaria phaea',
        'Ascocoryne sarcoides',
        'Chlorociboria aeruginascens',
        'Bulgaria inquinans',
        'Leotia lubrica',
        'Rhytisma acerinum',
        'Caloscypha fulgens',
        'Gyromitra esculenta',
        'Gyromitra infula',
        'Helvella acetabulum',
        'Helvella crispa',
        'Helvella vespertina',
        'Morchella americana',
        'Morchella rufobrunnea',
        'Galiella rufa',
        'Urnula craterium',
        'Sirococcus clavigignenti-juglandacearum',
        'Hypomyces chrysospermus',
        'Hypomyces hyalinus',
        'Annulohypoxylon thouarsianum',
        'Kretzschmaria deusta',
        'Agaricus augustus',
        'Agaricus bernardii',
        'Agaricus californicus',
        'Agaricus campestris',
        'Agaricus xanthodermus',
        'Battarrea phalloides',
        'Calvatia sculpta',
        'Chlorophyllum brunneum',
        'Chlorophyllum rhacodes',
        'Coprinus comatus',
        'Lepiota magnispora',
        'Lepiota rubrotinctoides',
        'Leucoagaricus americanus',
        'Leucoagaricus leucothites',
        'Leucocoprinus cepistipes',
        'Montagnea arenaria',
        'Podaxis pistillaris',
        'Amanita amerirubescens',
        'Amanita augusta',
        'Amanita bisporigera',
        'Amanita caesarea',
        'Amanita calyptroderma',
        'Amanita gemmata',
        'Amanita jacksonii',
        'Amanita muscaria',
        'Amanita novinupta',
        'Amanita ocreata',
        'Amanita rubescens',
        'Amanita vaginata',
        'Conocybe apala',
        'Panaeolina foenisecii',
        'Clavaria zollingeri',
        'Clavulinopsis fusiformis',
        'Clavulinopsis laeticolor',
        'Ramariopsis kunzei',
        'Cortinarius iodes',
        'Cortinarius violaceus',
        'Crepidotus mollis',
        'Baeospora myosura',
        'Chondrostereum purpureum',
        'Entoloma abortivum',
        'Entoloma medianox',
        'Fistulina hepatica',
        'Laccaria amethysteo-occidentalis',
        'Laccaria amethystina',
        'Laccaria laccata',
        'Gliophorus psittacinus',
        'Hygrocybe acutoconica',
        'Hygrocybe singeri',
        'Hygrophorus eburneus',
        'Lichenomphalia umbellifera',
        'Hebeloma crustuliniforme',
        'Inocybe geophylla',
        'Lycoperdon perlatum',
        'Marasmius oreades',
        'Marasmius plicatulus',
        'Marasmius rotula',
        'Megacollybia rodmanii',
        'Cruentomycena viscidocruenta',
        'Mycena acicula',
        'Mycena epipterygia',
        'Mycena haematopus',
        'Mycena interrupta',
        'Mycena pura',
        'Panellus pusillus',
        'Panellus serotinus',
        'Panellus stipticus',
        'Sarcomyxa serotina',
        'Xeromphalina campanella',
        'Cyathus olla',
        'Gymnopus brassicolens',
        'Gymnopus dryophilus',
        'Gymnopus quercophilus',
        'Marasmiellus candidus',
        'Armillaria novae-zelandiae',
        'Armillaria sinapina',
        'Armillaria tabescens',
        'Strobilurus trullisatus',
        'Pleurotus ostreatus',
        'Pleurotus pulmonarius',
        'Phyllotopsis nidulans',
        'Lacrymaria lacrymabunda',
        'Panaeolus antillarum',
        'Panaeolus papilionaceus',
        'Psathyrella candolleana',
        'Psathyrella longipes',
        'Schizophyllum commune',
        'Agrocybe pediades',
        'Agrocybe praecox',
        'Agrocybe putaminum',
        'Deconica coprophila',
        'Galerina marginata',
        'Gymnopilus sapineus',
        'Gymnopilus ventricosus',
        'Hypholoma capnoides',
        'Leratiomyces percevalii',
        'Pholiota aurivella',
        'Psilocybe allenii',
        'Psilocybe cubensis',
        'Psilocybe cyanescens',
        'Stropharia rugosoannulata',
        'Caulorhiza umbonata',
        'Clitocybe nebularis',
        'Lepista nuda',
        'Leucopaxillus albissimus',
        'Leucopaxillus gentianeus',
        'Tricholoma dryophilum',
        'Tricholoma terreum',
        'Tricholomopsis rutilans',
        'Tubaria furfuracea',
        'Exidia nucleata',
        'Exidia recisa',
        'Rickenella fibula',
        'Chalciporus piperatus',
        'Exsudoporus frostii',
        'Retiboletus ornatipes',
        'Suillellus amygdalinus',
        'Xerocomellus chrysenteron',
        'Xerocomellus dryophilus',
        'Xerocomellus zelleri',
        'Xerocomus subtomentosus',
        'Chroogomphus vinicolor',
        'Gomphidius oregonensis',
        'Hygrophoropsis aurantiaca',
        'Paxillus involutus',
        'Pisolithus arhizus',
        'Pisolithus arrhizus',
        'Calostoma cinnabarinum',
        'Suillus americanus',
        'Suillus brevipes',
        'Suillus caerulescens',
        'Suillus granulatus',
        'Suillus luteus',
        'Suillus pictus',
        'Suillus pungens',
        'Tapinella atrotomentosa',
        'Cantharellus californicus',
        'Cantharellus lateritius',
        'Craterellus calicornucopioides',
        'Craterellus cornucopioides',
        'Craterellus fallax',
        'Craterellus tubaeformis',
        'Clavulina cristata',
        'Clavulina rugosa',
        'Hydnum repandum',
        'Gloeophyllum sepiarium',
        'Neolentinus ponderosus',
        'Ramaria stricta',
        'Inonotus dryadeus',
        'Inonotus obliquus',
        'Phellinus gilvus',
        'Phellinus tremulae',
        'Porodaedalea pini',
        'Contumyces rosellus',
        'Clathrus archeri',
        'Clathrus columnatus',
        'Lysurus mokusin',
        'Mutinus elegans',
        'Phallus hadriani',
        'Phallus indusiatus',
        'Phallus ravenelii',
        'Phallus rubicundus',
        'Daedalea quercina',
        'Fomitopsis cajanderi',
        'Fomitopsis pini-canadensis',
        'Ischnoderma resinosum',
        'Piptoporus betulinus',
        'Postia caesia',
        'Ganoderma applanatum',
        'Ganoderma brownii',
        'Ganoderma curtisii',
        'Ganoderma oregonense',
        'Ganoderma sessile',
        'Grifola frondosa',
        'Meripilus sumstinei',
        'Abortiporus biennis',
        'Bjerkandera adusta',
        'Gloeoporus dichrous',
        'Irpex lacteus',
        'Phlebia incarnata',
        'Phlebia tremellosa',
        'Podoscypha petalodes',
        'Climacodon septentrionalis',
        'Daedaleopsis confragosa',
        'Hexagonia hydnoides',
        'Laetiporus cincinnatus',
        'Laetiporus conifericola',
        'Laetiporus sulphureus',
        'Neofavolus alveolaris',
        'Panus conchatus',
        'Picipes badius',
        'Polyporus alveolaris',
        'Polyporus squamosus',
        'Poronidulus conchifer',
        'Pycnoporus sanguineus',
        'Trametes gibbosa',
        'Trametes hirsuta',
        'Trametes versicolor',
        'Trichaptum abietinum',
        'Tyromyces chioneus',
        'Auriscalpium vulgare',
        'Hericium americanum',
        'Lactarius deliciosus',
        'Lactarius rubrilacteus',
        'Lactarius rufulus',
        'Lactarius xanthogalactus',
        'Russula brevipes',
        'Russula cerolens',
        'Russula cremoricolor',
        'Russula sanguinea',
        'Stereum complicatum',
        'Stereum ostrea',
        'Xylobolus frustulatus',
        'Calocera cornea',
        'Calocera viscosa',
        'Dacrymyces chrysospermus',
        'Guepiniopsis alpina',
        'Allodus podophylli',
        'Puccinia lagenophorae',
        'Puccinia malvacearum',
        'Ductifera pululahuana',
        'Tremella foliacea',
        'Tremella mesenterica',
        'Entomophthora muscae',
        'Bryum argenteum',
        'Leucolepis acanthoneura',
        'Plagiomnium cuspidatum',
        'Plagiomnium insigne',
        'Rhizomnium glabrescens',
        'Dicranum scoparium',
        'Cyathophorum bulbosum',
        'Pleurozium schreberi',
        'Rhytidiadelphus loreus',
        'Rhytidiadelphus triquetrus',
        'Buckiella undulata',
        'Ptilium crista-castrensis',
        'Ptychomnion aciculare',
        'Dawsonia superba',
        'Polytrichum commune',
        'Polytrichum piliferum',
        'Cephaleuros lagerheimii',
        'Trentepohlia aurea',
        'Ulva intestinalis',
        'Equisetum arvense',
        'Equisetum fluviatile',
        'Equisetum hyemale',
        'Equisetum laevigatum',
        'Equisetum telmateia',
        'Equisetum variegatum',
        'Ephedra trifurca',
        'Ephedra viridis',
        'Huperzia lucidula',
        'Phlegmariurus varius',
        'Dendrolycopodium dendroideum',
        'Dendrolycopodium hickeyi',
        'Dendrolycopodium obscurum',
        'Diphasiastrum digitatum',
        'Lycopodium clavatum',
        'Lycopodium deuterodensum',
        'Lycopodium fastigiatum',
        'Lycopodium scariosum',
        'Spinulum annotinum',
        'Selaginella kraussiana',
        'Selaginella lepidophylla',
        'Selaginella rupestris',
        'Sagittaria latifolia',
        'Anthurium schlechtendalii',
        'Arisaema dracontium',
        'Arisaema triphyllum',
        'Lysichiton americanus',
        'Orontium aquaticum',
        'Spirodela polyrhiza',
        'Symplocarpus foetidus',
        'Syngonium podophyllum',
        'Xanthosoma robustum',
        'Zantedeschia aethiopica',
        'Elodea canadensis',
        'Hydrocharis morsus-ranae',
        'Najas minor',
        'Triglochin maritima',
        'Potamogeton crispus',
        'Potamogeton natans',
        'Triantha glutinosa',
        'Phyllospadix scouleri',
        'Phyllospadix torreyi',
        'Chamaerops humilis',
        'Cocos nucifera',
        'Rhopalostylis sapida',
        'Sabal mexicana',
        'Sabal minor',
        'Agapanthus praecox',
        'Allium acuminatum',
        'Allium campanulatum',
        'Allium dichlamydeum',
        'Allium falcifolium',
        'Allium schoenoprasum',
        'Allium stellatum',
        'Allium tricoccum',
        'Allium triquetrum',
        'Crinum americanum',
        'Habranthus tubispathus',
        'Hymenocallis liriosme',
        'Leucojum aestivum',
        'Narcissus tazetta',
        'Nothoscordum bivalve',
        'Sprekelia formosissima',
        'Zephyranthes atamasco',
        'Zephyranthes chlorosolen',
        'Zephyranthes drummondii',
        'Agave americana',
        'Agave deserti',
        'Agave inaequidens',
        'Agave palmeri',
        'Agave parryi',
        'Agave potatorum',
        'Agave salmiana',
        'Agave scabra',
        'Agave shawii',
        'Agave striata',
        'Agave utahensis',
        'Agave victoriae-reginae',
        'Agave vivipara',
        'Arthropodium cirratum',
        'Asparagus asparagoides',
        'Asparagus scandens',
        'Brodiaea coronaria',
        'Brodiaea terrestris',
        'Camassia leichtlinii',
        'Chlorogalum pomeridianum',
        'Cordyline australis',
        'Cordyline banksii',
        'Cordyline fruticosa',
        'Dasylirion texanum',
        'Dasylirion wheeleri',
        'Dichelostemma capitatum',
        'Dichelostemma volubile',
        'Drimia maritima',
        'Echeandia flavescens',
        'Hesperaloe parviflora',
        'Hesperocallis undulata',
        'Hesperoyucca whipplei',
        'Leopoldia comosa',
        'Leucocrinum montanum',
        'Liriope muscari',
        'Luzuriaga parviflora',
        'Maianthemum bifolium',
        'Maianthemum canadense',
        'Maianthemum dilatatum',
        'Maianthemum racemosum',
        'Maianthemum stellatum',
        'Maianthemum trifolium',
        'Manfreda virginica',
        'Muilla maritima',
        'Muscari armeniacum',
        'Nolina microcarpa',
        'Nolina parryi',
        'Nolina texana',
        'Ornithogalum umbellatum',
        'Polygonatum multiflorum',
        'Polygonatum odoratum',
        'Polygonatum pubescens',
        'Prospero autumnale',
        'Scilla bifolia',
        'Triteleia hyacinthina',
        'Triteleia laxa',
        'Yucca brevifolia',
        'Yucca constricta',
        'Yucca glauca',
        'Yucca madrensis',
        'Yucca pallida',
        'Yucca schidigera',
        'Yucca schottii',
        'Yucca torreyi',
        'Aloe arborescens',
        'Aloidendron dichotomum',
        'Asphodelus ramosus',
        'Dianella nigra',
        'Phormium tenax',
        'Astelia hastata',
        'Astelia nervosa',
        'Astelia solandri',
        'Hypoxis hirsuta',
        'Alophia drummondii',
        'Aristea ecklonii',
        'Chasmanthe floribunda',
        'Crocus vernus',
        'Dietes bicolor',
        'Iris douglasiana',
        'Iris fernaldii',
        'Iris germanica',
        'Iris macrosiphon',
        'Iris pseudacorus',
        'Iris tenax',
        'Iris versicolor',
        'Iris virginica',
        'Libertia ixioides',
        'Moraea sisyrinchium',
        'Sisyrinchium albidum',
        'Sisyrinchium bellum',
        'Sisyrinchium campestre',
        'Sisyrinchium rosulatum',
        'Acianthus sinclairii',
        'Anacamptis papilionacea',
        'Aporostylis bifolia',
        'Arethusa bulbosa',
        'Bulbophyllum pygmaeum',
        'Caladenia chlorostyla',
        'Caladenia lyallii',
        'Calypso bulbosa',
        'Cephalanthera austiniae',
        'Cephalanthera damasonium',
        'Cephalanthera longifolia',
        'Cephalanthera rubra',
        'Chiloglottis cornuta',
        'Corallorhiza maculata',
        'Corallorhiza odontorhiza',
        'Corallorhiza trifida',
        'Corallorhiza wisteriana',
        'Corybas cheesemanii',
        'Corybas macranthus',
        'Corybas oblongus',
        'Corybas trilobus',
        'Cypripedium acaule',
        'Cypripedium candidum',
        'Cypripedium montanum',
        'Cypripedium parviflorum',
        'Dactylorhiza maculata',
        'Dactylorhiza sambucina',
        'Dactylorhiza viridis',
        'Dendrobium cunninghamii',
        'Dichromanthus aurantiacus',
        'Drymoanthus adversus',
        'Earina mucronata',
        'Epidendrum radicans',
        'Epipactis helleborine',
        'Gastrodia cunninghamii',
        'Glossodia major',
        'Goodyera oblongifolia',
        'Goodyera pubescens',
        'Goodyera repens',
        'Goodyera tesselata',
        'Hexalectris spicata',
        'Himantoglossum hircinum',
        'Himantoglossum robertianum',
        'Limodorum abortivum',
        'Liparis liliifolia',
        'Liparis loeselii',
        'Malaxis unifolia',
        'Neotinea tridentata',
        'Neottia cordata',
        'Neottia nidus-avis',
        'Neottia ovata',
        'Oeceoclades maculata',
        'Ophrys apifera',
        'Ophrys bertolonii',
        'Ophrys insectifera',
        'Ophrys lutea',
        'Ophrys scolopax',
        'Ophrys speculum',
        'Ophrys sphegodes',
        'Orchis anthropophora',
        'Orchis militaris',
        'Orthoceras novae-zeelandiae',
        'Piperia elegans',
        'Piperia transversa',
        'Platanthera bifolia',
        'Platanthera blephariglottis',
        'Platanthera chlorantha',
        'Platanthera clavellata',
        'Platanthera dilatata',
        'Platanthera lacera',
        'Platanthera stricta',
        'Prasophyllum colensoi',
        'Pterostylis agathicola',
        'Pterostylis alobula',
        'Pterostylis banksii',
        'Pterostylis graminea',
        'Pterostylis montana',
        'Pterostylis patens',
        'Pterostylis trullifolia',
        'Serapias lingua',
        'Spathoglottis plicata',
        'Spiranthes lacera',
        'Spiranthes magnicamporum',
        'Spiranthes ovalis',
        'Spiranthes vernalis',
        'Thelymitra longifolia',
        'Thelymitra pauciflora',
        'Thelymitra pulchella',
        'Tipularia discolor',
        'Commelina coelestis',
        'Commelina communis',
        'Commelina dianthifolia',
        'Commelina diffusa',
        'Commelina erecta',
        'Tradescantia fluminensis',
        'Tradescantia occidentalis',
        'Tradescantia ohiensis',
        'Tradescantia spathacea',
        'Tradescantia zebrina',
        'Eichhornia crassipes',
        'Pontederia cordata',
        'Dioscorea bulbifera',
        'Dioscorea villosa',
        'Burchardia umbellata',
        'Gloriosa superba',
        'Uvularia grandiflora',
        'Uvularia sessilifolia',
        'Calochortus albus',
        'Calochortus argillosus',
        'Calochortus barbatus',
        'Calochortus catalinae',
        'Calochortus clavatus',
        'Calochortus gunnisonii',
        'Calochortus kennedyi',
        'Calochortus leichtlinii',
        'Calochortus luteus',
        'Calochortus minimus',
        'Calochortus monophyllus',
        'Calochortus pulchellus',
        'Calochortus uniflorus',
        'Calochortus weedii',
        'Clintonia borealis',
        'Erythronium albidum',
        'Erythronium americanum',
        'Erythronium montanum',
        'Erythronium umbilicatum',
        'Fritillaria affinis',
        'Fritillaria atropurpurea',
        'Fritillaria biflora',
        'Fritillaria camschatcensis',
        'Fritillaria liliacea',
        'Fritillaria pudica',
        'Fritillaria recurva',
        'Lilium bulbiferum',
        'Lilium canadense',
        'Lilium kelleyanum',
        'Medeola virginiana',
        'Prosartes lanuginosa',
        'Prosartes smithii',
        'Anticlea elegans',
        'Toxicoscordion fremontii',
        'Toxicoscordion paniculatum',
        'Toxicoscordion venenosum',
        'Trillium albidum',
        'Trillium catesbaei',
        'Trillium chloropetalum',
        'Trillium erectum',
        'Trillium flexipes',
        'Trillium grandiflorum',
        'Trillium nivale',
        'Trillium ovatum',
        'Trillium undulatum',
        'Veratrum californicum',
        'Veratrum viride',
        'Ripogonum scandens',
        'Smilax aspera',
        'Smilax bona-nox',
        'Smilax glauca',
        'Smilax laurifolia',
        'Smilax pumila',
        'Smilax smallii',
        'Freycinetia banksii',
        'Pandanus tectorius',
        'Aechmea bracteata',
        'Ananas comosus',
        'Bromelia pinguin',
        'Hechtia texensis',
        'Tillandsia caput-medusae',
        'Tillandsia ionantha',
        'Tillandsia recurvata',
        'Tillandsia schiedeana',
        'Tillandsia usneoides',
        'Carex albursina',
        'Carex aurea',
        'Carex buxbaumii',
        'Carex cherokeensis',
        'Carex comosa',
        'Carex crinita',
        'Carex eburnea',
        'Carex flava',
        'Carex gracillima',
        'Carex granularis',
        'Carex hystericina',
        'Carex lacustris',
        'Carex lurida',
        'Carex magellanica',
        'Carex oligosperma',
        'Carex pedunculata',
        'Carex pellita',
        'Carex pendula',
        'Carex plantaginea',
        'Carex pumila',
        'Carex rosea',
        'Carex secta',
        'Carex sprengelii',
        'Carex squarrosa',
        'Carex stipata',
        'Carex vulpinoidea',
        'Cladium mariscoides',
        'Cyperus echinatus',
        'Cyperus involucratus',
        'Cyperus papyrus',
        'Cyperus ustulatus',
        'Dulichium arundinaceum',
        'Eriophorum angustifolium',
        'Eriophorum vaginatum',
        'Eriophorum virginicum',
        'Ficinia nodosa',
        'Ficinia spiralis',
        'Rhynchospora corniculata',
        'Schoenoplectus acutus',
        'Schoenoplectus californicus',
        'Schoenoplectus pungens',
        'Scirpus atrovirens',
        'Scirpus microcarpus',
        'Scirpus pendulus',
        'Eriocaulon aquaticum',
        'Juncus acutus',
        'Juncus bufonius',
        'Juncus marginatus',
        'Juncus patens',
        'Juncus tenuis',
        'Juncus torreyi',
        'Luzula acuminata',
        'Luzula multiflora',
        'Agropyron cristatum',
        'Aira caryophyllea',
        'Alopecurus pratensis',
        'Andropogon gerardii',
        'Andropogon glomeratus',
        'Andropogon ternarius',
        'Andropogon virginicus',
        'Aristida purpurea',
        'Arundo donax',
        'Austroderia richardii',
        'Avena barbata',
        'Avena sativa',
        'Bothriochloa barbinodis',
        'Bouteloua curtipendula',
        'Bouteloua hirsuta',
        'Bouteloua rigidiseta',
        'Briza maxima',
        'Bromus catharticus',
        'Bromus diandrus',
        'Bromus inermis',
        'Bromus kalmii',
        'Bromus madritensis',
        'Bromus rubens',
        'Calamagrostis canadensis',
        'Cenchrus spinifex',
        'Chasmanthium latifolium',
        'Chloris cucullata',
        'Chloris verticillata',
        'Cynodon dactylon',
        'Dactylis glomerata',
        'Dactyloctenium aegyptium',
        'Danthonia spicata',
        'Dasyochloa pulchella',
        'Dichanthelium clandestinum',
        'Ehrharta erecta',
        'Eleusine indica',
        'Elymus elymoides',
        'Elymus glaucus',
        'Eragrostis secundiflora',
        'Glyceria striata',
        'Hilaria rigida',
        'Hordeum pusillum',
        'Koeleria macrantha',
        'Lachnagrostis billardierei',
        'Lamarckia aurea',
        'Leersia oryzoides',
        'Leymus condensatus',
        'Leymus mollis',
        'Lolium multiflorum',
        'Microlaena avenacea',
        'Microstegium vimineum',
        'Miscanthus sinensis',
        'Muhlenbergia lindheimeri',
        'Muhlenbergia rigens',
        'Nassella leucotricha',
        'Oplismenus hirtellus',
        'Oryzopsis asperifolia',
        'Panicum capillare',
        'Paspalum notatum',
        'Paspalum plicatulum',
        'Paspalum urvillei',
        'Pennisetum ciliare',
        'Pennisetum setaceum',
        'Pennisetum villosum',
        'Phragmites australis',
        'Poa bulbosa',
        'Poa pratensis',
        'Schizachyrium scoparium',
        'Setaria faberi',
        'Setaria parviflora',
        'Setaria pumila',
        'Setaria viridis',
        'Sorghastrum nutans',
        'Sorghum halepense',
        'Spartina alterniflora',
        'Spinifex sericeus',
        'Sporobolus heterolepis',
        'Stenotaphrum secundatum',
        'Stipa miliacea',
        'Tridens albescens',
        'Tridens strictus',
        'Apodasmia similis',
        'Sparganium eurycarpum',
        'Typha latifolia',
        'Typha orientalis',
        'Heliconia latispatha',
        'Heliconia rostrata',
        'Thalia geniculata',
        'Alpinia purpurata',
        'Etlingera elatior',
        'Hedychium coronarium',
        'Aciphylla aurea',
        'Aciphylla subflabellata',
        'Aegopodium podagraria',
        'Angelica sylvestris',
        'Anisotome aromatica',
        'Anthriscus caucalis',
        'Anthriscus sylvestris',
        'Apium nodiflorum',
        'Apium prostratum',
        'Astrantia major',
        'Bifora americana',
        'Bowlesia incana',
        'Centella uniflora',
        'Cicuta bulbifera',
        'Conium maculatum',
        'Crithmum maritimum',
        'Cyclospermum leptophyllum',
        'Daucus carota',
        'Eryngium campestre',
        'Eryngium carlinae',
        'Eryngium hookeri',
        'Eryngium leavenworthii',
        'Eryngium maritimum',
        'Eryngium proteiflorum',
        'Eryngium yuccifolium',
        'Ferula communis',
        'Foeniculum vulgare',
        'Gingidia montana',
        'Heracleum mantegazzianum',
        'Heracleum maximum',
        'Heracleum sphondylium',
        'Ligusticum grayi',
        'Ligusticum scoticum',
        'Lomatium mohavense',
        'Lomatium triternatum',
        'Oenanthe sarmentosa',
        'Osmorhiza claytonii',
        'Oxypolis rigidior',
        'Pastinaca sativa',
        'Petroselinum crispum',
        'Polytaenia texana',
        'Pseudocymopterus montanus',
        'Sanicula bipinnatifida',
        'Sanicula canadensis',
        'Sanicula crassicaulis',
        'Sanicula laciniata',
        'Sanicula odorata',
        'Sanicula tuberosa',
        'Sium suave',
        'Sphenosciadium capitellatum',
        'Taenidia integerrima',
        'Torilis arvensis',
        'Torilis nodosa',
        'Zizia aptera',
        'Zizia aurea',
        'Aralia elata',
        'Aralia hispida',
        'Aralia nudicaulis',
        'Aralia racemosa',
        'Fatsia japonica',
        'Hedera helix',
        'Hydrocotyle heteromeria',
        'Hydrocotyle moschata',
        'Hydrocotyle umbellata',
        'Oplopanax horridus',
        'Panax quinquefolius',
        'Panax trifolius',
        'Pseudopanax arboreus',
        'Pseudopanax colensoi',
        'Pseudopanax crassifolius',
        'Pseudopanax ferox',
        'Pseudopanax lessonii',
        'Pseudopanax linearis',
        'Raukaua anomalus',
        'Raukaua edgerleyi',
        'Schefflera actinophylla',
        'Schefflera digitata',
        'Griselinia littoralis',
        'Griselinia lucida',
        'Pennantia corymbosa',
        'Pittosporum crassifolium',
        'Pittosporum eugenioides',
        'Pittosporum ralphii',
        'Pittosporum tenuifolium',
        'Pittosporum undulatum',
        'Ilex aquifolium',
        'Ilex cassine',
        'Ilex glabra',
        'Ilex mucronata',
        'Ilex opaca',
        'Ilex verticillata',
        'Ilex vomitoria',
        'Alseuosmia macrophylla',
        'Achillea millefolium',
        'Achillea ptarmica',
        'Acmella repens',
        'Acourtia microcephala',
        'Adenophyllum porophylloides',
        'Ageratina adenophora',
        'Ageratina altissima',
        'Ageratina havanensis',
        'Ambrosia ambrosioides',
        'Ambrosia artemisiifolia',
        'Ambrosia chamissonis',
        'Ambrosia deltoidea',
        'Ambrosia dumosa',
        'Ambrosia psilostachya',
        'Ambrosia trifida',
        'Anaphalioides bellidioides',
        'Anaphalioides trinervis',
        'Anaphalis margaritacea',
        'Anisocoma acaulis',
        'Antennaria neglecta',
        'Antennaria plantaginifolia',
        'Anthemis cotula',
        'Arctium lappa',
        'Arctium minus',
        'Arctium tomentosum',
        'Arctotheca prostrata',
        'Argyroxiphium sandwicense',
        'Arnica cordifolia',
        'Arnica montana',
        'Arnoglossum atriplicifolium',
        'Arnoglossum ovatum',
        'Artemisia californica',
        'Artemisia campestris',
        'Artemisia douglasiana',
        'Artemisia dracunculus',
        'Artemisia filifolia',
        'Artemisia frigida',
        'Artemisia stelleriana',
        'Artemisia tridentata',
        'Artemisia vulgaris',
        'Aster alpinus',
        'Baccharis conferta',
        'Baccharis glutinosa',
        'Baccharis pilularis',
        'Baccharis salicifolia',
        'Bahiopsis laciniata',
        'Bahiopsis parishii',
        'Balsamorhiza hookeri',
        'Balsamorhiza sagittata',
        'Bebbia juncea',
        'Bellis perennis',
        'Berlandiera lyrata',
        'Bidens aristosa',
        'Bidens bipinnata',
        'Bidens laevis',
        'Bidens pilosa',
        'Blennosperma nanum',
        'Brachyglottis bellidioides',
        'Brachyglottis kirkii',
        'Brachyglottis lagopus',
        'Brachyglottis repanda',
        'Bradburia pilosa',
        'Brickellia eupatorioides',
        'Calendula arvensis',
        'Calycadenia multiglandulosa',
        'Calyptocarpus vialis',
        'Carduus acanthoides',
        'Carduus pycnocephalus',
        'Carlina acaulis',
        'Carthamus glaucus',
        'Celmisia gracilenta',
        'Celmisia semicordata',
        'Celmisia sessiliflora',
        'Celmisia spectabilis',
        'Centaurea americana',
        'Centaurea melitensis',
        'Centaurea nigra',
        'Centaurea scabiosa',
        'Centaurea solstitialis',
        'Chaenactis douglasii',
        'Chaenactis fremontii',
        'Chaenactis glabriuscula',
        'Chaenactis stevioides',
        'Chaetopappa ericoides',
        'Chrysactinia mexicana',
        'Chrysanthemoides monilifera',
        'Chrysothamnus viscidiflorus',
        'Cichorium intybus',
        'Cirsium altissimum',
        'Cirsium arizonicum',
        'Cirsium arvense',
        'Cirsium edule',
        'Cirsium ehrenbergii',
        'Cirsium neomexicanum',
        'Cirsium occidentale',
        'Cirsium oleraceum',
        'Cirsium palustre',
        'Cirsium pitcheri',
        'Cirsium quercetorum',
        'Cirsium texanum',
        'Cirsium vulgare',
        'Conoclinium coelestinum',
        'Coreopsis basalis',
        'Coreopsis palmata',
        'Coreopsis tinctoria',
        'Cosmos sulphureus',
        'Cotula australis',
        'Cotula coronopifolia',
        'Deinandra fasciculata',
        'Delairea odorata',
        'Diaperia prolifera',
        'Diaperia verna',
        'Dicoria canescens',
        'Dieteria canescens',
        'Dittrichia graveolens',
        'Dittrichia viscosa',
        'Dyssodia papposa',
        'Echinacea angustifolia',
        'Echinacea purpurea',
        'Echinacea sanguinea',
        'Elephantopus tomentosus',
        'Emilia fosbergii',
        'Emilia sonchifolia',
        'Encelia californica',
        'Encelia farinosa',
        'Engelmannia peristenia',
        'Ericameria cuneata',
        'Ericameria ericoides',
        'Ericameria laricifolia',
        'Erigeron bonariensis',
        'Erigeron canadensis',
        'Erigeron coulteri',
        'Erigeron foliosus',
        'Erigeron glaucus',
        'Erigeron philadelphicus',
        'Erigeron pulchellus',
        'Erigeron sumatrensis',
        'Eriophyllum confertiflorum',
        'Eriophyllum lanosum',
        'Eriophyllum staechadifolium',
        'Eucephalus breweri',
        'Eupatorium perfoliatum',
        'Eupatorium serotinum',
        'Euthamia graminifolia',
        'Eutrochium fistulosum',
        'Eutrochium maculatum',
        'Facelis retusa',
        'Gaillardia aristata',
        'Gaillardia pinnatifida',
        'Gaillardia pulchella',
        'Galactites tomentosa',
        'Gamochaeta pensylvanica',
        'Gazania linearis',
        'Gnaphalium uliginosum',
        'Grindelia ciliata',
        'Grindelia integrifolia',
        'Grindelia lanceolata',
        'Gymnosperma glutinosum',
        'Hazardia squarrosa',
        'Helenium amarum',
        'Helenium elegans',
        'Helenium flexuosum',
        'Helianthus angustifolius',
        'Helianthus annuus',
        'Helianthus debilis',
        'Helianthus divaricatus',
        'Helianthus giganteus',
        'Helianthus grosseserratus',
        'Helianthus occidentalis',
        'Helianthus petiolaris',
        'Helichrysum lanceolatum',
        'Helminthotheca echioides',
        'Heterotheca grandiflora',
        'Heterotheca sessiliflora',
        'Heterotheca subaxillaris',
        'Heterotheca villosa',
        'Hieracium aurantiacum',
        'Hieracium horridum',
        'Hieracium lepidulum',
        'Hymenopappus artemisiifolius',
        'Hypochaeris glabra',
        'Hypochaeris radicata',
        'Isocoma acradenia',
        'Iva frutescens',
        'Jacobaea maritima',
        'Jacobaea vulgaris',
        'Krigia biflora',
        'Lactuca biennis',
        'Lactuca canadensis',
        'Lactuca floridana',
        'Lactuca serriola',
        'Lagenophora pumila',
        'Lasthenia gracilis',
        'Layia glandulosa',
        'Leontopodium nivale',
        'Lepidospartum squamatum',
        'Leptinella dioica',
        'Leptinella squalida',
        'Leucanthemum vulgare',
        'Leucogenes grandiceps',
        'Liatris acidota',
        'Liatris cylindracea',
        'Liatris elegans',
        'Liatris punctata',
        'Lindheimera texana',
        'Logfia gallica',
        'Machaeranthera tanacetifolia',
        'Madia gracilis',
        'Malacothrix saxatilis',
        'Matricaria chamomilla',
        'Matricaria discoidea',
        'Melampodium leucanthum',
        'Micropus californicus',
        'Monoptilon bellioides',
        'Mycelis muralis',
        'Nabalus albus',
        'Nabalus trifoliolatus',
        'Oclemena acuminata',
        'Olearia arborescens',
        'Olearia avicenniifolia',
        'Olearia colensoi',
        'Olearia ilicifolia',
        'Olearia lineata',
        'Olearia nummulariifolia',
        'Olearia rani',
        'Olearia solandri',
        'Oncosiphon piluliferum',
        'Onopordum acanthium',
        'Osteospermum fruticosum',
        'Ozothamnus leptophyllus',
        'Packera multilobata',
        'Palafoxia arida',
        'Palafoxia rosea',
        'Parthenium incanum',
        'Pectis papposa',
        'Perityle emoryi',
        'Petasites albus',
        'Petasites frigidus',
        'Petasites hybridus',
        'Peucephyllum schottii',
        'Pinaropappus roseus',
        'Pityopsis graminifolia',
        'Pluchea odorata',
        'Pluchea sericea',
        'Polymnia canadensis',
        'Porophyllum gracile',
        'Prenanthes purpurea',
        'Psathyrotes ramosissima',
        'Pseudognaphalium biolettii',
        'Pseudognaphalium luteoalbum',
        'Pseudognaphalium stramineum',
        'Psilocarphus tenellus',
        'Psilostrophe cooperi',
        'Psilostrophe tagetina',
        'Pulicaria dysenterica',
        'Rafinesquia californica',
        'Rafinesquia neomexicana',
        'Raoulia australis',
        'Raoulia grandiflora',
        'Raoulia subsericea',
        'Ratibida columnifera',
        'Rudbeckia hirta',
        'Rudbeckia subtomentosa',
        'Rudbeckia texana',
        'Sanvitalia procumbens',
        'Scorzoneroides autumnalis',
        'Senecio ampullaceus',
        'Senecio angulatus',
        'Senecio aronicoides',
        'Senecio bipinnatisectus',
        'Senecio biserratus',
        'Senecio flaccidus',
        'Senecio glastifolius',
        'Senecio glaucophyllus',
        'Senecio glomeratus',
        'Senecio hispidulus',
        'Senecio inaequidens',
        'Senecio integerrimus',
        'Senecio lautus',
        'Senecio praecox',
        'Senecio quadridentatus',
        'Senecio salignus',
        'Senecio skirrhodon',
        'Senecio vulgaris',
        'Silphium albiflorum',
        'Silphium integrifolium',
        'Silphium laciniatum',
        'Silybum marianum',
        'Solidago caesia',
        'Solidago canadensis',
        'Solidago flexicaulis',
        'Solidago gigantea',
        'Solidago juncea',
        'Solidago multiradiata',
        'Solidago nemoralis',
        'Solidago ohioensis',
        'Solidago ptarmicoides',
        'Solidago riddellii',
        'Solidago rugosa',
        'Solidago speciosa',
        'Solidago uliginosa',
        'Solidago ulmifolia',
        'Solidago velutina',
        'Solidago virgaurea',
        'Soliva sessilis',
        'Sonchus arvensis',
        'Sonchus asper',
        'Sonchus oleraceus',
        'Sphagneticola trilobata',
        'Stephanomeria pauciflora',
        'Symphyotrichum chilense',
        'Symphyotrichum divaricatum',
        'Symphyotrichum drummondii',
        'Symphyotrichum laeve',
        'Symphyotrichum lanceolatum',
        'Symphyotrichum novae-angliae',
        'Symphyotrichum novi-belgii',
        'Symphyotrichum oolentangiense',
        'Symphyotrichum pilosum',
        'Symphyotrichum puniceum',
        'Symphyotrichum sericeum',
        'Symphyotrichum subulatum',
        'Symphyotrichum urophyllum',
        'Tagetes erecta',
        'Tagetes lucida',
        'Tanacetum vulgare',
        'Taraxacum erythrospermum',
        'Taraxacum officinale',
        'Tetraneuris linearifolia',
        'Thelesperma filifolium',
        'Thelesperma megapotamicum',
        'Tithonia tubaeformis',
        'Tragopogon dubius',
        'Tragopogon porrifolius',
        'Tridax procumbens',
        'Tripleurospermum inodorum',
        'Trixis californica',
        'Tussilago farfara',
        'Uropappus lindleyi',
        'Urospermum dalechampii',
        'Venegasia carpesioides',
        'Verbesina alternifolia',
        'Verbesina virginica',
        'Vernonia baldwinii',
        'Vernonia fasciculata',
        'Vernonia gigantea',
        'Vernonia noveboracensis',
        'Viguiera dentata',
        'Wedelia hispida',
        'Wyethia amplexicaulis',
        'Wyethia helenioides',
        'Xanthisma spinulosum',
        'Xanthium spinosum',
        'Xanthium strumarium',
        'Xylorhiza tortifolia',
        'Zinnia acerosa',
        'Zinnia elegans',
        'Zinnia grandiflora',
        'Campanula aparinoides',
        'Campanula glomerata',
        'Campanula persicifolia',
        'Campanula rapunculus',
        'Campanula rotundifolia',
        'Campanula trachelium',
        'Campanulastrum americanum',
        'Hippobroma longiflora',
        'Jasione montana',
        'Lobelia anceps',
        'Lobelia angulata',
        'Lobelia cardinalis',
        'Lobelia laxiflora',
        'Lobelia puberula',
        'Lobelia siphilitica',
        'Wahlenbergia albomarginata',
        'Selliera radicans',
        'Donatia novae-zelandiae',
        'Phyllachne colensoi',
        'Amsinckia intermedia',
        'Amsinckia menziesii',
        'Amsinckia tessellata',
        'Anchusa arvensis',
        'Anchusa officinalis',
        'Cordia sebestena',
        'Cynoglossum grande',
        'Cynoglossum virginianum',
        'Echium plantagineum',
        'Echium vulgare',
        'Ehretia anacua',
        'Eriodictyon californicum',
        'Eriodictyon trichocalyx',
        'Eucrypta chrysanthemifolia',
        'Heliotropium angiospermum',
        'Heliotropium curassavicum',
        'Heliotropium europaeum',
        'Heliotropium procumbens',
        'Hydrophyllum appendiculatum',
        'Hydrophyllum canadense',
        'Hydrophyllum virginianum',
        'Lithospermum caroliniense',
        'Lithospermum incisum',
        'Lithospermum latifolium',
        'Lithospermum ruderale',
        'Mertensia maritima',
        'Mertensia virginica',
        'Myosotis arvensis',
        'Myosotis discolor',
        'Myosotis laxa',
        'Myosotis macrosperma',
        'Nama hispida',
        'Nemophila maculata',
        'Nemophila menziesii',
        'Nemophila parviflora',
        'Onosmodium bejariense',
        'Pentaglottis sempervirens',
        'Phacelia campanularia',
        'Phacelia fremontii',
        'Phacelia hastata',
        'Phacelia imbricata',
        'Phacelia malvifolia',
        'Phacelia parryi',
        'Phacelia ramosissima',
        'Phacelia sericea',
        'Phacelia tanacetifolia',
        'Pholisma arenarium',
        'Pholistoma membranaceum',
        'Pulmonaria officinalis',
        'Tiquilia plicata',
        'Tournefortia gnaphalodes',
        'Wigandia urens',
        'Batis maritima',
        'Alliaria petiolata',
        'Arabidopsis lyrata',
        'Arabidopsis thaliana',
        'Barbarea orthoceras',
        'Barbarea vulgaris',
        'Boechera laevigata',
        'Brassica oleracea',
        'Brassica tournefortii',
        'Cakile maritima',
        'Capsella bursa-pastoris',
        'Cardamine californica',
        'Cardamine concatenata',
        'Cardamine diphylla',
        'Cardamine pensylvanica',
        'Caulanthus inflatus',
        'Chorispora tenella',
        'Diplotaxis tenuifolia',
        'Dithyrea californica',
        'Draba cuneifolia',
        'Eruca vesicaria',
        'Erysimum capitatum',
        'Erysimum cheiranthoides',
        'Erysimum cheiri',
        'Hesperidanthus linearifolius',
        'Hesperis matronalis',
        'Hirschfeldia incana',
        'Lepidium campestre',
        'Lepidium didymum',
        'Lepidium draba',
        'Lepidium lasiocarpum',
        'Lobularia maritima',
        'Nasturtium officinale',
        'Physaria gordonii',
        'Raphanus sativus',
        'Sisymbrium altissimum',
        'Sisymbrium officinale',
        'Streptanthus glandulosus',
        'Turritis glabra',
        'Crateva tapia',
        'Peritoma arborea',
        'Peritoma serrulata',
        'Koeberlinia spinosa',
        'Floerkea proserpinacoides',
        'Limnanthes douglasii',
        'Reseda lutea',
        'Tropaeolum majus',
        'Buxus sempervirens',
        'Pseudowintera axillaris',
        'Carpobrotus edulis',
        'Conicosia pugioniformis',
        'Disphyma australe',
        'Mesembryanthemum nodiflorum',
        'Tetragonia implexicoma',
        'Allenrolfea occidentalis',
        'Alternanthera caracasana',
        'Amaranthus albus',
        'Amaranthus palmeri',
        'Amaranthus retroflexus',
        'Atriplex leucophylla',
        'Atriplex polycarpa',
        'Atriplex semibaccata',
        'Bassia scoparia',
        'Beta vulgaris',
        'Chenopodium album',
        'Chenopodium murale',
        'Dysphania ambrosioides',
        'Dysphania pumilio',
        'Froelichia floridana',
        'Gomphrena serrata',
        'Grayia spinosa',
        'Krascheninnikovia lanata',
        'Salicornia pacifica',
        'Salsola australis',
        'Salsola kali',
        'Sarcocornia quinqueflora',
        'Tidestromia lanuginosa',
        'Acanthocereus tetragonus',
        'Ariocarpus fissuratus',
        'Ariocarpus retusus',
        'Astrophytum ornatum',
        'Bergerocactus emoryi',
        'Carnegiea gigantea',
        'Coryphantha erecta',
        'Coryphantha radians',
        'Cylindropuntia acanthocarpa',
        'Cylindropuntia alcahes',
        'Cylindropuntia arbuscula',
        'Cylindropuntia californica',
        'Cylindropuntia cholla',
        'Cylindropuntia ganderi',
        'Cylindropuntia imbricata',
        'Cylindropuntia leptocaulis',
        'Cylindropuntia spinosior',
        'Cylindropuntia tunicata',
        'Cylindropuntia versicolor',
        'Echinocactus platyacanthus',
        'Echinocactus polycephalus',
        'Echinocereus cinerascens',
        'Echinocereus coccineus',
        'Echinocereus dasyacanthus',
        'Echinocereus engelmannii',
        'Echinocereus enneacanthus',
        'Echinocereus pectinatus',
        'Echinocereus pentalophus',
        'Echinocereus rigidissimus',
        'Epithelantha micromeris',
        'Escobaria missouriensis',
        'Escobaria vivipara',
        'Ferocactus echidne',
        'Ferocactus histrix',
        'Ferocactus viridescens',
        'Hylocereus undatus',
        'Lophocereus schottii',
        'Mammillaria carnea',
        'Mammillaria chionocephala',
        'Mammillaria grahamii',
        'Mammillaria haageana',
        'Mammillaria magnimamma',
        'Mammillaria prolifera',
        'Mammillaria tetrancistra',
        'Marginatocereus marginatus',
        'Neolloydia conoidea',
        'Opuntia basilaris',
        'Opuntia chlorotica',
        'Opuntia cochenillifera',
        'Opuntia decumbens',
        'Opuntia engelmannii',
        'Opuntia fragilis',
        'Opuntia humifusa',
        'Opuntia littoralis',
        'Opuntia macrocentra',
        'Opuntia macrorhiza',
        'Opuntia oricola',
        'Opuntia robusta',
        'Opuntia santa-rita',
        'Opuntia stenopetala',
        'Opuntia streptacantha',
        'Opuntia tomentosa',
        'Pachycereus pecten-aboriginum',
        'Peniocereus greggii',
        'Rhipsalis baccifera',
        'Stenocereus gummosus',
        'Thelocactus bicolor',
        'Arenaria serpyllifolia',
        'Cardionema ramosissimum',
        'Dianthus armeria',
        'Dianthus carthusianorum',
        'Dianthus deltoides',
        'Herniaria glabra',
        'Honckenya peploides',
        'Minuartia douglasii',
        'Minuartia groenlandica',
        'Moehringia lateriflora',
        'Petrorhagia prolifera',
        'Polycarpon tetraphyllum',
        'Sagina procumbens',
        'Saponaria officinalis',
        'Silene dioica',
        'Silene gallica',
        'Silene laciniata',
        'Silene latifolia',
        'Silene nutans',
        'Silene regia',
        'Silene uniflora',
        'Silene vulgaris',
        'Spergula arvensis',
        'Spergularia macrotheca',
        'Stellaria graminea',
        'Stellaria holostea',
        'Stellaria longipes',
        'Stellaria media',
        'Dionaea muscipula',
        'Drosera auriculata',
        'Drosera binata',
        'Drosera brevifolia',
        'Drosera capillaris',
        'Drosera pygmaea',
        'Drosera rotundifolia',
        'Mollugo verticillata',
        'Calyptridium monandrum',
        'Calyptridium umbellatum',
        'Claytonia caroliniana',
        'Claytonia exigua',
        'Claytonia gypsophiloides',
        'Claytonia parviflora',
        'Claytonia perfoliata',
        'Claytonia rubra',
        'Claytonia sibirica',
        'Claytonia virginica',
        'Lewisia leeana',
        'Lewisia triphylla',
        'Montia parvifolia',
        'Abronia latifolia',
        'Boerhavia erecta',
        'Mirabilis albida',
        'Mirabilis jalapa',
        'Mirabilis multiflora',
        'Mirabilis nyctaginea',
        'Nyctaginia capitata',
        'Phytolacca americana',
        'Phytolacca icosandra',
        'Phytolacca octandra',
        'Rivina humilis',
        'Limonium californicum',
        'Limonium carolinianum',
        'Limonium perezii',
        'Plumbago auriculata',
        'Aconogonon davisiae',
        'Bistorta bistortoides',
        'Chorizanthe brevicornu',
        'Chorizanthe fimbriata',
        'Chorizanthe rigida',
        'Eriogonum arborescens',
        'Eriogonum cinereum',
        'Eriogonum deflexum',
        'Eriogonum elongatum',
        'Eriogonum fasciculatum',
        'Eriogonum grande',
        'Eriogonum incanum',
        'Eriogonum latifolium',
        'Eriogonum longifolium',
        'Eriogonum nudum',
        'Eriogonum ovalifolium',
        'Eriogonum wrightii',
        'Fallopia japonica',
        'Fallopia scandens',
        'Muehlenbeckia axillaris',
        'Muehlenbeckia complexa',
        'Persicaria arifolia',
        'Persicaria bistorta',
        'Persicaria hydropiperoides',
        'Persicaria longiseta',
        'Persicaria punctata',
        'Persicaria virginiana',
        'Persicaria vivipara',
        'Polygonum aviculare',
        'Pterostegia drymarioides',
        'Rumex acetosella',
        'Rumex crispus',
        'Rumex hymenosepalus',
        'Rumex pulcher',
        'Rumex sagittatus',
        'Portulaca oleracea',
        'Tamarix aphylla',
        'Canotia holacantha',
        'Celastrus orbiculatus',
        'Euonymus alatus',
        'Euonymus americanus',
        'Euonymus atropurpureus',
        'Euonymus europaeus',
        'Euonymus obovatus',
        'Maytenus boaria',
        'Parnassia fimbriata',
        'Parnassia glauca',
        'Paxistima myrsinites',
        'Ascarina lucida',
        'Cornus alternifolia',
        'Cornus canadensis',
        'Cornus drummondii',
        'Cornus florida',
        'Cornus kousa',
        'Cornus rugosa',
        'Cornus sericea',
        'Cornus suecica',
        'Hydrangea macrophylla',
        'Hydrangea quercifolia',
        'Whipplea modesta',
        'Cevallia sinuata',
        'Eucnide urens',
        'Mentzelia albicaulis',
        'Mentzelia hispida',
        'Mentzelia involucrata',
        'Mentzelia oligosperma',
        'Petalonyx thurberi',
        'Begonia gracilis',
        'Coriaria sarmentosa',
        'Brandegea bigelovii',
        'Bryonia cretica',
        'Cucurbita foetidissima',
        'Cucurbita palmata',
        'Cucurbita pepo',
        'Ecballium elaterium',
        'Echinocystis lobata',
        'Marah fabacea',
        'Marah macrocarpa',
        'Sambucus ebulus',
        'Sambucus nigra',
        'Sambucus racemosa',
        'Viburnum acerifolium',
        'Viburnum edule',
        'Viburnum lantana',
        'Viburnum lantanoides',
        'Viburnum lentago',
        'Viburnum nudum',
        'Viburnum opulus',
        'Viburnum rafinesquianum',
        'Viburnum tinus',
        'Centranthus ruber',
        'Dipsacus fullonum',
        'Dipsacus laciniatus',
        'Dipsacus sativus',
        'Knautia arvensis',
        'Linnaea borealis',
        'Lonicera albiflora',
        'Lonicera hispidula',
        'Lonicera involucrata',
        'Lonicera japonica',
        'Lonicera maackii',
        'Lonicera periclymenum',
        'Lonicera sempervirens',
        'Lonicera subspicata',
        'Lonicera xylosteum',
        'Plectritis congesta',
        'Plectritis macrocera',
        'Succisa pratensis',
        'Symphoricarpos albus',
        'Symphoricarpos oreophilus',
        'Triosteum aurantiacum',
        'Valeriana californica',
        'Valeriana edulis',
        'Valeriana officinalis',
        'Valerianella locusta',
        'Impatiens capensis',
        'Impatiens pallida',
        'Impatiens walleriana',
        'Cyrilla racemiflora',
        'Diospyros virginiana',
        'Acrothamnus colensoi',
        'Andromeda polifolia',
        'Androstoma empetrifolium',
        'Arbutus arizonica',
        'Arbutus menziesii',
        'Arctostaphylos glandulosa',
        'Arctostaphylos nevadensis',
        'Chamaedaphne calyculata',
        'Chimaphila maculata',
        'Chimaphila umbellata',
        'Dracophyllum filifolium',
        'Dracophyllum longifolium',
        'Dracophyllum traversii',
        'Epacris impressa',
        'Epigaea repens',
        'Erica cinerea',
        'Erica lusitanica',
        'Gaultheria antipoda',
        'Gaultheria crassa',
        'Gaultheria depressa',
        'Gaultheria macrostigma',
        'Gaultheria procumbens',
        'Gaultheria shallon',
        'Gaylussacia baccata',
        'Kalmia latifolia',
        'Kalmia procumbens',
        'Leucopogon fraseri',
        'Leucothoe fontanesiana',
        'Menziesia ferruginea',
        'Monotropa hypopitys',
        'Monotropa uniflora',
        'Pentachondra pumila',
        'Pterospora andromedea',
        'Pyrola americana',
        'Pyrola picta',
        'Rhododendron calendulaceum',
        'Rhododendron canescens',
        'Rhododendron catawbiense',
        'Rhododendron ferrugineum',
        'Rhododendron groenlandicum',
        'Rhododendron lapponicum',
        'Rhododendron macrophyllum',
        'Rhododendron tomentosum',
        'Sarcodes sanguinea',
        'Vaccinium arboreum',
        'Vaccinium myrtilloides',
        'Vaccinium ovalifolium',
        'Vaccinium ovatum',
        'Vaccinium parvifolium',
        'Vaccinium stamineum',
        'Xylococcus bicolor',
        'Fouquieria columnaris',
        'Fouquieria diguetii',
        'Fouquieria splendens',
        'Collomia heterophylla',
        'Collomia linearis',
        'Eriastrum densifolium',
        'Eriastrum eremicum',
        'Eriastrum sapphirinum',
        'Gilia achilleifolia',
        'Ipomopsis aggregata',
        'Ipomopsis tenuituba',
        'Langloisia setosissima',
        'Leptosiphon androsaceus',
        'Leptosiphon bicolor',
        'Leptosiphon parviflorus',
        'Linanthus dianthiflorus',
        'Linanthus pungens',
        'Loeseliastrum matthewsii',
        'Navarretia leptalea',
        'Phlox divaricata',
        'Phlox drummondii',
        'Phlox hoodii',
        'Phlox longifolia',
        'Phlox paniculata',
        'Polemonium pulcherrimum',
        'Polemonium reptans',
        'Anagallis arvensis',
        'Cyclamen hederifolium',
        'Cyclamen persicum',
        'Lysimachia borealis',
        'Lysimachia ciliata',
        'Lysimachia latifolia',
        'Lysimachia maritima',
        'Lysimachia punctata',
        'Lysimachia quadriflora',
        'Lysimachia terrestris',
        'Lysimachia vulgaris',
        'Myrsine australis',
        'Myrsine salicina',
        'Primula elatior',
        'Primula hendersonii',
        'Primula jeffreyi',
        'Primula meadia',
        'Primula paucifolia',
        'Primula suffrutescens',
        'Samolus repens',
        'Samolus valerandi',
        'Trientalis europaea',
        'Sarracenia leucophylla',
        'Sarracenia minor',
        'Sarracenia purpurea',
        'Symplocos tinctoria',
        'Abrus precatorius',
        'Acacia angustissima',
        'Acacia cochliacantha',
        'Acacia constricta',
        'Acacia cornigera',
        'Acacia dealbata',
        'Acacia longifolia',
        'Acacia melanoxylon',
        'Acacia paradoxa',
        'Acacia retinodes',
        'Acacia rigidula',
        'Acacia verticillata',
        'Acmispon argophyllus',
        'Acmispon glaber',
        'Acmispon rigidus',
        'Acmispon strigosus',
        'Acmispon wrangelianus',
        'Albizia julibrissin',
        'Amorpha californica',
        'Amorpha canescens',
        'Amphicarpaea bracteata',
        'Anthyllis vulneraria',
        'Astragalus canadensis',
        'Astragalus crassicarpus',
        'Astragalus lentiginosus',
        'Astragalus mollissimus',
        'Astragalus purshii',
        'Astragalus trichopodus',
        'Baptisia alba',
        'Baptisia sphaerocarpa',
        'Bauhinia divaricata',
        'Bituminaria bituminosa',
        'Caesalpinia bonduc',
        'Caesalpinia gilliesii',
        'Caesalpinia mexicana',
        'Caesalpinia platyloba',
        'Calliandra eriophylla',
        'Calliandra grandiflora',
        'Canavalia rosea',
        'Carmichaelia australis',
        'Centrosema virginianum',
        'Ceratonia siliqua',
        'Cercis canadensis',
        'Cercis siliquastrum',
        'Chamaecrista fasciculata',
        'Chamaecrista nictitans',
        'Clitoria ternatea',
        'Crotalaria pumila',
        'Cytisus proliferus',
        'Cytisus scoparius',
        'Dalea aurea',
        'Dalea formosa',
        'Dalea multiflora',
        'Desmodium rotundifolium',
        'Enterolobium cyclocarpum',
        'Erythrina flabelliformis',
        'Eysenhardtia polystachya',
        'Genista monspessulana',
        'Gleditsia triacanthos',
        'Gliricidia sepium',
        'Glycyrrhiza lepidota',
        'Gymnocladus dioicus',
        'Haematoxylum brasiletto',
        'Hoffmannseggia glauca',
        'Hosackia crassifolia',
        'Hylodesmum glutinosum',
        'Lathyrus hirsutus',
        'Lathyrus latifolius',
        'Lathyrus littoralis',
        'Lathyrus ochroleucus',
        'Lathyrus palustris',
        'Lathyrus pratensis',
        'Lathyrus vernus',
        'Lespedeza virginica',
        'Lotus corniculatus',
        'Lotus pedunculatus',
        'Lupinus albifrons',
        'Lupinus arboreus',
        'Lupinus arcticus',
        'Lupinus argenteus',
        'Lupinus arizonicus',
        'Lupinus bicolor',
        'Lupinus chamissonis',
        'Lupinus concinnus',
        'Lupinus latifolius',
        'Lupinus lepidus',
        'Lupinus microcarpus',
        'Lupinus nootkatensis',
        'Lupinus sparsiflorus',
        'Lupinus texensis',
        'Lupinus variicolor',
        'Macroptilium atropurpureum',
        'Macroptilium gibbosifolium',
        'Macroptilium lathyroides',
        'Marina parryi',
        'Medicago lupulina',
        'Medicago minima',
        'Medicago polymorpha',
        'Medicago sativa',
        'Melilotus albus',
        'Melilotus officinalis',
        'Mimosa aculeaticarpa',
        'Mimosa borealis',
        'Mimosa quadrivalvis',
        'Mimosa strigillosa',
        'Neptunia pubescens',
        'Olneya tesota',
        'Onobrychis viciifolia',
        'Oxytropis lambertii',
        'Parkinsonia microphylla',
        'Pediomelum linearifolium',
        'Pediomelum tenuiflorum',
        'Phaseolus coccineus',
        'Pithecellobium dulce',
        'Prosopis glandulosa',
        'Prosopis velutina',
        'Psorothamnus emoryi',
        'Psorothamnus schottii',
        'Robinia neomexicana',
        'Robinia pseudoacacia',
        'Securigera varia',
        'Senna armata',
        'Senna covesii',
        'Senna marilandica',
        'Senna obtusifolia',
        'Sophora microphylla',
        'Sophora prostrata',
        'Sophora secundiflora',
        'Strophostyles umbellata',
        'Stylosanthes biflora',
        'Tamarindus indica',
        'Thermopsis californica',
        'Trifolium angustifolium',
        'Trifolium arvense',
        'Trifolium depauperatum',
        'Trifolium fucatum',
        'Trifolium pratense',
        'Trifolium repens',
        'Trifolium resupinatum',
        'Trifolium subterraneum',
        'Trifolium wormskioldii',
        'Ulex europaeus',
        'Vachellia farnesiana',
        'Vicia cracca',
        'Vicia gigantea',
        'Vicia hirsuta',
        'Vicia ludoviciana',
        'Vicia sativa',
        'Vicia sepium',
        'Vicia villosa',
        'Polygala lutea',
        'Polygala paucifolia',
        'Polygala polygama',
        'Polygala senega',
        'Alnus rhombifolia',
        'Alnus rubra',
        'Alnus serrulata',
        'Alnus viridis',
        'Betula alleghaniensis',
        'Betula cordifolia',
        'Betula nana',
        'Betula papyrifera',
        'Betula populifolia',
        'Betula pumila',
        'Carpinus caroliniana',
        'Corylus cornuta',
        'Ostrya virginiana',
        'Casuarina equisetifolia',
        'Castanea dentata',
        'Castanea pumila',
        'Fagus grandifolia',
        'Notholithocarpus densiflorus',
        'Quercus agrifolia',
        'Quercus alba',
        'Quercus berberidifolia',
        'Quercus buckleyi',
        'Quercus coccifera',
        'Quercus coccinea',
        'Quercus ellipsoidalis',
        'Quercus emoryi',
        'Quercus engelmannii',
        'Quercus havardii',
        'Quercus hypoleucoides',
        'Quercus imbricaria',
        'Quercus kelloggii',
        'Quercus laevis',
        'Quercus lobata',
        'Quercus macrocarpa',
        'Quercus marilandica',
        'Quercus montana',
        'Quercus nigra',
        'Quercus phellos',
        'Quercus rubra',
        'Quercus rugosa',
        'Quercus shumardii',
        'Quercus stellata',
        'Quercus suber',
        'Quercus turbinella',
        'Quercus vacciniifolia',
        'Quercus velutina',
        'Carya cordiformis',
        'Carya illinoinensis',
        'Carya ovata',
        'Carya tomentosa',
        'Juglans californica',
        'Juglans cinerea',
        'Juglans major',
        'Juglans nigra',
        'Comptonia peregrina',
        'Myrica gale',
        'Fuscospora fusca',
        'Lophozonia menziesii',
        'Garrya ovata',
        'Garrya wrightii',
        'Amsonia ciliata',
        'Amsonia tabernaemontana',
        'Amsonia tomentosa',
        'Apocynum androsaemifolium',
        'Apocynum cannabinum',
        'Araujia hortorum',
        'Araujia sericifera',
        'Asclepias albicans',
        'Asclepias arenaria',
        'Asclepias asperula',
        'Asclepias californica',
        'Asclepias curassavica',
        'Asclepias engelmanniana',
        'Asclepias erosa',
        'Asclepias fascicularis',
        'Asclepias humistrata',
        'Asclepias incarnata',
        'Asclepias latifolia',
        'Asclepias linearis',
        'Asclepias longifolia',
        'Asclepias obovata',
        'Asclepias oenotheroides',
        'Asclepias purpurascens',
        'Asclepias speciosa',
        'Asclepias stenophylla',
        'Asclepias subulata',
        'Asclepias subverticillata',
        'Asclepias sullivantii',
        'Asclepias syriaca',
        'Asclepias texana',
        'Asclepias tuberosa',
        'Asclepias verticillata',
        'Asclepias viridiflora',
        'Asclepias viridis',
        'Calotropis procera',
        'Cascabela ovata',
        'Cynanchum laeve',
        'Cynanchum louiseae',
        'Cynanchum rossicum',
        'Funastrum cynanchoides',
        'Funastrum hirtellum',
        'Gomphocarpus physocarpus',
        'Nerium oleander',
        'Parsonsia heterophylla',
        'Vinca major',
        'Vinca minor',
        'Vincetoxicum hirundinaria',
        'Blackstonia perfoliata',
        'Centaurium pulchellum',
        'Frasera caroliniensis',
        'Gentiana alba',
        'Gentiana asclepiadea',
        'Gentiana lutea',
        'Gentiana verna',
        'Gentianella quinquefolia',
        'Gentianopsis ciliata',
        'Gentianopsis virgata',
        'Halenia deflexa',
        'Sabatia campestris',
        'Swertia perennis',
        'Zeltnera beyrichii',
        'Zeltnera muehlenbergii',
        'Zeltnera texensis',
        'Zeltnera venusta',
        'Geniostoma ligustrifolium',
        'Mitreola petiolata',
        'Bouvardia ternifolia',
        'Cephalanthus occidentalis',
        'Chiococca alba',
        'Coffea arabica',
        'Coprosma acerosa',
        'Coprosma areolata',
        'Coprosma crassifolia',
        'Coprosma dumosa',
        'Coprosma grandifolia',
        'Coprosma linariifolia',
        'Coprosma lucida',
        'Coprosma perpusilla',
        'Coprosma repens',
        'Coprosma rhamnoides',
        'Coprosma robusta',
        'Coprosma rugosa',
        'Cruciata laevipes',
        'Diodia teres',
        'Diodia virginiana',
        'Galium andrewsii',
        'Galium angustifolium',
        'Galium aparine',
        'Galium circaezans',
        'Galium mollugo',
        'Galium odoratum',
        'Galium palustre',
        'Galium porrigens',
        'Galium virgatum',
        'Houstonia caerulea',
        'Houstonia longifolia',
        'Houstonia purpurea',
        'Houstonia pusilla',
        'Ixora coccinea',
        'Mitchella repens',
        'Nertera depressa',
        'Psychotria nervosa',
        'Psychotria poeppigiana',
        'Richardia grandiflora',
        'Richardia scabra',
        'Sherardia arvensis',
        'Erodium cicutarium',
        'Geranium brevicaule',
        'Geranium caespitosum',
        'Geranium erianthum',
        'Geranium lucidum',
        'Geranium maculatum',
        'Geranium pratense',
        'Geranium pyrenaicum',
        'Geranium robertianum',
        'Geranium sanguineum',
        'Geranium sylvaticum',
        'Gunnera monoica',
        'Gunnera tinctoria',
        'Asystasia gangetica',
        'Avicennia marina',
        'Dicliptera brachiata',
        'Dyschoriste linearis',
        'Hypoestes phyllostachya',
        'Justicia pilosella',
        'Pachystachys lutea',
        'Ruellia simplex',
        'Campsis radicans',
        'Catalpa bignonioides',
        'Chilopsis linearis',
        'Crescentia alata',
        'Crescentia cujete',
        'Handroanthus chrysanthus',
        'Jacaranda mimosifolia',
        'Kigelia africana',
        'Macfadyena unguis-cati',
        'Podranea ricasoliana',
        'Tabebuia rosea',
        'Tecoma stans',
        'Rhabdothamnus solandri',
        'Agastache foeniculum',
        'Agastache scrophulariifolia',
        'Agastache urticifolia',
        'Ajuga reptans',
        'Ballota nigra',
        'Blephilia hirsuta',
        'Condea emoryi',
        'Galeopsis bifida',
        'Galeopsis tetrahit',
        'Glechoma hederacea',
        'Hedeoma acinoides',
        'Hedeoma reverchonii',
        'Hyptis alata',
        'Lamium amplexicaule',
        'Lamium purpureum',
        'Leonotis nepetifolia',
        'Lycopus europaeus',
        'Lycopus uniflorus',
        'Marrubium vulgare',
        'Melittis melissophyllum',
        'Mentha aquatica',
        'Mentha canadensis',
        'Mentha longifolia',
        'Mentha pulegium',
        'Mentha spicata',
        'Monarda citriodora',
        'Monarda fistulosa',
        'Monarda punctata',
        'Monardella villosa',
        'Origanum vulgare',
        'Prunella vulgaris',
        'Pycnanthemum tenuifolium',
        'Salvia apiana',
        'Salvia ballotiflora',
        'Salvia columbariae',
        'Salvia dorrii',
        'Salvia elegans',
        'Salvia greggii',
        'Salvia lavanduloides',
        'Salvia leucophylla',
        'Salvia lyrata',
        'Salvia mellifera',
        'Salvia mexicana',
        'Salvia microphylla',
        'Salvia polystachya',
        'Salvia roemeriana',
        'Salvia spathacea',
        'Salvia texana',
        'Scutellaria drummondii',
        'Scutellaria integrifolia',
        'Scutellaria lateriflora',
        'Scutellaria ovata',
        'Scutellaria parvula',
        'Stachys byzantina',
        'Stachys chamissonis',
        'Stachys coccinea',
        'Stachys officinalis',
        'Stachys rigida',
        'Teucrium canadense',
        'Teucrium chamaedrys',
        'Teucrium cubense',
        'Teucrium laciniatum',
        'Teucrium scorodonia',
        'Trichostema lanceolatum',
        'Warnockia scutellarioides',
        'Pinguicula moranensis',
        'Utricularia cornuta',
        'Utricularia gibba',
        'Utricularia intermedia',
        'Utricularia subulata',
        'Utricularia vulgaris',
        'Lindernia dubia',
        'Proboscidea althaeifolia',
        'Chionanthus virginicus',
        'Fraxinus americana',
        'Fraxinus pennsylvanica',
        'Fraxinus quadrangulata',
        'Fraxinus texensis',
        'Fraxinus velutina',
        'Ligustrum obtusifolium',
        'Ligustrum quihoui',
        'Ligustrum sinense',
        'Agalinis heterophylla',
        'Agalinis tenuifolia',
        'Aureolaria grandiflora',
        'Aureolaria pedicularia',
        'Bellardia viscosa',
        'Castilleja ambigua',
        'Castilleja applegatei',
        'Castilleja arvensis',
        'Castilleja attenuata',
        'Castilleja densiflora',
        'Castilleja exserta',
        'Castilleja indivisa',
        'Castilleja integra',
        'Castilleja latifolia',
        'Castilleja lemmonii',
        'Castilleja linariifolia',
        'Castilleja parviflora',
        'Castilleja rubicundula',
        'Castilleja sessiliflora',
        'Castilleja subinclusa',
        'Castilleja wightii',
        'Chloropyron maritimum',
        'Conopholis americana',
        'Cordylanthus rigidus',
        'Epifagus virginiana',
        'Euphrasia cuneata',
        'Euphrasia nemorosa',
        'Melampyrum nemorosum',
        'Melampyrum pratense',
        'Orobanche hederae',
        'Orobanche minor',
        'Orobanche ramosa',
        'Orthocarpus cuspidatus',
        'Pedicularis bracteosa',
        'Pedicularis densiflora',
        'Pedicularis lanceolata',
        'Pedicularis racemosa',
        'Rhinanthus alectorolophus',
        'Triphysaria pusilla',
        'Martynia annua',
        'Mimulus alatus',
        'Mimulus alsinoides',
        'Mimulus aurantiacus',
        'Mimulus brevipes',
        'Mimulus breweri',
        'Mimulus cardinalis',
        'Mimulus dentatus',
        'Mimulus guttatus',
        'Mimulus kelloggii',
        'Mimulus moschatus',
        'Mimulus nanus',
        'Mimulus ringens',
        'Mimulus tilingii',
        'Mimulus torreyi',
        'Antirrhinum nuttallianum',
        'Chaenorhinum minus',
        'Chelone glabra',
        'Collinsia heterophylla',
        'Collinsia sparsiflora',
        'Collinsia tinctoria',
        'Collinsia torreyi',
        'Collinsia verna',
        'Cymbalaria muralis',
        'Digitalis purpurea',
        'Hippuris vulgaris',
        'Keckiella breviflora',
        'Keckiella cordifolia',
        'Keckiella ternata',
        'Linaria dalmatica',
        'Linaria vulgaris',
        'Mecardonia procumbens',
        'Mohavea breviflora',
        'Mohavea confertiflora',
        'Ourisia macrophylla',
        'Penstemon davidsonii',
        'Penstemon deustus',
        'Penstemon eatonii',
        'Penstemon grandiflorus',
        'Penstemon heterodoxus',
        'Penstemon heterophyllus',
        'Penstemon hirsutus',
        'Penstemon laxiflorus',
        'Penstemon newberryi',
        'Penstemon palmeri',
        'Penstemon parryi',
        'Penstemon procerus',
        'Penstemon roseus',
        'Penstemon rostriflorus',
        'Penstemon speciosus',
        'Penstemon spectabilis',
        'Penstemon whippleanus',
        'Plantago aristata',
        'Plantago helleri',
        'Plantago lanceolata',
        'Plantago major',
        'Plantago ovata',
        'Plantago patagonica',
        'Plantago rhodosperma',
        'Plantago rugelii',
        'Plantago triandra',
        'Plantago virginica',
        'Veronica elliptica',
        'Veronica hederifolia',
        'Veronica odora',
        'Veronica peregrina',
        'Veronica persica',
        'Veronica scutellata',
        'Veronica stricta',
        'Buchnera americana',
        'Buddleja cordata',
        'Buddleja davidii',
        'Leucospora multifida',
        'Myoporum laetum',
        'Scoparia dulcis',
        'Scrophularia californica',
        'Scrophularia marilandica',
        'Scrophularia nodosa',
        'Verbascum blattaria',
        'Verbascum nigrum',
        'Verbascum thapsus',
        'Polypremum procumbens',
        'Callicarpa americana',
        'Duranta erecta',
        'Glandularia bipinnatifida',
        'Glandularia canadensis',
        'Glandularia pumila',
        'Lantana achyranthifolia',
        'Lantana camara',
        'Lantana horrida',
        'Lantana urticoides',
        'Lantana velutina',
        'Phyla nodiflora',
        'Stachytarpheta jamaicensis',
        'Verbena bracteata',
        'Verbena halei',
        'Verbena hastata',
        'Verbena rigida',
        'Beilschmiedia tarairi',
        'Laurus nobilis',
        'Lindera benzoin',
        'Litsea calicaris',
        'Persea borbonia',
        'Sassafras albidum',
        'Umbellularia californica',
        'Hedycarya arborea',
        'Annona glabra',
        'Annona muricata',
        'Asimina triloba',
        'Liriodendron tulipifera',
        'Magnolia acuminata',
        'Magnolia fraseri',
        'Magnolia grandiflora',
        'Magnolia macrophylla',
        'Magnolia tripetala',
        'Magnolia virginiana',
        'Chrysobalanus icaco',
        'Triadenum fraseri',
        'Triadenum virginicum',
        'Acalypha ostryifolia',
        'Acalypha phleoides',
        'Acalypha rhomboidea',
        'Cnidoscolus stimulosus',
        'Cnidoscolus texanus',
        'Croton fruticulosus',
        'Croton glandulosus',
        'Croton lindheimeri',
        'Croton monanthogynus',
        'Croton setiger',
        'Ditaxis lanceolata',
        'Euphorbia abramsiana',
        'Euphorbia bicolor',
        'Euphorbia capitellata',
        'Euphorbia characias',
        'Euphorbia corollata',
        'Euphorbia dentata',
        'Euphorbia esula',
        'Euphorbia fendleri',
        'Euphorbia helioscopia',
        'Euphorbia heterophylla',
        'Euphorbia hirta',
        'Euphorbia hyssopifolia',
        'Euphorbia indivisa',
        'Euphorbia maculata',
        'Euphorbia melanadenia',
        'Euphorbia micromera',
        'Euphorbia misera',
        'Euphorbia missurica',
        'Euphorbia oblongata',
        'Euphorbia ophthalmica',
        'Euphorbia polycarpa',
        'Euphorbia polygonifolia',
        'Euphorbia serpillifolia',
        'Euphorbia setiloba',
        'Euphorbia spathulata',
        'Euphorbia stictospora',
        'Euphorbia terracina',
        'Hura polyandra',
        'Mercurialis annua',
        'Mercurialis perennis',
        'Ricinus communis',
        'Stillingia sylvatica',
        'Tragia ramosa',
        'Triadica sebifera',
        'Hypericum anagalloides',
        'Hypericum drummondii',
        'Hypericum gentianoides',
        'Hypericum kalmianum',
        'Hypericum perforatum',
        'Hypericum punctatum',
        'Linum berlandieri',
        'Linum monogynum',
        'Linum pratense',
        'Byrsonima crassifolia',
        'Passiflora incarnata',
        'Passiflora lutea',
        'Passiflora suberosa',
        'Passiflora subpeltata',
        'Passiflora tripartita',
        'Turnera ulmifolia',
        'Phyllanthus polygonoides',
        'Phyllanthus urinaria',
        'Rhizophora mangle',
        'Populus deltoides',
        'Populus fremontii',
        'Populus grandidentata',
        'Populus nigra',
        'Populus tremula',
        'Populus tremuloides',
        'Salix alba',
        'Salix bebbiana',
        'Salix cinerea',
        'Salix discolor',
        'Salix eriocephala',
        'Salix gooddingii',
        'Salix interior',
        'Salix laevigata',
        'Salix nigra',
        'Salix ×fragilis',
        'Melicytus alpinus',
        'Melicytus lanceolatus',
        'Melicytus ramiflorus',
        'Viola blanda',
        'Viola cunninghamii',
        'Viola filicaulis',
        'Viola labradorica',
        'Viola lanceolata',
        'Viola macloskeyi',
        'Viola pedatifida',
        'Viola pedunculata',
        'Viola pubescens',
        'Viola sagittata',
        'Viola sempervirens',
        'Viola sororia',
        'Bixa orellana',
        'Cochlospermum vitifolium',
        'Cistus albidus',
        'Cistus salviifolius',
        'Helianthemum glomeratum',
        'Helianthemum nummularium',
        'Helianthemum scoparium',
        'Hudsonia tomentosa',
        'Adansonia digitata',
        'Allowissadula holosericea',
        'Callirhoe involucrata',
        'Ceiba aesculifolia',
        'Entelea arborescens',
        'Herissantia crispa',
        'Hibiscus denudatus',
        'Hibiscus martianus',
        'Hibiscus moscheutos',
        'Hibiscus tiliaceus',
        'Hoheria angustifolia',
        'Hoheria sexstylosa',
        'Malacothamnus davidsonii',
        'Malva parviflora',
        'Malvastrum coromandelianum',
        'Malvaviscus arboreus',
        'Malvella leprosa',
        'Melochia pyramidata',
        'Napaea dioica',
        'Pachira aquatica',
        'Pavonia lasiopetala',
        'Plagianthus divaricatus',
        'Plagianthus regius',
        'Pseudobombax ellipticum',
        'Sida ciliaris',
        'Sida rhombifolia',
        'Sidalcea malviflora',
        'Sphaeralcea ambigua',
        'Sphaeralcea coccinea',
        'Theobroma cacao',
        'Thespesia populnea',
        'Tilia americana',
        'Tilia cordata',
        'Urena lobata',
        'Waltheria indica',
        'Muntingia calabura',
        'Daphne gnidium',
        'Daphne mezereum',
        'Dirca occidentalis',
        'Dirca palustris',
        'Pimelea villosa',
        'Laguncularia racemosa',
        'Ammannia coccinea',
        'Cuphea aequipetala',
        'Cuphea carthagenensis',
        'Lythrum hyssopifolia',
        'Lythrum salicaria',
        'Trapa natans',
        'Clidemia hirta',
        'Rhexia virginica',
        'Eucalyptus camaldulensis',
        'Kunzea ericoides',
        'Kunzea robusta',
        'Leptospermum scoparium',
        'Lophomyrtus bullata',
        'Lophomyrtus obcordata',
        'Metrosideros excelsa',
        'Metrosideros robusta',
        'Myrtus communis',
        'Syzygium maire',
        'Camissoniopsis bistorta',
        'Chamaenerion angustifolium',
        'Chamaenerion latifolium',
        'Chylismia claviformis',
        'Circaea canadensis',
        'Clarkia rhomboidea',
        'Clarkia rubicunda',
        'Clarkia unguiculata',
        'Epilobium canum',
        'Epilobium coloratum',
        'Epilobium densiflorum',
        'Epilobium melanocaulon',
        'Eremothera boothii',
        'Eulobus californicus',
        'Fuchsia excorticata',
        'Lopezia racemosa',
        'Ludwigia palustris',
        'Oenothera biennis',
        'Oenothera californica',
        'Oenothera cespitosa',
        'Oenothera curtiflora',
        'Oenothera deltoides',
        'Oenothera drummondii',
        'Oenothera gaura',
        'Oenothera speciosa',
        'Taraxia ovata',
        'Nymphaea ampla',
        'Nymphaea odorata',
        'Weinmannia racemosa',
        'Aristotelia fruticosa',
        'Aristotelia serrata',
        'Elaeocarpus dentatus',
        'Oxalis debilis',
        'Oxalis dillenii',
        'Oxalis drummondii',
        'Oxalis exilis',
        'Oxalis incarnata',
        'Oxalis magellanica',
        'Oxalis oregana',
        'Oxalis pes-caprae',
        'Oxalis purpurea',
        'Oxalis stricta',
        'Oxalis violacea',
        'Quintinia serrata',
        'Aristolochia clematitis',
        'Asarum canadense',
        'Asarum hartwegii',
        'Endodeca serpentaria',
        'Piper auritum',
        'Piper excelsum',
        'Anemopsis californica',
        'Nelumbo nucifera',
        'Platanus mexicana',
        'Platanus occidentalis',
        'Platanus racemosa',
        'Platanus wrightii',
        'Banksia integrifolia',
        'Banksia marginata',
        'Grevillea robusta',
        'Hakea sericea',
        'Protea cynaroides',
        'Achlys triphylla',
        'Berberis aquifolium',
        'Berberis darwinii',
        'Berberis glaucocarpa',
        'Berberis nervosa',
        'Berberis pinnata',
        'Berberis thunbergii',
        'Berberis trifoliolata',
        'Caulophyllum thalictroides',
        'Mahonia bealei',
        'Nandina domestica',
        'Podophyllum peltatum',
        'Vancouveria hexandra',
        'Akebia quinata',
        'Cocculus carolinus',
        'Argemone pleiacantha',
        'Argemone polyanthemos',
        'Chelidonium majus',
        'Corydalis aurea',
        'Corydalis cava',
        'Corydalis flavula',
        'Corydalis solida',
        'Dicentra canadensis',
        'Dicentra cucullaria',
        'Dicentra formosa',
        'Dicentra uniflora',
        'Ehrendorferia chrysantha',
        'Eschscholzia californica',
        'Eschscholzia lobbii',
        'Eschscholzia minutiflora',
        'Eschscholzia parishii',
        'Fumaria muralis',
        'Platystemon californicus',
        'Pseudofumaria lutea',
        'Sanguinaria canadensis',
        'Aconitum delphiniifolium',
        'Aconitum napellus',
        'Actaea pachypoda',
        'Actaea rubra',
        'Anemone acutiloba',
        'Anemone americana',
        'Anemone berlandieri',
        'Anemone coronaria',
        'Anemone cylindrica',
        'Anemone hortensis',
        'Anemone narcissiflora',
        'Anemone patens',
        'Anemone quinquefolia',
        'Anemone ranunculoides',
        'Aquilegia canadensis',
        'Aquilegia chrysantha',
        'Aquilegia coerulea',
        'Aquilegia formosa',
        'Aquilegia pubescens',
        'Caltha palustris',
        'Clematis crispa',
        'Clematis foetida',
        'Clematis forsteri',
        'Clematis lasiantha',
        'Clematis ligusticifolia',
        'Clematis texensis',
        'Clematis virginiana',
        'Clematis vitalba',
        'Coptis aspleniifolia',
        'Coptis trifolia',
        'Delphinium californicum',
        'Delphinium carolinianum',
        'Delphinium nudicaule',
        'Delphinium parishii',
        'Delphinium patens',
        'Delphinium variegatum',
        'Eranthis hyemalis',
        'Ficaria verna',
        'Nigella damascena',
        'Pulsatilla alpina',
        'Ranunculus abortivus',
        'Ranunculus alismifolius',
        'Ranunculus aquatilis',
        'Ranunculus bulbosus',
        'Ranunculus californicus',
        'Ranunculus cymbalaria',
        'Ranunculus fascicularis',
        'Ranunculus flammula',
        'Ranunculus glaberrimus',
        'Ranunculus hispidus',
        'Ranunculus insignis',
        'Ranunculus lyallii',
        'Ranunculus muricatus',
        'Ranunculus recurvatus',
        'Ranunculus reflexus',
        'Ranunculus repens',
        'Thalictrum dasycarpum',
        'Thalictrum thalictroides',
        'Trollius europaeus',
        'Cannabis sativa',
        'Celtis australis',
        'Celtis laevigata',
        'Celtis occidentalis',
        'Celtis reticulata',
        'Humulus japonicus',
        'Elaeagnus umbellata',
        'Hippophae rhamnoides',
        'Shepherdia canadensis',
        'Artocarpus altilis',
        'Artocarpus heterophyllus',
        'Ficus aurea',
        'Ficus carica',
        'Ficus petiolaris',
        'Maclura pomifera',
        'Streblus heterophyllus',
        'Ceanothus crassifolius',
        'Ceanothus fendleri',
        'Ceanothus integerrimus',
        'Ceanothus jepsonii',
        'Ceanothus leucodermis',
        'Ceanothus megacarpus',
        'Ceanothus oliganthus',
        'Ceanothus spinosus',
        'Condalia hookeri',
        'Frangula alnus',
        'Frangula californica',
        'Paliurus spina-christi',
        'Pomaderris amoena',
        'Pomaderris kumeraho',
        'Rhamnus alaternus',
        'Rhamnus alnifolia',
        'Rhamnus cathartica',
        'Rhamnus crocea',
        'Rhamnus humboldtiana',
        'Ziziphus obtusifolia',
        'Acaena anserinifolia',
        'Acaena novae-zelandiae',
        'Acaena pinnatifida',
        'Adenostoma fasciculatum',
        'Adenostoma sparsifolium',
        'Agrimonia eupatoria',
        'Agrimonia gryposepala',
        'Agrimonia parviflora',
        'Amelanchier arborea',
        'Amelanchier utahensis',
        'Aronia melanocarpa',
        'Cercocarpus betuloides',
        'Cercocarpus ledifolius',
        'Cercocarpus montanus',
        'Chamaebatiaria millefolium',
        'Coleogyne ramosissima',
        'Cotoneaster simonsii',
        'Crataegus douglasii',
        'Crataegus marshallii',
        'Crataegus monogyna',
        'Dasiphora fruticosa',
        'Drymocallis arguta',
        'Filipendula rubra',
        'Fragaria chiloensis',
        'Fragaria vesca',
        'Fragaria virginiana',
        'Geum aleppicum',
        'Geum canadense',
        'Geum fragarioides',
        'Geum triflorum',
        'Heteromeles arbutifolia',
        'Holodiscus discolor',
        'Ivesia santolinoides',
        'Malus fusca',
        'Malus pumila',
        'Petrophytum caespitosum',
        'Photinia serratifolia',
        'Potentilla anserina',
        'Potentilla argentea',
        'Potentilla canadensis',
        'Potentilla erecta',
        'Potentilla indica',
        'Potentilla recta',
        'Potentilla reptans',
        'Prunus angustifolia',
        'Prunus caroliniana',
        'Prunus cerasifera',
        'Prunus dulcis',
        'Prunus fasciculata',
        'Prunus fremontii',
        'Prunus padus',
        'Prunus pensylvanica',
        'Prunus pumila',
        'Prunus serotina',
        'Prunus serrulata',
        'Prunus virginiana',
        'Purshia stansburyana',
        'Rhodotypos scandens',
        'Rosa acicularis',
        'Rosa bracteata',
        'Rosa californica',
        'Rosa carolina',
        'Rosa multiflora',
        'Rosa setigera',
        'Rubus arcticus',
        'Rubus armeniacus',
        'Rubus australis',
        'Rubus bifrons',
        'Rubus caesius',
        'Rubus chamaemorus',
        'Rubus cissoides',
        'Rubus dalibarda',
        'Rubus flagellaris',
        'Rubus fruticosus',
        'Rubus hispidus',
        'Rubus leucodermis',
        'Rubus odoratus',
        'Rubus parviflorus',
        'Rubus pedatus',
        'Rubus phoenicolasius',
        'Rubus pubescens',
        'Rubus schmidelioides',
        'Rubus spectabilis',
        'Rubus trivialis',
        'Rubus ulmifolius',
        'Rubus ursinus',
        'Sanguisorba canadensis',
        'Sanguisorba officinalis',
        'Sibbaldia procumbens',
        'Sorbaria sorbifolia',
        'Sorbus aucuparia',
        'Sorbus sitchensis',
        'Spiraea alba',
        'Spiraea tomentosa',
        'Ulmus americana',
        'Ulmus crassifolia',
        'Ulmus parvifolia',
        'Ulmus pumila',
        'Ulmus thomasii',
        'Cecropia obtusifolia',
        'Elatostema rugosum',
        'Laportea canadensis',
        'Parietaria officinalis',
        'Parietaria pensylvanica',
        'Pilea microphylla',
        'Urtica dioica',
        'Urtica ferox',
        'Urtica sykesii',
        'Alepis flavida',
        'Peraxilla tetrapetala',
        'Arceuthobium campylopodum',
        'Korthalsella lindsayi',
        'Korthalsella salicornioides',
        'Phoradendron juniperinum',
        'Cotinus coggygria',
        'Malosma laurina',
        'Pachycormus discolor',
        'Pistacia chinensis',
        'Rhus aromatica',
        'Rhus copallinum',
        'Rhus glabra',
        'Rhus integrifolia',
        'Rhus microphylla',
        'Rhus ovata',
        'Rhus typhina',
        'Schinus molle',
        'Toxicodendron diversilobum',
        'Toxicodendron radicans',
        'Bursera microphylla',
        'Azadirachta indica',
        'Dysoxylum spectabile',
        'Melia azedarach',
        'Citrus trifoliata',
        'Melicope simplex',
        'Melicope ternata',
        'Ptelea crenulata',
        'Thamnosma montana',
        'Zanthoxylum fagara',
        'Zanthoxylum hirsutum',
        'Acer circinatum',
        'Acer floridanum',
        'Acer ginnala',
        'Acer macrophyllum',
        'Acer negundo',
        'Acer nigrum',
        'Acer pensylvanicum',
        'Acer platanoides',
        'Acer pseudoplatanus',
        'Acer rubrum',
        'Acer saccharinum',
        'Acer saccharum',
        'Acer spicatum',
        'Aesculus californica',
        'Aesculus flava',
        'Aesculus pavia',
        'Aesculus sylvatica',
        'Dodonaea viscosa',
        'Koelreuteria paniculata',
        'Ailanthus altissima',
        'Liquidambar styraciflua',
        'Cotyledon orbiculata',
        'Crassula connata',
        'Crassula moschata',
        'Dudleya cymosa',
        'Dudleya edulis',
        'Dudleya farinosa',
        'Dudleya saxosa',
        'Echeveria gibbiflora',
        'Echeveria secunda',
        'Echeveria simulans',
        'Hylotelephium telephium',
        'Bryophyllum delagoense',
        'Sedum acre',
        'Sedum dasyphyllum',
        'Sedum nuttallii',
        'Sedum obtusatum',
        'Sedum palmeri',
        'Sedum ternatum',
        'Umbilicus rupestris',
        'Ribes americanum',
        'Ribes aureum',
        'Ribes bracteosum',
        'Ribes californicum',
        'Ribes cynosbati',
        'Ribes lacustre',
        'Ribes malvaceum',
        'Ribes montigenum',
        'Ribes sanguineum',
        'Ribes triste',
        'Ribes uva-crispa',
        'Myriophyllum spicatum',
        'Hamamelis virginiana',
        'Boykinia occidentalis',
        'Chrysosplenium alternifolium',
        'Chrysosplenium americanum',
        'Heuchera richardsonii',
        'Heuchera rubescens',
        'Heuchera sanguinea',
        'Lithophragma glabrum',
        'Lithophragma parviflorum',
        'Micranthes oregana',
        'Micranthes pensylvanica',
        'Micranthes tolmiei',
        'Mitella diphylla',
        'Mitella nuda',
        'Pectiantia breweri',
        'Saxifraga aizoides',
        'Saxifraga oppositifolia',
        'Saxifraga paniculata',
        'Saxifraga tricuspidata',
        'Tellima grandiflora',
        'Tiarella cordifolia',
        'Calystegia collina',
        'Calystegia purpurata',
        'Calystegia sepium',
        'Calystegia silvatica',
        'Calystegia soldanella',
        'Calystegia spithamaea',
        'Calystegia subacaulis',
        'Calystegia tuguriorum',
        'Convolvulus althaeoides',
        'Convolvulus arvensis',
        'Convolvulus equitans',
        'Cuscuta pacifica',
        'Cuscuta salina',
        'Dichondra carolinensis',
        'Dichondra repens',
        'Evolvulus arizonicus',
        'Evolvulus sericeus',
        'Ipomoea alba',
        'Ipomoea batatas',
        'Ipomoea bracteata',
        'Ipomoea cairica',
        'Ipomoea carnea',
        'Ipomoea coccinea',
        'Ipomoea cordatotriloba',
        'Ipomoea cristulata',
        'Ipomoea hederifolia',
        'Ipomoea murucoides',
        'Ipomoea nil',
        'Ipomoea orizabensis',
        'Ipomoea pedicellaris',
        'Ipomoea quamoclit',
        'Ipomoea sagittata',
        'Ipomoea stans',
        'Ipomoea ternifolia',
        'Ipomoea tricolor',
        'Ipomoea triloba',
        'Jacquemontia tamnifolia',
        'Merremia umbellata',
        'Hydrolea ovata',
        'Datura discolor',
        'Datura inoxia',
        'Datura stramonium',
        'Datura wrightii',
        'Lycium andersonii',
        'Lycium berlandieri',
        'Lycium californicum',
        'Lycium carolinianum',
        'Nicandra physalodes',
        'Nicotiana glauca',
        'Physalis alkekengi',
        'Physalis angulata',
        'Physalis cinerascens',
        'Physalis crassifolia',
        'Physalis heterophylla',
        'Physalis peruviana',
        'Solanum aviculare',
        'Solanum carolinense',
        'Solanum dimidiatum',
        'Solanum dulcamara',
        'Solanum elaeagnifolium',
        'Solanum hindsianum',
        'Solanum laciniatum',
        'Solanum mauritianum',
        'Solanum nigrum',
        'Solanum pseudocapsicum',
        'Solanum ptychanthum',
        'Solanum rostratum',
        'Solanum tridynamum',
        'Solanum triquetrum',
        'Solanum umbelliferum',
        'Ampelopsis arborea',
        'Ampelopsis glandulosa',
        'Cissus trifoliata',
        'Parthenocissus inserta',
        'Parthenocissus quinquefolia',
        'Parthenocissus tricuspidata',
        'Vitis aestivalis',
        'Vitis arizonica',
        'Vitis girdiana',
        'Vitis rotundifolia',
        'Krameria bicolor',
        'Krameria erecta',
        'Fagonia laevis',
        'Guaiacum angustifolium',
        'Guaiacum coulteri',
        'Kallstroemia grandiflora',
        'Larrea tridentata',
        'Tribulus terrestris',
        'Bazzania trilobata',
        'Asterella californica',
        'Conocephalum salebrosum',
        'Monoclea forsteri',
        'Ricciocarpos natans',
        'Calocedrus decurrens',
        'Cupressus lusitanica',
        'Cupressus macrocarpa',
        'Cupressus nootkatensis',
        'Juniperus ashei',
        'Juniperus californica',
        'Juniperus communis',
        'Juniperus flaccida',
        'Juniperus horizontalis',
        'Juniperus monosperma',
        'Juniperus occidentalis',
        'Juniperus scopulorum',
        'Juniperus virginiana',
        'Libocedrus bidwillii',
        'Libocedrus plumosa',
        'Sequoia sempervirens',
        'Taxodium distichum',
        'Taxodium mucronatum',
        'Thuja occidentalis',
        'Thuja plicata',
        'Abies alba',
        'Abies balsamea',
        'Abies fraseri',
        'Abies lasiocarpa',
        'Abies magnifica',
        'Abies religiosa',
        'Larix laricina',
        'Picea engelmannii',
        'Picea rubens',
        'Pinus albicaulis',
        'Pinus attenuata',
        'Pinus balfouriana',
        'Pinus cembroides',
        'Pinus contorta',
        'Pinus edulis',
        'Pinus elliottii',
        'Pinus flexilis',
        'Pinus halepensis',
        'Pinus hartwegii',
        'Pinus longaeva',
        'Pinus muricata',
        'Pinus pinaster',
        'Pinus pinea',
        'Pinus ponderosa',
        'Pinus pseudostrobus',
        'Pinus resinosa',
        'Pinus sabiniana',
        'Pinus strobus',
        'Pinus torreyana',
        'Pinus virginiana',
        'Pseudotsuga macrocarpa',
        'Pseudotsuga menziesii',
        'Tsuga canadensis',
        'Dacrycarpus dacrydioides',
        'Dacrydium cupressinum',
        'Halocarpus bidwillii',
        'Lepidothamnus laxifolius',
        'Phyllocladus trichomanoides',
        'Podocarpus laetus',
        'Torreya californica',
        'Ptisana salicina',
        'Cyathea cunninghamii',
        'Cyathea dealbata',
        'Cyathea medullaris',
        'Cyathea smithii',
        'Dicksonia antarctica',
        'Dicksonia fibrosa',
        'Dicksonia squarrosa',
        'Gleichenia dicarpa',
        'Gleichenia microphylla',
        'Sticherus cunninghamii',
        'Hymenophyllum bivalve',
        'Hymenophyllum flabellatum',
        'Hymenophyllum frankliniae',
        'Hymenophyllum multifidum',
        'Hymenophyllum nephrophyllum',
        'Hymenophyllum rarum',
        'Hymenophyllum revolutum',
        'Hymenophyllum sanguinolentum',
        'Hymenophyllum scabrum',
        'Hymenophyllum villosum',
        'Polyphlebium venosum',
        'Trichomanes elongatum',
        'Leptopteris hymenophylloides',
        'Osmunda claytoniana',
        'Osmunda regalis',
        'Osmundastrum cinnamomeum',
        'Asplenium appendiculatum',
        'Asplenium bulbiferum',
        'Asplenium ceterach',
        'Asplenium flaccidum',
        'Asplenium gracillimum',
        'Asplenium hookerianum',
        'Asplenium lamprophyllum',
        'Asplenium lyallii',
        'Asplenium montanum',
        'Asplenium oblongifolium',
        'Asplenium obtusatum',
        'Asplenium platyneuron',
        'Asplenium polyodon',
        'Asplenium rhizophyllum',
        'Asplenium scolopendrium',
        'Asplenium trichomanes',
        'Asplenium viride',
        'Athyrium filix-femina',
        'Deparia acrostichoides',
        'Deparia petersenii',
        'Diplazium australe',
        'Blechnum blechnoides',
        'Blechnum chambersii',
        'Blechnum colensoi',
        'Blechnum discolor',
        'Blechnum filiforme',
        'Blechnum fluviatile',
        'Blechnum fraseri',
        'Blechnum membranaceum',
        'Blechnum minus',
        'Blechnum montanum',
        'Blechnum novae-zelandiae',
        'Blechnum parrisiae',
        'Blechnum spicant',
        'Blechnum vulcanicum',
        'Woodwardia fimbriata',
        'Cystopteris protrusa',
        'Gymnocarpium dryopteris',
        'Histiopteris incisa',
        'Hypolepis ambigua',
        'Hypolepis millefolium',
        'Paesia scaberula',
        'Pteridium aquilinum',
        'Pteridium esculentum',
        'Cyrtomium falcatum',
        'Dryopteris affinis',
        'Dryopteris arguta',
        'Dryopteris cristata',
        'Dryopteris expansa',
        'Dryopteris fragrans',
        'Dryopteris goldiana',
        'Dryopteris intermedia',
        'Dryopteris marginalis',
        'Lastreopsis glabella',
        'Lastreopsis hispida',
        'Lastreopsis microsora',
        'Polystichum acrostichoides',
        'Polystichum braunii',
        'Polystichum imbricans',
        'Polystichum lonchitis',
        'Polystichum munitum',
        'Polystichum neozelandicum',
        'Polystichum vestitum',
        'Polystichum wawranum',
        'Rumohra adiantiformis',
        'Lindsaea linearis',
        'Lindsaea trichomanoides',
        'Nephrolepis cordifolia',
        'Matteuccia struthiopteris',
        'Onoclea sensibilis',
        'Loxogramme dictyopteris',
        'Microsorum pustulatum',
        'Notogrammitis billardierei',
        'Notogrammitis heterophylla',
        'Phlebodium pseudoaureum',
        'Pleopeltis polypodioides',
        'Polypodium glycyrrhiza',
        'Polypodium virginianum',
        'Pyrrosia eleagnifolia',
        'Adiantum aleuticum',
        'Adiantum capillus-veneris',
        'Adiantum cunninghamii',
        'Adiantum hispidulum',
        'Adiantum jordanii',
        'Adiantum pedatum',
        'Astrolepis sinuata',
        'Cheilanthes sieberi',
        'Cryptogramma acrostichoides',
        'Myriopteris alabamensis',
        'Myriopteris aurea',
        'Myriopteris covillei',
        'Pellaea andromedifolia',
        'Pellaea bridgesii',
        'Pellaea cordifolia',
        'Pellaea ovata',
        'Pellaea rotundifolia',
        'Pellaea ternifolia',
        'Pentagramma triangularis',
        'Pteris cretica',
        'Pteris macilenta',
        'Pteris vittata',
        'Arthropteris tenella',
        'Parathelypteris noveboracensis',
        'Phegopteris connectilis',
        'Woodsia ilvensis',
        'Woodsia obtusa',
        'Azolla filiculoides',
        'Azolla rubra',
        'Lygodium articulatum',
        'Schizaea dichotoma',
        'Botrychium lunaria',
        'Botrypus virginianus',
        'Sceptridium dissectum',
        'Psilotum nudum',
        'Tmesipteris elongata',
        'Tmesipteris tannensis',
        'Smithora naiadum',
        'Callithamnion pikeanum',
        'Microcladia coulteri',
        'Cryptopleura ruprechtiana',
        'Neorhodomela larix',
        'Corallina officinalis',
        'Endocladia muricata',
        'Chondracanthus canaliculatus',
        'Chondrus crispus',
        'Mazzaella flaccida',
        'Mazzaella splendens',
        'Erythrophyllum delesserioides',
        'Mastocarpus papillatus',
        'Sarcodiotheca gaudichaudii',
        'Neogastroclonium subarticulatum',
        'Lycogala epidendrum',
        'Fuligo septica',
        'Leocarpus fragilis',
        'Brefeldia maxima'
    ]
    def __init__(self, soft_file=None, **kwargs):
        self.soft_file = soft_file
        super(iNat18, self).__init__(**kwargs)

    def load_annotations(self):
        if self.ann_file is None:
            folder_to_idx = find_folders(self.data_prefix)
            samples = get_samples(
                self.data_prefix,
                folder_to_idx,
                extensions=self.IMG_EXTENSIONS)
            if len(samples) == 0:
                raise (RuntimeError('Found 0 files in subfolders of: '
                                    f'{self.data_prefix}. '
                                    'Supported extensions are: '
                                    f'{",".join(self.IMG_EXTENSIONS)}'))

            self.folder_to_idx = folder_to_idx
        if isinstance(self.ann_file, str):
            with open(self.ann_file) as f:
                samples = [x.strip().split(' ') for x in f.readlines()]
        else:
            raise TypeError('ann_file must be a str or None')
        self.samples = samples

        if self.soft_file is not None:
            soft_labels = mmcv.load(self.soft_file)

        data_infos = []
        for idx, (filename, gt_label) in enumerate(self.samples):
            info = {'img_prefix': self.data_prefix}
            info['img_info'] = {'filename': filename}
            info['gt_label'] = np.array(gt_label, dtype=np.int64)
            if self.soft_file is not None:
                info['gt_logit'] = soft_labels[idx]
            data_infos.append(info)
        return data_infos

    def get_gt_logits(self):
        """Get all soft ground-truth labels (categories).
        Returns:
            list[int]: categories for all images.
        """

        gt_logits = np.array([np.argmax(data['gt_logit']) for data in self.data_infos])
        return gt_logits