import taichi as ti
import numpy as np
import imageio
import os
import json
from math import pi, cos, sin, sqrt

ti.init(arch=ti.gpu, default_fp=ti.f32, debug=True)

SEED = 24
np.random.seed(SEED)

# Simulation parameters
max_particles = 2300  
min_particles = 1200  
n_grid = 128
dx = 1 / n_grid
dt = 2.5e-4
dim = 2
p_rho = 1
p_vol = (dx * 0.5) ** 2
p_mass = p_vol * p_rho
gravity = 9.8
lower_bound = 0.1
upper_bound = 0.9
bound = int(0.115 * n_grid)
E = 400

save_data = {}
velocities = []
accelerations = []
total_frames = 600
steps_per_frame = 10

# Particle types
PARTICLE_TYPE={"Rigid":0, "Droplet": 1, "Boundary": 3, "Water": 5, "Sand": 6, "Goop": 7}

# Line obstacle parameters (will be randomized per simulation)
n_obstacle_particles = 300  # Total particles
line_thickness = 0.008

# Store lines for visualization
current_lines = []  # Will store tuples of (start, end, length)


def initialize_fields(n_particles):
    # Field definitions
    x = ti.Vector.field(2, dtype=ti.f32,  shape=n_particles)
    v = ti.Vector.field(2, dtype=ti.f32,  shape=n_particles)
    C = ti.Matrix.field(2, 2, dtype=ti.f32,  shape=n_particles)
    J = ti.field(dtype=ti.f32,  shape=n_particles)

    grid_v = ti.Vector.field(2, dtype=ti.f32,  shape=(n_grid, n_grid))
    grid_m = ti.field(dtype=ti.f32,  shape=(n_grid, n_grid))

    return x, v, C, J, grid_v, grid_m

@ti.func
def point_to_line_distance(p: ti.types.vector(2, ti.f32), 
                         a: ti.types.vector(2, ti.f32), 
                         b: ti.types.vector(2, ti.f32)) -> ti.f32:
    ab = b - a
    ap = p - a
    t = ap.dot(ab) / ab.norm_sqr()
    t = ti.max(0.0, ti.min(1.0, t))
    projection = a + t * ab
    return (p - projection).norm()

@ti.func
def handle_line_collision(pos: ti.types.vector(2, ti.f32),
                        vel: ti.types.vector(2, ti.f32),
                        line_start: ti.types.vector(2, ti.f32),
                        line_end: ti.types.vector(2, ti.f32),
                        thickness: ti.f32) -> ti.types.vector(2, ti.f32):
    
    ab = line_end - line_start
    ap = pos - line_start
    t = ap.dot(ab) / ab.norm_sqr()
    t = ti.max(0.0, ti.min(1.0, t))
    closest = line_start + t * ab
    dist_vec = pos - closest
    distance = dist_vec.norm()
    
    
    if distance < thickness:
        normal = dist_vec.normalized()
        penetration = thickness - distance
        
       
        vel_normal = vel.dot(normal)
        if vel_normal < 0:  
            vel -= (1.0 + 0.3) * vel_normal * normal  
        
        
        pos += normal * penetration * 0.8 
    
    return pos, vel

@ti.kernel
def substep(x: ti.template(), v: ti.template(), C: ti.template(), J: ti.template(),
        grid_v: ti.template(), grid_m: ti.template(), n_particles: int, lines: ti.types.ndarray()):
    # Reset grid
    for i, j in grid_m:
        grid_v[i, j] = [0, 0]
        grid_m[i, j] = 0
    
    # Particle to grid (P2G)
    for p in x:
        Xp = x[p] / dx
        base = int(Xp - 0.5)
        fx = Xp - base
        w = [0.5 * (1.5 - fx) ** 2, 0.75 - (fx - 1) ** 2, 0.5 * (fx - 0.5) ** 2]
        stress = -dt * 4 * E * p_vol * (J[p] - 1) / dx**2
        affine = ti.Matrix([[stress, 0], [0, stress]]) + p_mass * C[p]
        
        for i, j in ti.static(ti.ndrange(3, 3)):
            offset = ti.Vector([i, j])
            dpos = (offset - fx) * dx
            weight = w[i].x * w[j].y
            # cell = ti.Vector([
            #     ti.min(base[0] + offset[0], n_grid - 1),
            #     ti.min(base[1] + offset[1], n_grid - 1)
            # ]) 
            cell = base + offset
            # Ensure cell is within grid bounds
            cell.x = ti.max(0, ti.min(n_grid - 1, cell.x))
            cell.y = ti.max(0, ti.min(n_grid - 1, cell.y))

            grid_v[cell] += weight * (p_mass * v[p] + affine @ dpos)
            grid_m[cell] += weight * p_mass
            # grid_v[base + offset] += weight * (p_mass * v[p] + affine @ dpos)
            # grid_m[base + offset] += weight * p_mass
    
    # Handle grid collisions and gravity
    for i, j in grid_m:
        if grid_m[i, j] > 0:
            grid_v[i, j] /= grid_m[i, j]
            grid_v[i, j].y -= dt * gravity
            
            # Boundary conditions
            if i < bound and grid_v[i, j].x < 0:
                grid_v[i, j].x = 0
            if i > n_grid - bound and grid_v[i, j].x > 0:
                grid_v[i, j].x = 0
            if j < bound and grid_v[i, j].y < 0:
                grid_v[i, j].y = 0
            if j > n_grid - bound and grid_v[i, j].y > 0:
                grid_v[i, j].y = 0
    
    # Grid to particle (G2P)
    for p in x:
        Xp = x[p] / dx
        base = int(Xp - 0.5)
        fx = Xp - base
        w = [0.5 * (1.5 - fx) ** 2, 0.75 - (fx - 1) ** 2, 0.5 * (fx - 0.5) ** 2]
        new_v = ti.Vector.zero(ti.f32, 2)
        new_C = ti.Matrix.zero(ti.f32, 2, 2)
        
        for i, j in ti.static(ti.ndrange(3, 3)):
            offset = ti.Vector([i, j])
            dpos = (offset - fx) * dx
            weight = w[i].x * w[j].y
            # cell = ti.Vector([
            #     ti.min(base[0] + offset[0], n_grid - 1),
            #     ti.min(base[1] + offset[1], n_grid - 1)
            # ])
            # if cell.x < 0 or cell.x > n_grid-1 or cell.y < 0 or cell.y >= n_grid:
            #     print(cell)
            cell = base + offset
            # Ensure cell is within grid bounds
            cell.x = ti.max(0, ti.min(n_grid - 1, cell.x))
            cell.y = ti.max(0, ti.min(n_grid - 1, cell.y))
            g_v = grid_v[cell]
            # g_v = grid_v[base + offset]
            new_v += weight * g_v
            new_C += 4 * weight * g_v.outer_product(dpos) / dx**2
        
        v[p] = new_v
        x[p] += dt * v[p]
        
        # Simplified line collision handling
        num_lines = lines.shape[0] // 4
        for n in range(num_lines):
            idx = n * 4
            line_start = ti.Vector([lines[idx], lines[idx+1]])
            line_end = ti.Vector([lines[idx+2], lines[idx+3]])
            
            x[p], v[p] = handle_line_collision(x[p], v[p], line_start, line_end, line_thickness)
            # distance = point_to_line_distance(x[p], line_start, line_end)
            
            # if distance < 0.01:
            #     line_dir = (line_end - line_start).normalized()
            #     normal = ti.Vector([-line_dir[1], line_dir[0]])
            #     x[p] += normal * (0.01 - distance)
            #     v[p] -= v[p].dot(normal) * normal
        
         
        J[p] *= 1 + dt * new_C.trace()
        C[p] = new_C

def initialize_particles(shape_type, n_particles):
    
    available_size = upper_bound - lower_bound  

    if shape_type == 0:  
        
        max_shape_size = available_size * 0.6
        width = np.random.uniform(0.1, max_shape_size)
        height = np.random.uniform(0.1, max_shape_size)
        
        
        min_x = lower_bound + width/2
        max_x = upper_bound - width/2
        min_y = lower_bound + height/2
        max_y = upper_bound - height/2
        
        center_x = np.random.uniform(min_x, max_x)
        center_y = np.random.uniform(min_y, max_y)
        
        
        x_np = np.random.rand(n_particles, 2).astype(np.float32) * [width, height] + [center_x - width/2, center_y - height/2]

    elif shape_type == 1:  
        max_radius = available_size * 0.3
        radius = np.random.uniform(0.05, max_radius)
        
     
        min_x = lower_bound + radius
        max_x = upper_bound - radius
        min_y = lower_bound + radius
        max_y = upper_bound - radius
        
        center = np.array([
            np.random.uniform(min_x, max_x),
            np.random.uniform(min_y, max_y)
        ])
        
        
        angles = np.random.rand(n_particles) * 2 * np.pi
        rads = np.sqrt(np.random.rand(n_particles)) * radius
        x_np = np.column_stack([
            center[0] + rads * np.cos(angles),
            center[1] + rads * np.sin(angles)
        ])

    elif shape_type == 2:  
        max_base_size = available_size * 0.4 
        base_size = np.random.uniform(0.1, max_base_size)
        
        
        safe_margin = base_size * 0.6  
        min_x = lower_bound + safe_margin
        max_x = upper_bound - safe_margin
        min_y = lower_bound + safe_margin
        max_y = upper_bound - safe_margin
        
        center = np.array([
            np.random.uniform(min_x, max_x),
            np.random.uniform(min_y, max_y)
        ])
        
        angle = np.random.uniform(0, 2*np.pi)
        rot_matrix = np.array([
            [np.cos(angle), -np.sin(angle)],
            [np.sin(angle), np.cos(angle)]
        ])
        
        vertices = np.array([
            [-base_size/2, -base_size/2],  
            [base_size/2, -base_size/2],   
            [0, base_size/2]               
        ])
        
        vertices = vertices @ rot_matrix.T + center
        
        u = np.random.rand(n_particles, 2).astype(np.float32)
        mask = u.sum(1) > 1
        u[mask] = 1 - u[mask]
        x_np = (u[:, 0:1] * vertices[0] + 
                u[:, 1:2] * vertices[1] + 
                (1 - u.sum(1, keepdims=True)) * vertices[2])
    
    x_np = np.clip(x_np, lower_bound, upper_bound)
    
    return x_np

@ti.kernel
def initialize_fields_values(v: ti.template(), C: ti.template(), J: ti.template()):
    
    v.fill(0)
    J.fill(1)
    C.fill(0)

def generate_random_lines(particle_positions=None):
    global current_lines
    current_lines = []
    
    # Random number of lines (1-3)
    n_lines = np.random.randint(1, 3)
    
    # Calculate average y-position of particles if positions are provided
    avg_y = 0.3  # Default value if no positions
    if particle_positions is not None:
        avg_y = np.mean(particle_positions[:, 1])
        # We want lines to be below the particles, so set max_y to be slightly below average
        max_line_y = min(avg_y * 0.9, 0.7)  # Cap at 0.7 to keep some margin
    else:
        max_line_y = 0.7
    
    # Generate lines with y-coordinates between 0 and max_line_y
    lines = []
    line_lengths = []
    for _ in range(n_lines):
        # Random start and end points with y between 0 and max_line_y
        start_x = np.random.uniform(0.1, 0.9)
        start_y = np.random.uniform(0.1, max_line_y)
        end_x = np.random.uniform(0.1, 0.9)
        end_y = np.random.uniform(0.1, max_line_y)
        
        # Ensure line has minimum length
        while ((end_x - start_x)**2 + (end_y - start_y)**2) < 0.04:  # Minimum length squared
            end_x = np.random.uniform(0.1, 0.9)
            end_y = np.random.uniform(0.1, max_line_y)
        
        length = sqrt((end_x - start_x)**2 + (end_y - start_y)**2)
        lines.extend([start_x, start_y, end_x, end_y])
        current_lines.append(([start_x, start_y], [end_x, end_y], length))
        line_lengths.append(length)
    
    return np.array(lines, dtype=np.float32), line_lengths

def run_simulation(sim_id, output_dir="WaterRamps_simulations"):
    # Initialize particles first
    # Randomly choose initialization shape
    n_particles = np.random.randint(min_particles, max_particles + 1)
    
    shape_type = np.random.randint(0, 3)  # 0=rect, 1=circle, 2=triangle
    x, v, C, J, grid_v, grid_m = initialize_fields(n_particles)
    x_np = initialize_particles(shape_type, n_particles)
    x.from_numpy(x_np)
    initialize_fields_values(v, C, J)

    initial_positions = x.to_numpy()
    
    # Generate random lines for this simulation, using initial particle positions
    lines, line_lengths = generate_random_lines(initial_positions)
    # Create obstacle particles from lines
    total_length = sum(line_lengths)
    particles_per_line = [max(1, int(n_obstacle_particles * length / total_length)) for length in line_lengths]
    diff = n_obstacle_particles - sum(particles_per_line)
    if diff != 0:
        particles_per_line[0] += diff
    
    obstacle_positions = []
    for i, line in enumerate(current_lines):
        start, end, _ = line
        t = np.linspace(0, 1, particles_per_line[i])
        obstacle_pos = np.outer(1 - t, start) + np.outer(t, end)
        obstacle_positions.append(obstacle_pos)
    obstacle_positions = np.concatenate(obstacle_positions)

    positions = []
    velocities = []
    accelerations = []
    # particle_types = PARTICLE_TYPE['Water'] * np.ones(n_particles)
    particle_types = np.concatenate([
        PARTICLE_TYPE['Boundary'] * np.ones(n_obstacle_particles),
        PARTICLE_TYPE['Water'] * np.ones(n_particles)
    ])
    prev_vel = v.to_numpy()

    for frame in range(total_frames):
        for s in range(steps_per_frame):
            substep(x, v, C, J, grid_v, grid_m, n_particles, lines)
        pos = x.to_numpy()
        vel = v.to_numpy()
        vel = vel * dt* steps_per_frame

        if np.isnan(vel).any():
            return None, None, None
            
        acc = (vel - prev_vel)
        
        # Combine obstacle and water particles (obstacles first)
        full_pos = np.concatenate([obstacle_positions, pos])
        full_vel = np.concatenate([np.zeros((n_obstacle_particles, 2)), vel])  # Obstacle vel=0
        full_acc = np.concatenate([np.zeros((n_obstacle_particles, 2)), acc])  # Obstacle acc=0
        
        positions.append(full_pos)
        velocities.append(full_vel)
        accelerations.append(full_acc)
        
        prev_vel = vel.copy()
    
    trajectory = np.empty(2, dtype=object)
    trajectory[0] = np.array(positions)
    trajectory[1] = particle_types
    save_data[f'simulation_trajectory_{sim_id}'] = trajectory
    # save_data[f'simulation_trajectory_{sim_id}'] = np.array(
    #     [np.array(positions), particle_types], 
    #     dtype=object
    # )
      
    if sim_id < 10:  # Save videos for first 2 sims only
        save_video(sim_id, positions, output_dir)

    return velocities, accelerations, current_lines

def save_video(sim_id, positions, output_dir):
    output_path = os.path.join(output_dir, f"sim_{sim_id}.mp4")
    
    gui = ti.GUI(f"Simulation {sim_id}", background_color=0xFFFFFF, show_gui=False)
    fps = 96
    writer = imageio.get_writer(output_path, fps=fps)
    
    # Positions array contains both obstacle and water particles
    # First n_obstacle_particles are obstacles, rest are water
    for frame_pos in positions:
        gui.clear(0xFFFFFF)
        
        # Split into obstacle and water particles
        obstacle_pos = frame_pos[:n_obstacle_particles]
        water_pos = frame_pos[n_obstacle_particles:]
        
        # Draw water particles (blue)
        if len(water_pos) > 0:
            gui.circles(water_pos, radius=1.5, color=0x068587)  # Water blue
        
        # Draw obstacle particles (black)
        if len(obstacle_pos) > 0:
            gui.circles(obstacle_pos, radius=3, color=0x000000)  # Obstacle black
        
        frame = gui.get_image()
        frame = np.rot90(frame, k=1)
        frame = (frame * 255).astype(np.uint8)
        writer.append_data(frame)
    
    writer.close()

if __name__ == "__main__":
    num_simulations = 100
    output_dir = 'WaterRamps_simulations_xl_test'
    os.makedirs(output_dir, exist_ok=True)
    all_velocities = []
    all_accelerations = []
    all_obstacles = []  # To store obstacle configurations
    
    idx = 0
    # for i in range(num_simulations):
    while idx < num_simulations:
        print(f"Running simulation {idx}/{num_simulations}")
        result = run_simulation(idx,output_dir)
        if result[0] is None:
            print(f"Skipping simulation {idx} due to NaN values in velocities.")
            continue
        velocities, accelerations, obstacles = result
        all_velocities.append(velocities)
        all_accelerations.append(accelerations)
        all_obstacles.append(obstacles)
        idx += 1

    vel_stats = {
        "mean": np.mean(np.array([np.mean(np.array(v)[:, n_obstacle_particles:, :], axis=(0, 1)) for v in all_velocities]), axis=(0)),
        "std": np.std(np.array([np.std(np.array(v)[:, n_obstacle_particles:, :], axis=(0, 1)) for v in all_velocities]), axis=(0))
    }
    
    acc_stats = {
        "mean": np.mean(np.array([np.mean(np.array(a)[:, n_obstacle_particles:, :], axis=(0, 1)) for a in all_accelerations]), axis=(0)),
        "std": np.mean(np.array([np.std(np.array(a)[:, n_obstacle_particles:, :], axis=(0, 1)) for a in all_accelerations]), axis=(0))
    }
    # Prepare obstacles metadata
    obstacles_metadata = []
    for obs in all_obstacles:
        obs_config = []
        for line in obs:
            obs_config.append([line[0], line[1]])  # [start, end]
        obstacles_metadata.append(obs_config)
    
    metadata = {
        "bounds": [[lower_bound, upper_bound], [lower_bound, upper_bound]],
        "sequence_length": total_frames,
        "default_connectivity_radius": 0.015,
        "dim": dim,
        "dt": dt*steps_per_frame,
        "vel_mean": vel_stats['mean'].tolist(),
        "vel_std": vel_stats['std'].tolist(),
        "acc_mean": acc_stats['mean'].tolist(),
        "acc_std": acc_stats['std'].tolist(),
        "obstacles_configs": obstacles_metadata
    }
    
    # Save to files
    # all_sim_ids = list(save_data.keys())
    # train_ids = all_sim_ids[:1000]      
    # valid_ids = all_sim_ids[1100:]  
    # test_ids = all_sim_ids[1000:1100]   

    # 
    # train_data = {k: save_data[k] for k in train_ids}
    # valid_data = {k: save_data[k] for k in valid_ids}
    # test_data = {k: save_data[k] for k in test_ids}

    # 
    # np.savez(os.path.join(output_dir, 'train.npz'), **train_data)
    # np.savez(os.path.join(output_dir, 'valid.npz'), **valid_data)
    # np.savez(os.path.join(output_dir, 'test.npz'), **test_data)
    np.savez(os.path.join(output_dir, f'test.npz'), **save_data)
    with open(os.path.join(output_dir, f'metadata.json'), 'w') as f:
        json.dump(metadata, f, indent=2)
    print("All simulations completed!")