import taichi as ti
import imageio  
import numpy as np
import json
import os
from math import sqrt, cos, sin, pi


PARTICLE_TYPE={"Rigid":0, "Droplet": 1, "Boundary": 3, "Water": 5, "Sand": 6, "Goop": 7}

# Initialize Taichi
ti.init(arch=ti.gpu, default_fp=ti.f32)

SEED = 42
np.random.seed(SEED)

# Simulation parameters
max_particles = 4000 
min_particles = 1200  
n_grid = 64
dx = 1 / n_grid
dt = 2.5e-4
dim = 3
p_rho = 1
p_vol = (dx * 0.5) ** 2
p_mass = p_vol * p_rho
gravity = 9.8
neighbour = (3,) * dim

lower_bound = 0.125
upper_bound = 0.875
bound = int(0.15 * n_grid)
E = 400

save_data = {}
velocities = []
accelerations = []
total_frames = 800
steps_per_frame = 20

from sklearn.cluster import KMeans
from sklearn.preprocessing import StandardScaler

class ParticleReducer:
    def __init__(self, target_particles=256, random_state=42):

        self.n_clusters = target_particles
        self.kmeans = KMeans(
            n_clusters=target_particles,
            init='k-means++',
            n_init=10,
            random_state=random_state
        )
        self.scaler = StandardScaler()
        
    def reduce_system(self, positions, velocities):
        
        
        features = np.hstack([positions, velocities])
        scaled_features = self.scaler.fit_transform(features)
        
        
        self.kmeans.fit(scaled_features)
        
        
        cluster_centers_scaled = self.kmeans.cluster_centers_
        cluster_centers = self.scaler.inverse_transform(cluster_centers_scaled)
        dim = positions.shape[1]
        new_positions = cluster_centers[:, :dim]
        
        
        new_velocities = np.zeros_like(new_positions)
        for c in range(self.n_clusters):
            cluster_mask = self.kmeans.labels_ == c
            if np.any(cluster_mask):
                new_velocities[c] = np.mean(velocities[cluster_mask], axis=0)
                
        return new_positions, new_velocities



def initialize_fields(n_particles):
    # Field definitions
    x = ti.Vector.field(dim, dtype=ti.f32,  shape=n_particles)
    v = ti.Vector.field(dim, dtype=ti.f32,  shape=n_particles)
    C = ti.Matrix.field(dim, dim, dtype=ti.f32,  shape=n_particles)
    J = ti.field(dtype=ti.f32,  shape=n_particles)

    grid_v = ti.Vector.field(dim, dtype=ti.f32,  shape=(n_grid,) * dim)
    grid_m = ti.field(dtype=ti.f32,  shape=(n_grid,) * dim)

    return x, v, C, J, grid_v, grid_m



@ti.kernel
def substep(x: ti.template(), v: ti.template(), C: ti.template(), J: ti.template(),
        grid_v: ti.template(), grid_m: ti.template(), n_particles: int):
    

    for I in ti.grouped(grid_m):
        grid_v[I] = ti.zero(grid_v[I])
        grid_m[I] = 0
    ti.loop_config(block_dim=n_grid)

    for p in x:
        Xp = x[p] / dx
        base = int(Xp - 0.5)
        fx = Xp - base
        w = [0.5 * (1.5 - fx) ** 2, 0.75 - (fx - 1) ** 2, 0.5 * (fx - 0.5) ** 2]
        stress = -dt * 4 * E * p_vol * (J[p] - 1) / dx**2
        affine = ti.Matrix.identity(ti.f32, dim) * stress + p_mass * C[p]
        for offset in ti.static(ti.grouped(ti.ndrange(*neighbour))): # transmit particle information to the surrounding grid
            dpos = (offset - fx) * dx
            weight = 1.0
            for i in ti.static(range(dim)):
                weight *= w[offset[i]][i]
            grid_v[base + offset] += weight * (p_mass * v[p] + affine @ dpos)   # project momentum onto the grid
            grid_m[base + offset] += weight * p_mass  
        
    for I in ti.grouped(grid_m):
        if grid_m[I] > 0:
            grid_v[I] /= grid_m[I]         # calculate velocity using momentum
        grid_v[I][1] -= dt * gravity      # aplly gravity
        cond = (I < bound) & (grid_v[I] < 0) | (I > n_grid - bound) & (grid_v[I] > 0)  # boundary conditions
        grid_v[I] = ti.select(cond, 0, grid_v[I])
    ti.loop_config(block_dim=n_grid)

    
    for p in x:
        Xp = x[p] / dx
        base = int(Xp - 0.5)
        fx = Xp - base
        w = [0.5 * (1.5 - fx) ** 2, 0.75 - (fx - 1) ** 2, 0.5 * (fx - 0.5) ** 2]
        
        new_v = ti.zero(v[p])         # new velocity
        new_C = ti.zero(C[p])  
        for offset in ti.static(ti.grouped(ti.ndrange(*neighbour))):
            dpos = (offset - fx) * dx
            weight = 1.0
            for i in ti.static(range(dim)):
                weight *= w[offset[i]][i]
            g_v = grid_v[base + offset]       # get grid velocity
            new_v += weight * g_v               # transfer grid velocity to particles
            new_C += 4 * weight * g_v.outer_product(dpos) / dx**2       # transfer grid affine velocity to particles
        
        v[p] = new_v
        x[p] += dt * v[p]
        if x[p].x < lower_bound or x[p].x > upper_bound:
            x[p].x = ti.max(lower_bound, ti.min(upper_bound, x[p].x))
        if x[p].y < lower_bound or x[p].y > upper_bound:
            x[p].y = ti.max(lower_bound, ti.min(upper_bound, x[p].y))
        if x[p].z < lower_bound or x[p].z > upper_bound:
            x[p].z = ti.max(lower_bound, ti.min(upper_bound, x[p].z))
        
        J[p] *= 1 + dt * new_C.trace()
        C[p] = new_C

def T(a):
    if dim == 2:
        return a

    phi, theta = np.radians(28), np.radians(32)

    a = a - 0.5
    x, y, z = a[:, 0], a[:, 1], a[:, 2]
    cp, sp = np.cos(phi), np.sin(phi)
    ct, st = np.cos(theta), np.sin(theta)
    x, z = x * cp + z * sp, z * cp - x * sp
    u, v = x, y * ct + z * st
    return np.array([u, v]).swapaxes(0, 1) + 0.5
    
def initialize_particles(shape_type, n_particles):
    
    available_size = upper_bound - lower_bound
    
    
    num_parts = np.random.randint(1, 5)
    part_sizes = np.random.multinomial(n_particles, np.ones(num_parts)/num_parts)
    
    
    part_sizes[-1] = n_particles - sum(part_sizes[:-1])
    
    
    part_shapes = np.random.randint(0, 4, size=num_parts)  
    
    part_vel_azimuth = np.random.uniform(0, 2*np.pi, size=num_parts)  
    part_vel_polar = np.random.uniform(0, np.pi, size=num_parts)      
    
    
    x_np = np.zeros((n_particles, 3))
    v_np = np.zeros((n_particles, 3))
    
    
    current_idx = 0
    
    for part in range(num_parts):
        size = part_sizes[part]
        shape_type = part_shapes[part]
        azimuth = part_vel_azimuth[part]
        polar = part_vel_polar[part]
        
        
        part_vel = np.array([
            np.sin(polar) * np.cos(azimuth),
            np.sin(polar) * np.sin(azimuth),
            np.cos(polar)
        ])
        
        if shape_type == 0: 
           
            max_size = available_size * 0.3 # 0.5
            width = np.random.uniform(0.1, max_size)
            height = np.random.uniform(0.1, max_size)
            depth = np.random.uniform(0.1, max_size)
            
            
            min_x = lower_bound + width/2
            max_x = upper_bound - width/2
            min_y = lower_bound + height/2
            max_y = upper_bound - height/2
            min_z = lower_bound + depth/2
            max_z = upper_bound - depth/2
            
            center_x = np.random.uniform(min_x, max_x)
            center_y = np.random.uniform(min_y, max_y)
            center_z = np.random.uniform(min_z, max_z)
            
            
            part_pos = np.random.rand(size, 3).astype(np.float32) * [width, height, depth] + \
                       [center_x - width/2, center_y - height/2, center_z - depth/2]
            
        elif shape_type == 1:  
            
            max_radius = available_size * 0.15 # 0.25
            radius = np.random.uniform(0.05, max_radius)
            
            
            min_x = lower_bound + radius
            max_x = upper_bound - radius
            min_y = lower_bound + radius
            max_y = upper_bound - radius
            min_z = lower_bound + radius
            max_z = upper_bound - radius
            
            center = np.array([
                np.random.uniform(min_x, max_x),
                np.random.uniform(min_y, max_y),
                np.random.uniform(min_z, max_z)
            ])
            
            
            angles_azimuth = np.random.rand(size) * 2 * np.pi
            angles_polar = np.arccos(2 * np.random.rand(size) - 1)
            rads = np.cbrt(np.random.rand(size)) * radius 
            
            part_pos = np.column_stack([
                center[0] + rads * np.sin(angles_polar) * np.cos(angles_azimuth),
                center[1] + rads * np.sin(angles_polar) * np.sin(angles_azimuth),
                center[2] + rads * np.cos(angles_polar)
            ])
            
        elif shape_type == 2:  
            
            max_radius = available_size * 0.1 #0.2
            max_height = available_size * 0.2 #0.4
            radius = np.random.uniform(0.05, max_radius)
            height = np.random.uniform(0.1, max_height)
            
            
            min_x = lower_bound + radius
            max_x = upper_bound - radius
            min_y = lower_bound + radius
            max_y = upper_bound - radius
            min_z = lower_bound + height/2
            max_z = upper_bound - height/2
            
            center = np.array([
                np.random.uniform(min_x, max_x),
                np.random.uniform(min_y, max_y),
                np.random.uniform(min_z, max_z)
            ])
            
            
            axis_azimuth = np.random.uniform(0, 2*np.pi)
            axis_polar = np.random.uniform(0, np.pi)
            axis = np.array([
                np.sin(axis_polar) * np.cos(axis_azimuth),
                np.sin(axis_polar) * np.sin(axis_azimuth),
                np.cos(axis_polar)
            ])
            
            
            angles = np.random.rand(size) * 2 * np.pi
            rads = np.sqrt(np.random.rand(size)) * radius 
            h = (np.random.rand(size) - 0.5) * height
            
            
            
            if np.abs(axis[0]) > 0.1 or np.abs(axis[1]) > 0.1:
                perp_vec1 = np.array([-axis[1], axis[0], 0])
            else:
                perp_vec1 = np.array([0, -axis[2], axis[1]])
            perp_vec1 = perp_vec1 / np.linalg.norm(perp_vec1)
            perp_vec2 = np.cross(axis, perp_vec1)
            
            
            part_pos = np.column_stack([
                center[0] + rads * (np.cos(angles) * perp_vec1[0] + np.sin(angles) * perp_vec2[0]) + h * axis[0],
                center[1] + rads * (np.cos(angles) * perp_vec1[1] + np.sin(angles) * perp_vec2[1]) + h * axis[1],
                center[2] + rads * (np.cos(angles) * perp_vec1[2] + np.sin(angles) * perp_vec2[2]) + h * axis[2]
            ])
            
        elif shape_type == 3:  
            
            max_size = available_size * 0.2 #0.3
            base_size = np.random.uniform(0.1, max_size)
            
            
            safe_margin = base_size * 0.6
            min_x = lower_bound + safe_margin
            max_x = upper_bound - safe_margin
            min_y = lower_bound + safe_margin
            max_y = upper_bound - safe_margin
            min_z = lower_bound + safe_margin
            max_z = upper_bound - safe_margin
            
            center = np.array([
                np.random.uniform(min_x, max_x),
                np.random.uniform(min_y, max_y),
                np.random.uniform(min_z, max_z)
            ])
            
            
            vertices = np.array([
                [0, 0, 0],
                [base_size, 0, 0],
                [base_size/2, base_size*np.sqrt(3)/2, 0],
                [base_size/2, base_size*np.sqrt(3)/6, base_size*np.sqrt(6)/3]
            ])
            
            
            rot_azimuth = np.random.uniform(0, 2*np.pi)
            rot_polar = np.random.uniform(0, np.pi)
            axis = np.array([
                np.sin(rot_polar) * np.cos(rot_azimuth),
                np.sin(rot_polar) * np.sin(rot_azimuth),
                np.cos(rot_polar)
            ])
            rot_angle = np.random.uniform(0, 2*np.pi)
            
            
            K = np.array([
                [0, -axis[2], axis[1]],
                [axis[2], 0, -axis[0]],
                [-axis[1], axis[0], 0]
            ])
            R = np.eye(3) + np.sin(rot_angle) * K + (1 - np.cos(rot_angle)) * K @ K
            
            
            vertices = (R @ vertices.T).T + center
            
            
            u = np.random.rand(size, 3)
            mask = u.sum(1) > 1
            while np.any(mask):
                u[mask] = np.random.rand(np.sum(mask), 3)
                mask = u.sum(1) > 1
            
            part_pos = (u[:, 0:1] * vertices[0] + 
                       u[:, 1:2] * vertices[1] + 
                       u[:, 2:3] * vertices[2] +
                       (1 - u.sum(1, keepdims=True)) * vertices[3])
        
        
        x_np[current_idx:current_idx+size] = part_pos
        v_np[current_idx:current_idx+size] = part_vel
        current_idx += size
    
    
    x_np = np.clip(x_np, lower_bound, upper_bound)
    
    return x_np, v_np

@ti.kernel
def initialize_fields_values( C: ti.template(), J: ti.template()):
    
    J.fill(1)
    C.fill(0)

def run_simulation_mpm_thershold(next_position, next_velocity, step=50, output_dir="Gns_mpm_test"):
    
    # first_positions = data[0][0]
    # second_positions = data[0][1]
    # velocity = (second_positions - first_positions)/(dt* steps_per_frame)
    next_velocity = next_velocity/(dt* steps_per_frame)
    
    n_particles = int(next_position.shape[0])
    
    # reducer = ParticleReducer(target_particles=n_particles)
    # new_pos, new_vel = reducer.reduce_system(first_positions, velocity)
    # new_pos = np.ascontiguousarray(new_pos)
    # new_vel = np.ascontiguousarray(new_vel)
    # Randomly choose initialization shape
    # shape_type = np.random.randint(0, 4)  
    x, v, C, J, grid_v, grid_m = initialize_fields(n_particles)
    # x_np, v_np = initialize_particles(shape_type, n_particles)
    x.from_numpy(next_position.cpu().numpy())
    v.from_numpy(next_velocity.cpu().numpy())
    initialize_fields_values(C, J)
    
    # Data storage
    positions = []
    velocities = []
    accelerations = []
    particle_types = PARTICLE_TYPE['Water'] * np.ones(n_particles)
    prev_vel = v.to_numpy()

    for frame in range(step):
        for s in range(steps_per_frame):
            substep(x, v, C, J, grid_v, grid_m, n_particles)
        pos = x.to_numpy()
        vel = v.to_numpy()
        
        # Compute acceleration (finite difference)
        vel = vel * dt* steps_per_frame

        # acc = (vel - prev_vel)
        if np.isnan(vel).any():
            return None,None
        positions.append(pos)
        velocities.append(vel)
        # accelerations.append(acc)
        
        # prev_vel = vel.copy()
      
    # Optional: Save video
    # if sim_id < 10:  # Save videos for first 10 sims only to save space
    # sim_id = 0
    # save_video(sim_id, positions, output_dir)

    return positions, velocities

def save_video(sim_id, positions, output_dir):
    output_path = os.path.join(output_dir,f"sim_{sim_id}.gif")
    
    
    gui = ti.GUI(f"Simulation {sim_id}", background_color=0xFFFFFF, show_gui=False)
    fps = 1000
    # writer = imageio.get_writer(output_path, fps=fps)
    writer = imageio.get_writer(output_path, duration=1000/fps)
    positions = np.array(positions)

    for idx in range(positions.shape[0]):
        gui.clear(0xFFFFFF)
        gui.circles(T(positions[idx]), radius=1.5, color=0x66CCFF)
        
        frame = gui.get_image()
        frame = np.rot90(frame, k=1)
        frame = (frame * 255).astype(np.uint8)
        writer.append_data(frame)
    
    writer.close()

if __name__ == "__main__":
    # num_simulations = 1200
    output_dir = 'Water3d_simulations_k-means_1-1.75'
    os.makedirs(output_dir, exist_ok=True)
    all_velocities = []
    all_accelerations = []
    idx = 0

    Data_name = 'Water3d_simulations'
    split = 'test.npz'
    npz_filename = f''
    data = np.load(npz_filename, allow_pickle=True)
    num_simulations = len(data)

    idx = 0
    for idx in range(num_simulations):
    # while idx < num_simulations:
        if split == 'test.npz':
            trajectory_id = 1000 + idx
        else:
            trajectory_id = idx
        print(f"Running simulation {idx}/{num_simulations}")
        velocities,accelerations = run_simulation(idx,data[f'simulation_trajectory_{trajectory_id}'],output_dir)
        if velocities is None:
            with open(os.path.join(output_dir, 'test_no.txt'), "a") as f:
                f.write(f"{idx},")
            print(f"Skipping simulation {idx} due to NaN values in velocities.")
            continue
        all_velocities.append(velocities)
        all_accelerations.append(accelerations)

    vel_stats = {
        "mean": np.mean(np.array([np.mean(np.array(v), axis=(0, 1)) for v in all_velocities]), axis=(0)),
        "std": np.std(np.array([np.std(np.array(v), axis=(0, 1)) for v in all_velocities]), axis=(0))
    }
    
    acc_stats = {
        "mean": np.mean(np.array([np.mean(np.array(a), axis=(0, 1)) for a in all_accelerations]), axis=(0)),
        "std": np.mean(np.array([np.std(np.array(a), axis=(0, 1)) for a in all_accelerations]), axis=(0))
    }
    metadata = {
        "bounds": [[lower_bound, upper_bound], [lower_bound, upper_bound]],
        "sequence_length": total_frames,
        "default_connectivity_radius": 0.015,
        "dim": dim,
        "dt": dt*steps_per_frame,
        "vel_mean": vel_stats['mean'].tolist(),
        "vel_std": vel_stats['std'].tolist(),
        "acc_mean": acc_stats['mean'].tolist(),
        "acc_std": acc_stats['std'].tolist(),
        "obstacles_configs": [] # [[[,],[,]],], 
    }
    
    # Save to files
    # all_sim_ids = list(save_data.keys())
    # train_ids = all_sim_ids[:1000]      
    # valid_ids = all_sim_ids[1100:]  
    # test_ids = all_sim_ids[1000:1100]   

    # 
    # train_data = {k: save_data[k] for k in train_ids}
    # valid_data = {k: save_data[k] for k in valid_ids}
    # test_data = {k: save_data[k] for k in test_ids}

    # 
    # np.savez(os.path.join(output_dir, 'train.npz'), **train_data)
    # np.savez(os.path.join(output_dir, 'valid.npz'), **valid_data)
    # np.savez(os.path.join(output_dir, 'test.npz'), **test_data)
    np.savez(os.path.join(output_dir, split), **save_data)
    with open(os.path.join(output_dir, f'test.json'), 'w') as f:
        json.dump(metadata, f, indent=2)
    print("All simulations completed!")