use std::marker::PhantomData;

pub trait InfoPerStep<State> {
    type Output;
    fn default() -> Self;
    fn build(self) -> Self::Output;
    fn require_gs(&self) -> bool;
    fn push_g(&mut self, x: State, g: State) -> &mut Self;
    fn require_noise(&self) -> bool;
    fn push_noise(&mut self, ex: State, ev: State) -> &mut Self;
}

pub struct AllInfo<State> {
    pub ex: State,
    pub ev: State,
    pub gs: Vec<(State, State)>,
}
pub struct AllInfoBuilder<State> {
    pub ex: Option<State>,
    pub ev: Option<State>,
    pub gs: Vec<(State, State)>,
}
impl<State> InfoPerStep<State> for AllInfoBuilder<State> {
    type Output = AllInfo<State>;
    #[inline]
    fn default() -> Self {
        Self {
            ex: None,
            ev: None,
            gs: Vec::new(),
        }
    }
    #[inline]
    fn build(self) -> Self::Output {
        AllInfo {
            ex: self.ex.unwrap(),
            ev: self.ev.unwrap(),
            gs: self.gs,
        }
    }
    #[inline]
    fn require_gs(&self) -> bool {
        true
    }
    #[inline]
    fn push_g(&mut self, x: State, g: State) -> &mut Self {
        self.gs.push((x, g));
        self
    }
    #[inline]
    fn require_noise(&self) -> bool {
        true
    }
    #[inline]
    fn push_noise(&mut self, ex: State, ev: State) -> &mut Self {
        self.ex = Some(ex);
        self.ev = Some(ev);
        self
    }
}
pub struct BasicInfo<State> {
    phantom: PhantomData<State>,
}
pub struct BasicInfoBuilder<State> {
    phantom: PhantomData<State>,
}
impl<State> InfoPerStep<State> for BasicInfoBuilder<State> {
    type Output = BasicInfo<State>;
    #[inline]
    fn default() -> Self {
        Self {phantom: PhantomData}
    }
    #[inline]
    fn build(self) -> Self::Output {
        BasicInfo {phantom: PhantomData}
    }
    #[inline]
    fn require_gs(&self) -> bool {
        false
    }
    #[inline]
    fn push_g(&mut self, _x: State, _g: State) -> &mut Self {
        unimplemented!()
    }
    #[inline]
    fn require_noise(&self) -> bool {
        false
    }
    #[inline]
    fn push_noise(&mut self, _ex: State, _ev: State) -> &mut Self {
        unimplemented!()
    }
}
